/*	Copyright (C) 1997 Marius Vollmer
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <assert.h>
#include <gtk/gtk.h>
#include <gdk/gdkprivate.h>
#include <libguile.h>
#include <guile/gh.h>

#include "guile-gtk.h"
#include "guile-compat.h"
#include "gtk-types.h"
#include "gtk-compat.h"

/* GtkObjects.

   GtkObjects are wrapped with a smob.  The smob refers to the
   GtkObject with its cdr.  You can retrieve the smob cell from the
   "guile_scm_handle" data of the GtkObject.  There is always at most
   one smob per GtkObject and at most one GtkObject per smob, but
   there can be GtkObjects without an associated smob (then
   "guile_scm_handle" is NULL) and there can be smobs without a
   GtkObject (then the cdr is NULL).

   The Scheme side does not increment the reference count of the
   GtkObject when it stores a pointer to it in a smob.  That would
   prevent the object from ever being destroyed.  Rather it installs a
   "destroy" signal handler.  When that handler is called, the smob is
   detached from its GtkObject by setting its cdr to NULL (when there
   was a smob in the first place).  When on the other hand the smob is
   collected because no Scheme code has a reference to it, the
   "guile_scm_handle" of the associated object is set to NULL (when
   the object hasn't been destroyed already).

   To prevent GtkObjects from getting lost because there is no-one who
   can send them a destroy signal, a GtkObject is destroyed when its
   smob is collected and the reference count of the object is zero.
   This might or might not be the right thing.  Time will tell.

   Because there is atmost one smob per GtkObject, it is ok to compare
   wrapped GtkObjects with `eq?'.  The smob might change in the
   lifetime of a GtkObject, but Scheme code can't observe that. */

static long tc16_gtkobj;

#define GTKOBJP(x) (SCM_NIMP(x) && SCM_CAR(x) == tc16_gtkobj)
#define GTKOBJ(x)  ((GtkObject *)SCM_CDR(x))

static char *handle_key = "guile_scm_handle";

static int
gtkobj_print (SCM exp, SCM port, scm_print_state *pstate)
{
  scm_puts ("#<", port);
  if (GTKOBJ (exp) == NULL)
    scm_puts ("destroyed GtkObject", port);
  else
    {
      guint tid = GTK_OBJECT_TYPE (GTKOBJ (exp));
      scm_puts (gtk_type_name (tid), port);
      scm_puts (" ", port);
      scm_intprint ((long)GTKOBJ (exp), 16, port);
    }
  scm_puts (">", port);
  return 1;
}

static scm_sizet
gtkobj_free (SCM obj)
{
  GtkObject *o = GTKOBJ (obj);
  if (o)
    {
      if (o->ref_count == 0)
	{
	  fprintf (stderr, "leaking %p (%s)\n",
		   o, gtk_type_name (GTK_OBJECT_TYPE (o)));
	  /* gtk_object_destroy (o); */
	}
      else
	gtk_object_remove_data (o, handle_key);
    }
  return 0;
}

struct scm_smobfuns gtkobj_smob = {
  scm_mark0,
  gtkobj_free,
  gtkobj_print,
  NULL
};

static void
destroy_gtkobj (GtkObject *obj, gpointer unused)
{
  SCM cell = (SCM) gtk_object_get_data (obj, handle_key);
  if (cell != (SCM) NULL)
    SCM_SETCDR(cell, NULL);
}

static SCM
make_gtkobj (GtkObject *obj)
{
  SCM z;

  SCM_DEFER_INTS;
  SCM_NEWCELL (z);
  SCM_SETCAR (z, tc16_gtkobj);
  SCM_SETCDR (z, obj);
  gtk_object_set_data (obj, handle_key, (gpointer)z);
  gtk_signal_connect (obj, "destroy",
		      (GtkSignalFunc)destroy_gtkobj, NULL);
  SCM_ALLOW_INTS;

  return z;
}

SCM
sgtk_wrap_gtkobj (GtkObject *obj)
{
  SCM handle;

  if (obj == NULL)
    return SCM_BOOL_F;

  handle = (SCM) gtk_object_get_data (obj, handle_key);
  if (handle == (SCM) NULL)
    handle = make_gtkobj (obj);
  return handle;
}

int
sgtk_is_a_gtkobj (guint type, SCM obj)
{
  GtkObject *gobj;

  if (!(SCM_NIMP (obj) && GTKOBJP (obj)))
    return 0;
  gobj = GTKOBJ (obj);
  return gobj && gtk_type_is_a (GTK_OBJECT_TYPE (gobj), type);
}

GtkObject*
sgtk_get_gtkobj (SCM obj)
{
  if (obj == SCM_BOOL_F)
    return NULL;
  else
    return GTKOBJ (obj);
}

SCM_PROC (s_gtk_destroyed_p, "gtk-destroyed?", 1, 0, 0, sgtk_destroyed_p);

SCM
sgtk_destroyed_p (SCM obj)
{
  SCM_ASSERT (SCM_NIMP (obj) && GTKOBJP (obj), obj,
	      SCM_ARG1, s_gtk_destroyed_p);
  return GTKOBJ (obj) == NULL? SCM_BOOL_T : SCM_BOOL_F;
}

/* Protecting SCM values.  We don't use the Guile functions because
   they do not nest properly. */

static SCM protects;

SCM
sgtk_protect_scm (SCM obj)
{
  SCM_SETCAR (protects, scm_cons (obj, SCM_CAR (protects)));
  return obj;
}

void
sgtk_unprotect_scm (SCM obj)
{
  SCM walk;
  SCM *prev;

  for (prev = SCM_CARLOC (protects), walk = SCM_CAR (protects);
       SCM_NIMP (walk) && SCM_CONSP (walk);
       walk = SCM_CDR (walk))
    {
      if (SCM_CAR (walk) == obj)
	{
	  *prev = SCM_CDR (walk);
	  break;
	}
      else
	prev = SCM_CDRLOC (walk);
    }
}

/* Enums.

   Enumerations are described by a `sgtk_enum_info' structure.  That
   structure contains a list of all literals and their respective
   values.  In Scheme, an enum element is represented by a symbol
   whose name is the literal. */

int
sgtk_valid_enum (SCM obj, sgtk_enum_info *info)
{
  int i;

  if (!SCM_NIMP (obj) || !SCM_SYMBOLP (obj))
    return 0;

  for (i = 0; i < info->n_literals; i++)
    if (!strcmp (info->literals[i].name, SCM_CHARS (obj)))
      return 1;
  return 0;
}

SCM
sgtk_enum2scm (gint val, sgtk_enum_info *info)
{
  int i;
  for (i = 0; i < info->n_literals; i++)
    if (info->literals[i].value == val)
      return SCM_CAR (scm_intern0 (info->literals[i].name));
  SCM_ASSERT (0, SCM_MAKINUM (val), SCM_ARG1, "enum->symbol");
  return SCM_BOOL_F;
}

gint
sgtk_scm2enum (SCM obj, sgtk_enum_info *info)
{
  int i;
  for (i = 0; i < info->n_literals; i++)
    if (!strcmp (info->literals[i].name, SCM_CHARS (obj)))
      return info->literals[i].value;
  return -1;
}

/* Flags.

   Like enums, flags are described by a `sgtk_enum_info' structure.
   In Scheme, flags are represented by a list of symbols, one for each
   bit that is set in the flags value. */

int
sgtk_valid_flags (SCM obj, sgtk_enum_info *info)
{
  while (!SCM_NULLP (obj))
    {
      int i, valid;
      SCM sym;
      
      if (SCM_IMP (obj) || !SCM_CONSP (obj))
	return 0;
      sym = SCM_CAR (obj);
      if (SCM_IMP (sym) || !SCM_SYMBOLP (sym))
	return 0;
      
      for (i = 0, valid = 0; i < info->n_literals; i++)
	if (!strcmp (info->literals[i].name, SCM_CHARS (sym)))
	  {
	    valid = 1;
	    break;
	  }
      if (!valid)
	return 0;

      obj = SCM_CDR (obj);
    }
  
  return 1;
}

SCM
sgtk_flags2scm (gint val, sgtk_enum_info *info)
{
  SCM ans = SCM_EOL;
  int i;
  for (i = 0; i < info->n_literals; i++)
    if (val & info->literals[i].value)
      {
	ans = scm_cons (SCM_CAR (scm_intern0 (info->literals[i].name)), ans);
	val &= ~info->literals[i].value;
      }
  SCM_ASSERT (val == 0, SCM_MAKINUM (val), SCM_ARG1, "enum->symbol");
  return ans;
}

gint
sgtk_scm2flags (SCM obj, sgtk_enum_info *info)
{
  int ans = 0;

  while (!SCM_NULLP (obj))
    {
      int i;
      SCM sym = SCM_CAR (obj);
      
      for (i = 0; i < info->n_literals; i++)
	if (!strcmp (info->literals[i].name, SCM_CHARS (sym)))
	  {
	    ans |= info->literals[i].value;
	    break;
	  }
      obj = SCM_CDR (obj);
    }
  
  return ans;
}

/* Boxed Values.

   Boxed values are wrapped with a smob, much like GtkObjects.  The
   cdr of the smob points to a info struture.  This info structure
   holds the actual pointer to the Gtk structure and a pointer to a
   `sgtk_boxed_info' structure.

   The lifetime of a boxed structure is controlled by calling two
   functions: `copy', when a new reference to the structure has to be
   made and `destroy', when a reference is released.

   This works both for reference counted structures (like GdkColormap,
   with gdk_colormap_ref and gdk_colormap_unref) and structures that
   need to be copied (like GdkEvent, with gdk_event_copy and
   gdk_event_destroy).

   To help the Scheme garbage collector, the `sgtk_boxed_info'
   structure contains also the amount of memory allocated when the
   `copy' function is invoked (and freed when `destroy' is invoked).
   This lets the garbage collector keep track of how much memory is in
   use by the Scheme system and can start a collection from time to
   time.

   Reference counted structures should have `0' as their size, because
   no memory is allocated when the reference count is increased.

   Unlike GtkObjects, boxed values are not guaranteed to be wrapped by
   atmost one smob.  So they have to be compared with `equal?'.  */

static long tc16_boxed;

typedef struct _boxed {
  union {
    struct _boxed *free_link;
    struct {
      gpointer ptr;
      sgtk_boxed_info *info;
    } active;
  } d;
} boxed;

#define BOXED_CHUNK 256

static boxed *boxed_pool;

static boxed *
new_boxed ()
{
  /* SCM_CRITICAL_SECTION_START; */
  if (boxed_pool)
    {
      boxed *b = boxed_pool;
      boxed_pool = b->d.free_link;
      /* SCM_CRITICAL_SECTION_END; */
      return b;
    }
  else
    {
      int i;
      boxed *chunk = (boxed *)scm_must_malloc (BOXED_CHUNK*sizeof(boxed),
					       "new-boxed");
      boxed_pool = chunk;
      for (i = 0; i < BOXED_CHUNK-2; i++, chunk++)
	chunk->d.free_link = chunk+1;
      chunk->d.free_link = NULL;
      /* SCM_CRITICAL_SECTION_END; */
      return chunk+1;
    }
}

static void
free_boxed (boxed *b)
{
  /* SCM_CRITICAL_SECTION_START; */
  b->d.free_link = boxed_pool;
  boxed_pool = b;
  /* SCM_CRITICAL_SECTION_END; */
}

#define BOXED_P(x)    (SCM_NIMP(x) && (SCM_CAR(x) == tc16_boxed))
#define BOXED_PTR(x)  (((boxed *)SCM_CDR(x))->d.active.ptr)
#define BOXED_INFO(x) (((boxed *)SCM_CDR(x))->d.active.info)

static scm_sizet
boxed_free (SCM obj)
{
  scm_sizet size = BOXED_INFO (obj)->size;
  BOXED_INFO (obj)->destroy (BOXED_PTR (obj));
  free_boxed ((boxed *)SCM_CDR(obj));
  return size;
}

static int
boxed_print (SCM exp, SCM port, scm_print_state *pstate)
{
  scm_puts ("#<", port);
  scm_puts (BOXED_INFO (exp)->name, port);
  scm_puts (" ", port);
  scm_intprint ((long)BOXED_PTR (exp), 16, port);
  scm_puts (">", port);
  return 1;
}

static SCM
boxed_equal (SCM exp1, SCM exp2)
{
  return (BOXED_PTR (exp1) == BOXED_PTR (exp2))? SCM_BOOL_T : SCM_BOOL_F;
}

struct scm_smobfuns boxed_smob = {
  scm_mark0,
  boxed_free,
  boxed_print,
  boxed_equal
};

SCM
sgtk_boxed2scm (gpointer ptr, sgtk_boxed_info *info, int copyp)
{
  SCM z;
  boxed *b;

  if (ptr == NULL)
    return SCM_BOOL_F;

  SCM_DEFER_INTS;
  b = new_boxed ();
  if (copyp)
    {
      ptr = info->copy (ptr);
      scm_done_malloc (info->size);
    }
  b->d.active.ptr = ptr;
  b->d.active.info = info;
  SCM_NEWCELL (z);
  SCM_SETCAR (z, tc16_boxed);
  SCM_SETCDR (z, b);
  SCM_ALLOW_INTS;

  return z;
}

void *
sgtk_scm2boxed (SCM obj)
{
  if (obj == SCM_BOOL_F)
    return NULL;
  return BOXED_PTR (obj);
}

int
sgtk_valid_boxed (SCM obj, sgtk_boxed_info *info)
{
  return (SCM_NIMP (obj) && BOXED_P (obj) && BOXED_INFO (obj) == info);
}

sgtk_boxed_info*
sgtk_find_boxed_info (GtkType type)
{
  /* XXX - use a hash or something */

  char *name = gtk_type_name (type);
  sgtk_boxed_info *info;

  for (info = sgtk_boxed_infos; info->name; info++)
    if (!strcmp (info->name, name))
      return info;
  SCM_ASSERT (0, SCM_BOOL_F, SCM_ARG1, "find-boxed-info");
  return NULL;
}

sgtk_enum_info*
sgtk_find_enum_info (GtkType type)
{
  /* XXX - use a hash or something */

  char *name = gtk_type_name (type);
  sgtk_enum_info *info;

  for (info = sgtk_enum_infos; info->name; info++)
    if (!strcmp (info->name, name))
      return info;
  SCM_ASSERT (0, SCM_BOOL_F, SCM_ARG1, "find-enum-info");
  return NULL;
}

sgtk_enum_info*
sgtk_find_flags_info (GtkType type)
{
  /* XXX - use a hash or something */

  char *name = gtk_type_name (type);
  sgtk_enum_info *info;

  for (info = sgtk_flags_infos; info->name; info++)
    if (!strcmp (info->name, name))
      return info;
  SCM_ASSERT (0, SCM_BOOL_F, SCM_ARG1, "find-flags-info");
  return NULL;
}

/* Floats.

   Only here to set things straight. */

int
sgtk_valid_float (SCM obj)
{
  return SCM_NUMBERP (obj);
}

SCM gh_double2scm (double);

gfloat
sgtk_scm2float (SCM obj)
{
  return gh_scm2double (obj);
}

SCM
sgtk_float2scm (gfloat f)
{
  return gh_double2scm ((double)f);
}

/* converting between SCM and GtkArg */

SCM
sgtk_arg2scm (GtkArg *a)
{
  switch (GTK_FUNDAMENTAL_TYPE (a->type))
    {
    case GTK_TYPE_CHAR:
      return gh_char2scm (GTK_VALUE_CHAR(*a));
    case GTK_TYPE_INT:
      return scm_long2num (GTK_VALUE_INT(*a));
    case GTK_TYPE_POINTER:
      fprintf (stderr, "WARNING: gpointer in args.\n");
      return scm_ulong2num ((unsigned long)GTK_VALUE_POINTER(*a));
    case GTK_TYPE_OBJECT:
      return sgtk_wrap_gtkobj (GTK_VALUE_OBJECT(*a));
    case GTK_TYPE_BOXED:
      return sgtk_boxed2scm (GTK_VALUE_BOXED(*a),
			     sgtk_find_boxed_info (a->type),
			     TRUE);
    case GTK_TYPE_FLAGS:
      return sgtk_flags2scm (GTK_VALUE_FLAGS(*a),
			     sgtk_find_flags_info (a->type));
    default:
      fprintf (stderr, "illegal type %s in arg\n", 
	       gtk_type_name (a->type));
      return SCM_BOOL_F;
    }
}

void
sgtk_scm2arg (GtkArg *a, SCM obj)
{
  switch (GTK_FUNDAMENTAL_TYPE (a->type))
    {
    case GTK_TYPE_NONE:
      return;
    case GTK_TYPE_CHAR:
      *GTK_RETLOC_CHAR(*a) = gh_scm2char (obj);
      break;
    case GTK_TYPE_INT:
      *GTK_RETLOC_INT(*a) = gh_scm2long (obj);
      break;
    case GTK_TYPE_OBJECT:
      SCM_ASSERT (sgtk_is_a_gtkobj (a->type, obj), obj, SCM_ARG1, "scm->gtk");
      *GTK_RETLOC_OBJECT(*a) = sgtk_get_gtkobj (obj);
      break;
    case GTK_TYPE_BOOL:
      *GTK_RETLOC_BOOL(*a) = SCM_NFALSEP (obj);
      break;
    default:
      fprintf (stderr, "unhandled return type %s\n", gtk_type_name (a->type));
      break;
    }
}

/* Callbacks.

   Callbacks are exected within a new dynamic root.  That means that
   the flow of control can't leave them without Gtk noticing.  Throws
   are catched and briefly reported.  Calls to continuations that have
   been made outside the dynamic root can not be activated.

   Callbacks are invoked with whatever arguments that are specified by
   the Gtk documentation.  They do not, however, receive the GtkObject
   that has initiated the callback. */

struct callback_info {
  SCM proc;
  gint n_args;
  GtkArg *args;
};

static SCM
inner_callback_marshal (void *data, SCM jmpbuf)
{
  struct callback_info *info = (struct callback_info *)data;
  int i;
  SCM args = SCM_EOL, ans;

  for (i = info->n_args-1; i >= 0; i--)
    args = scm_cons (sgtk_arg2scm (info->args+i), args);
  ans = scm_apply (info->proc, args, SCM_EOL);
  if (info->args[info->n_args].type != GTK_TYPE_NONE)
    sgtk_scm2arg (info->args+info->n_args, ans);

  return SCM_UNSPECIFIED;
}
  
void
sgtk_callback_marshal (GtkObject *obj,
		       gpointer data,
		       gint n_args,
		       GtkArg *args)
{
  SCM_STACKITEM stack_item;
  struct callback_info info;
  
  info.proc = (SCM) data;
  info.n_args = n_args;
  info.args = args;

  scm_internal_cwdr (inner_callback_marshal, &info,
		     scm_handle_by_message_noexit, "gtk",
		     &stack_item);
}

void
sgtk_callback_destroy (gpointer data)
{
  sgtk_unprotect_scm ((SCM)data);
}



/* Type conversions */

SCM
sgtk_color_conversion (SCM color)
{
  SCM orig_color = color;
  SCM_COERCE_SUBSTR (color);

  if (SCM_NIMP (color) && SCM_STRINGP (color))
    {
      GdkColor colstruct;
      GdkColormap *colmap;

      SCM_DEFER_INTS;
      if (!gdk_color_parse (SCM_CHARS (color), &colstruct))
	{
	  SCM_ALLOW_INTS;
	  scm_misc_error ("string->color", "no such color: %S",
			  scm_cons (orig_color, SCM_EOL));
	}
      colmap = gtk_widget_peek_colormap ();
      if (!gdk_color_alloc (colmap, &colstruct))
	{
	  SCM_ALLOW_INTS;
	  scm_misc_error ("string->color", "can't allocate color: %S",
			  scm_cons (orig_color, SCM_EOL));
	}
      SCM_ALLOW_INTS;
      return sgtk_boxed2scm (&colstruct, &sgtk_gdk_color_info, 1);
    }
  return color;
}

extern SCM sgtk_gdk_font_load (SCM font);

SCM
sgtk_font_conversion (SCM font)
{
  SCM orig_font = font;
  SCM_COERCE_SUBSTR (font);

  if (SCM_NIMP (font) && SCM_STRINGP (font))
    {
      font = sgtk_gdk_font_load (font);
      if (font == SCM_BOOL_F)
	scm_misc_error ("string->font", "no such font: %S",
			scm_cons (orig_font, SCM_EOL));
    }
  return font;
}

SCM
sgtk_string_conversion (SCM str)
{
  SCM_COERCE_SUBSTR (str);
  return str;
}



/* Initialization */

extern void sgtk_init_gtk_defs ();

void
sgtk_init (int *argcp, char ***argvp)
{
  tc16_gtkobj = scm_newsmob (&gtkobj_smob);
  tc16_boxed = scm_newsmob (&boxed_smob);
  protects = scm_permanent_object (scm_cons (SCM_EOL, SCM_BOOL_F));

#include "guile-gtk.x"

  sgtk_init_gtk_defs ();

  /* XXX - Initialize Gtk only once.  We assume that Gtk has already
     been initialized when Gdk has.  That is not completely correct,
     but the best I can do. */

  if (gdk_display == NULL)
    gtk_init (argcp, argvp);
}

void
sgtk_init_argv (int argc, char **argv)
{
  sgtk_init (&argc, &argv);
}

static char*
xstrdup (char *str)
{
  if (str)
    {
      char *newstr = scm_must_malloc (strlen(str)+1, "strdup");
      strcpy (newstr, str);
      return newstr;
    }
  else
    return NULL;
}

static void
make_argv (SCM list, int *argc, char ***argv)
{
  static char *argv_storage[1] = { "guile-gtk" };

  int c = scm_ilength (list), i;
  char **v;

  *argv = argv_storage;
  *argc = 1;

  if (c < 0)
    return;

  v = (char **)scm_must_malloc ((c+1) * sizeof(char**), "make-argv");
  for (i = 0; i < c; i++, list = SCM_CDR (list))
    {
      if (SCM_IMP (SCM_CAR (list)) || SCM_NSTRINGP (SCM_CAR (list)))
	{
	  scm_must_free ((char *)v);
	  return;
	}
      v[i] = xstrdup (SCM_CHARS (SCM_CAR (list)));
    }
  v[c] = NULL;
  
  *argv = v;
  *argc = c;
}

static void
sgtk_init_noargs ()
{
  int argc;
  char **argv;
  make_argv (scm_program_arguments (), &argc, &argv);
  sgtk_init (&argc, &argv);
  scm_set_program_arguments (argc, argv, NULL);
}

void
scm_init_toolkits_gtkstubs_module ()
{
  scm_register_module_xxx ("toolkits gtkstubs", sgtk_init_noargs);
}
