#include <config.h>
#include <errno.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <glib.h>
#include "gnome-defs.h"
#include "gnome-file.h"


/* check $GNOMEDIR and then the hard coded path */

static gchar *
gnome_file_dirrelative (gchar *base, gchar *sub, gchar *filename)
{
        static gchar *gnomedir = NULL;
	gchar *f, *t, *u;
	
	/* First try the env GNOMEDIR relative path */
	if (!gnomedir)
		gnomedir = getenv ("GNOMEDIR");
	
	if (gnomedir){
		t = gnome_file_and_dir_concat (gnomedir, sub);
		u = gnome_copy_strings (t, "/", filename, NULL);
		g_free (t);
	    
	    /* if they have $GNOMEDIR set, use that */
	    return (u);
	}
	
	/* If not then go for the hardcoded path. */
	f = gnome_file_and_dir_concat (base, filename);
    
    /* return the hardcoded path.. even if the file doesn't exist. */
    return (f);
	
}

/* DOC: gnome_file_libdir (char *filename)
 * Returns a newly allocated pathname poiting to a file in the gnome libdir
 */
gchar *
gnome_file_from_libdir (gchar *filename)
{
	return (gnome_file_dirrelative (GNOMELIBDIR, "lib", filename));
}

/* DOC: gnome_sharedir_file (char *filename)
 * Returns a newly allocated pathname poiting to a file in the gnome sharedir
 */
gchar *
gnome_file_from_datadir (gchar *filename)
{
	return (gnome_file_dirrelative (GNOMEDATADIR, "share", filename));
}

/* DOC: g_file_exists (char *filename)
 * Returns true if filename exists
 */
gint
gnome_file_exists (gchar *filename)
{
	return access (filename, R_OK) == 0;
}


/* DOC: gnome_copy_strings (const char *first,...)
 * returns a new allocated char * with the concatenation of its arguments
 */
gchar *
gnome_copy_strings (const char *first, ...)
{
	va_list ap;
	int len;
	char *data, *result;
	
	if (!first)
		return NULL;
	
	len = strlen (first);
	va_start (ap, first);
	
	while ((data = va_arg (ap, char *)) != NULL)
		len += strlen (data);
	
	len++;
	
	result = (char *) g_malloc (len);
	va_end (ap);
	va_start (ap, first);
	strcpy (result, first);
	while ((data = va_arg (ap, char *)) != NULL)
		strcat (result, data);
	va_end (ap);
	
	return result;
}


/* DOC: g_unix_error_string (int error_num)
 * Returns a pointer to a static location with a description of the errno
 */

/* I removed this one.. it's identical to g_strerror() provided by glib.


/* DOC: gnome_file_and_dir_concat (const char *dir, const char *file)
 * returns a new allocated string that is the concatenation of dir and file,
 * takes care of the exact details for concatenating them.
 */
gchar *
gnome_file_and_dir_concat (const gchar *dir, const gchar *file)
{
	gint l = strlen (dir);
	
	if (dir [l - 1] == PATH_SEP)
		return gnome_copy_strings (dir, file, NULL);
	else
		return gnome_copy_strings (dir, PATH_SEP_STR, file, NULL);
}


/* returns the home directory of the user
 * This one is NOT to be free'd as it points into the 
 * env structure.
 * 
 */

gchar *gnome_file_user_home_dir(void)
{
	static gchar *home_dir;
	static gint init = 1;
	
	if (init) {
		home_dir = getenv("HOME");
		init = 0;
	}
	
	return(home_dir);
}

/* pass in a string, and it will add the users home dir ie,
 * pass in .gnome/bookmarks.html and it will return
 * /home/imain/.gnome/bookmarks.html
 * 
 * Remember to g_free() returned value! */
gchar *gnome_file_prepend_user_home(gchar *file)
{
	gchar *home_return;
	home_return = g_malloc (strlen(file) + strlen(gnome_file_user_home_dir()) + 3);
	
	sprintf(home_return, "%s/%s", gnome_file_user_home_dir(), file);
	return(home_return);
}

/* very similar to above, but adds $HOME/.gnome/ to beginning
 * This is meant to be the most useful version.
 */
gchar *gnome_file_from_home_dir(gchar *file)
{
	gchar *home_return;
	gchar *home_dir;
	
	home_dir = gnome_file_user_home_dir();
	
	home_return = g_malloc (strlen(file) + strlen(home_dir) + 10);
	
	sprintf(home_return, "%s/.gnome/%s", home_dir, file);

	return(home_return);
}

