/*
 * GNOME panel launcher module.
 * (C) 1997 The Free Software Foundation
 *
 * Authors: Miguel de Icaza
 *          Federico Mena
 */

#include <stdio.h>
#ifdef HAVE_LIBINTL
#include <libintl.h>
#define _(String) gettext(String)
#else
#define _(String) (String)
#endif
#include <sys/stat.h>
#include <dlfcn.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>
#include "gnome.h"
#include "../panel_cmds.h"
#include "../applet_cmds.h"
#include "../panel.h"


#define APPLET_ID "Launcher"


static PanelCmdFunc panel_cmd_func;

gpointer applet_cmd_func(AppletCommand *cmd);


typedef struct {
	GtkWidget *dialog;

	GtkWidget *execute_entry;
	GtkWidget *icon_entry;
	GtkWidget *documentation_entry;
	GtkWidget *description_entry;
	GtkWidget *terminal_toggle;

	GnomeDesktopEntry *dentry;
} Properties;


static void
free_user_data(GtkWidget *widget, gpointer data)
{
	g_free(gtk_object_get_user_data(GTK_OBJECT(widget)));
}

static void
cleanup (GtkWidget *widget, void *data)
{
	GnomeDesktopEntry *item = data;

	gnome_desktop_entry_free (item);
}

static void
launch (GtkWidget *widget, void *data)
{
	GnomeDesktopEntry *item = data;

	gnome_desktop_entry_launch (item);
}

static GtkWidget *
create_launcher (GtkWidget *window, char *parameters)
{
	GtkWidget *pixmap, *button;
	GnomeDesktopEntry *dentry;
	static char *default_app_pixmap;

	if (!default_app_pixmap)
		default_app_pixmap = gnome_pixmap_file ("launcher-program.xpm");

	if (*parameters == '/')
		dentry = gnome_desktop_entry_load (parameters);
	else {
		char *apps_par, *entry, *extension;

		if (strstr (parameters, ".desktop"))
			extension = NULL;
		else
			extension = ".desktop";
		
		apps_par = g_copy_strings ("apps/", parameters, extension, NULL);
		entry = gnome_datadir_file (apps_par);
		g_free (apps_par);
		
		if (!entry)
			return 0;
		dentry = gnome_desktop_entry_load (entry);
		g_free (entry);
	}
	if (!dentry)
		return 0;

	button = gtk_button_new ();
	pixmap = gnome_create_pixmap_widget (window, button, dentry->transparent_icon);
	if (!pixmap){
		if (default_app_pixmap)
			pixmap = gnome_create_pixmap_widget (window, button, default_app_pixmap);
		else
			pixmap = gtk_label_new ("App");
	}
	gtk_container_add (GTK_CONTAINER(button), pixmap);
	gtk_widget_show (pixmap);
	gtk_widget_show (button);

	gtk_signal_connect (GTK_OBJECT(button), "clicked", (GtkSignalFunc)launch, dentry);
	gtk_signal_connect (GTK_OBJECT(button), "destroy", (GtkSignalFunc)cleanup, dentry);

	/* The .desktop path is stored in the user data */
	
	gtk_object_set_user_data(GTK_OBJECT(button), g_strdup(parameters));

	gtk_signal_connect(GTK_OBJECT(button), "destroy",
			   (GtkSignalFunc) free_user_data,
			   NULL);

	return button;
}

static GtkWidget *
create_text_entry(GtkWidget *table, int row, char *label, char *text)
{
	GtkWidget *wlabel;
	GtkWidget *entry;

	wlabel = gtk_label_new(label);
	gtk_misc_set_alignment(GTK_MISC(wlabel), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), wlabel,
			 0, 1, row, row + 1,
			 GTK_EXPAND | GTK_FILL | GTK_SHRINK,
			 GTK_FILL | GTK_SHRINK,
			 0, 0);
	gtk_widget_show(wlabel);

	entry = gtk_entry_new();
	if (text)
		gtk_entry_set_text(GTK_ENTRY(entry), text);
	gtk_table_attach(GTK_TABLE(table), entry,
			 1, 2, row, row + 1,
			 GTK_EXPAND | GTK_FILL | GTK_SHRINK,
			 GTK_FILL | GTK_SHRINK,
			 0, 0);
	gtk_widget_show(entry);

	return entry;
}

static void
check_dentry_save(GnomeDesktopEntry *dentry)
{
	FILE *file;
	char *pruned;
	char *new_name;
	char *appsdir;

	file = fopen(dentry->location, "w");
	if (file) {
		fclose(file);
		return;
	}

	pruned = strrchr(dentry->location, '/');
	if (pruned)
		pruned++; /* skip over slash */
	else
		pruned = dentry->location;

	appsdir = gnome_util_home_file ("apps");
	mkdir (appsdir, 0755);

	new_name = g_concat_dir_and_file(appsdir, pruned);
	g_free(dentry->location);
	dentry->location = new_name;
}

#define free_and_nullify(x) { g_free(x); x = NULL; }

static void
properties_ok_callback(GtkWidget *widget, gpointer data)
{
	Properties        *prop;
	GnomeDesktopEntry *dentry;

	prop = data;
	dentry = prop->dentry;

	free_and_nullify(dentry->exec);
	free_and_nullify(dentry->tryexec);
	free_and_nullify(dentry->icon_base);
	free_and_nullify(dentry->docpath);
	free_and_nullify(dentry->info);
	free_and_nullify(dentry->type);

	dentry->exec      = g_strdup(gtk_entry_get_text(GTK_ENTRY(prop->execute_entry)));
	dentry->icon_base = g_strdup(gtk_entry_get_text(GTK_ENTRY(prop->icon_entry)));
	dentry->docpath   = g_strdup(gtk_entry_get_text(GTK_ENTRY(prop->documentation_entry)));
	dentry->info      = g_strdup(gtk_entry_get_text(GTK_ENTRY(prop->description_entry)));
	dentry->type      = g_strdup("Application"); /* FIXME: should handle more cases */
	dentry->terminal  = GTK_TOGGLE_BUTTON(prop->terminal_toggle)->active;

	check_dentry_save(dentry);
	gnome_desktop_entry_save(dentry);

	/* FIXME: should update the button and pixmap */

	gnome_desktop_entry_free(prop->dentry);
	gtk_widget_destroy(prop->dialog);
	g_free(prop);
}

#undef free_and_nullify

static void
properties_cancel_callback(GtkWidget *widget, gpointer data)
{
	Properties *prop;

	prop = data;

	gnome_desktop_entry_free(prop->dentry);
	gtk_widget_destroy(prop->dialog);
	g_free(prop);
}

static GtkWidget *
create_properties_dialog(GnomeDesktopEntry *dentry)
{
	Properties *prop;
	GtkWidget  *dialog;
	GtkWidget  *table;
	GtkWidget  *button;
	GtkWidget  *toggle;

	prop = g_new(Properties, 1);
	prop->dentry = dentry;

	prop->dialog = dialog = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(dialog), _("Launcher properties"));
	gtk_window_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
	gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, TRUE);

	table = gtk_table_new(5, 2, FALSE);
	gtk_container_border_width(GTK_CONTAINER(table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(table), 6);
	gtk_table_set_row_spacings(GTK_TABLE(table), 2);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), table, FALSE, FALSE, 0);
	gtk_widget_show(table);

	prop->execute_entry       = create_text_entry(table, 0, _("Execute"), dentry->exec);
	prop->icon_entry          = create_text_entry(table, 1, _("Icon"), dentry->icon_base);
	prop->documentation_entry = create_text_entry(table, 2, _("Documentation"), dentry->docpath);
	prop->description_entry   = create_text_entry(table, 3, _("Description"), dentry->info);

	prop->terminal_toggle = toggle = gtk_check_button_new_with_label("Run inside terminal");
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(toggle), dentry->terminal ? TRUE : FALSE);
	gtk_table_attach(GTK_TABLE(table), toggle,
			 0, 2, 4, 5,
			 GTK_EXPAND | GTK_FILL | GTK_SHRINK,
			 GTK_FILL | GTK_SHRINK,
			 0, 0);
	gtk_widget_show(toggle);

	button = gtk_button_new_with_label(_("Cancel"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   (GtkSignalFunc) properties_cancel_callback,
			   prop);
	gtk_box_pack_end(GTK_BOX(GTK_DIALOG(dialog)->action_area), button, TRUE, TRUE, 0);
	gtk_widget_show(button);

	gtk_container_border_width(GTK_CONTAINER(GTK_DIALOG(dialog)->action_area), 4);
	
	button = gtk_button_new_with_label(_("Ok"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   (GtkSignalFunc) properties_ok_callback,
			   prop);
	gtk_box_pack_end(GTK_BOX(GTK_DIALOG(dialog)->action_area), button, TRUE, TRUE, 0);
	gtk_widget_show(button);

	gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
			   (GtkSignalFunc) properties_cancel_callback,
			   prop);

	return dialog;
}

static void
properties(GtkWidget *launcher)
{
	GnomeDesktopEntry *dentry;
	char              *path;
	GtkWidget         *dialog;

	path = gtk_object_get_user_data(GTK_OBJECT(launcher));

	dentry = gnome_desktop_entry_load(path);
	if (!dentry) {
		fprintf(stderr,
			"launcher properties: oops, gnome_desktop_entry_load() returned NULL\n"
			"                     on \"%s\"\n", path);
		return;
	}

	dialog = create_properties_dialog(dentry);
	gtk_grab_add(dialog); /* make it modal */
	gtk_widget_show(dialog);
}

static void
create_instance (Panel *panel, char *params, int xpos, int ypos)
{
	GtkWidget *launcher;
	PanelCommand cmd;

	launcher = create_launcher (panel->window, params);

	if (!launcher)
		return;

	cmd.cmd = PANEL_CMD_REGISTER_TOY;
	cmd.params.register_toy.applet = launcher;
	cmd.params.register_toy.id     = APPLET_ID;
	cmd.params.register_toy.xpos   = xpos;
	cmd.params.register_toy.ypos   = ypos;
	cmd.params.register_toy.flags  = APPLET_HAS_PROPERTIES;

	(*panel_cmd_func) (&cmd);
}

gpointer
applet_cmd_func(AppletCommand *cmd)
{
	g_assert(cmd != NULL);

	switch (cmd->cmd) {
		case APPLET_CMD_QUERY:
			return APPLET_ID;

		case APPLET_CMD_INIT_MODULE:
			panel_cmd_func = cmd->params.init_module.cmd_func;
			break;

		case APPLET_CMD_DESTROY_MODULE:
			break;

		case APPLET_CMD_GET_DEFAULT_PARAMS:
			fprintf(stderr, "Launcher: APPLET_CMD_GET_DEFAULT_PARAMS not yet supported\n");
			return g_strdup(""); /* FIXME */

		case APPLET_CMD_CREATE_INSTANCE:
			create_instance(cmd->panel,
					cmd->params.create_instance.params,
					cmd->params.create_instance.xpos,
					cmd->params.create_instance.ypos);
			break;

		case APPLET_CMD_GET_INSTANCE_PARAMS:
			return g_strdup(gtk_object_get_user_data(GTK_OBJECT(cmd->applet)));

		case APPLET_CMD_PROPERTIES:
			properties(cmd->applet);
			break;

		default:
			fprintf(stderr,
				APPLET_ID " applet_cmd_func: Oops, unknown command type %d\n",
				(int) cmd->cmd);
			break;
	}

	return NULL;
}
