#include <assert.h>
#include <gtk/gtk.h>
#include <libguile.h>

/* Brutal breaking of the abstraction barrier.  This function belongs
   into gtksignal.c */

typedef struct _GtkHandler      GtkHandler;

struct _GtkHandler
{
  guint16 id;
  guint signal_type : 13;
  guint object_signal : 1;
  guint blocked : 1;
  guint after : 1;
  GtkSignalFunc func;
  gpointer func_data;
  GtkHandler *next;
};

void
gtk_signal_for_each_handler (GtkObject *object,
			     void (*fun)(GtkSignalFunc, gpointer))
{
  GtkHandler *handlers;

  g_return_if_fail (object != NULL);

  handlers = gtk_object_get_data (object, "signal_handlers");

  while (handlers)
    {
      fun (handlers->func, handlers->func_data);
      handlers = handlers->next;
    }
}

SCM_PROC (s_gtk_init, "gtk-init", 1, 0, 0, sgtk_init);

SCM
sgtk_init (SCM args)
{
  int argc;
  char *argv_storage[1], **argv;

  argc = 0;
  argv_storage[0] = NULL;
  argv = argv_storage;
  
  SCM_DEFER_INTS;
  gtk_init (&argc, &argv);
  SCM_ALLOW_INTS;

  return SCM_UNSPECIFIED;
}

SCM_PROC (s_gtk_main, "gtk-main", 0, 0, 0, sgtk_main);

SCM
sgtk_main ()
{
  gtk_main ();
  return SCM_UNSPECIFIED;
}

/* Typed Pointers */

static long tc16_tptr;

#define TPTR_P(x)    (SCM_NIMP(x) && (SCM_CAR(x)&0xFFFF) == tc16_tptr)
#define TPTR_PTR(x)  ((void *)SCM_CDR(x))
#define TPTR_TYPE(x) (SCM_CAR(x)>>16)

typedef struct {
  char *name;
} tptr_info;

static tptr_info *tptr_infos;
static int n_tptr_infos, max_tptr_infos;

static SCM
new_tptr (char *name)
{
  int i, j;

  // SCM_THREADS_CRITICAL_SECTION_START;
  i = n_tptr_infos++;
  if (n_tptr_infos > max_tptr_infos)
    {
      max_tptr_infos += 32;
      tptr_infos = (tptr_info *)
	scm_must_realloc ((char *)tptr_infos,
			  (max_tptr_infos-32) * sizeof (tptr_info),
			  max_tptr_infos * sizeof (tptr_info),
			  "new_tptr");
      for (j = n_tptr_infos; j < max_tptr_infos; j++)
	tptr_infos[j].name = NULL;
    }
  // SCM_THREADS_CRITICAL_SECTION_END;
  tptr_infos[i].name = name;
  return (SCM)((i << 16) | tc16_tptr);
}

static int
tptr_print (SCM exp, SCM port, scm_print_state *pstate)
{
  scm_gen_puts (scm_regular_string, "#<", port);
  scm_gen_puts (scm_regular_string, tptr_infos[TPTR_TYPE(exp)].name, port);
  scm_gen_puts (scm_regular_string, " ", port);
  scm_intprint ((long)TPTR_PTR(exp), 16, port);
  scm_gen_puts (scm_regular_string, ">", port);
  return 1;
}

static SCM
tptr_equal (SCM exp1, SCM exp2)
{
  return (TPTR_PTR (exp1) == TPTR_PTR (exp2))? SCM_BOOL_T : SCM_BOOL_F;
}

struct scm_smobfuns tptr_smob = {
  scm_mark0,
  scm_free0,
  tptr_print,
  tptr_equal
};

static SCM
wrap_tptr (void *ptr, SCM type)
{
  SCM z;

  SCM_DEFER_INTS;
  SCM_NEWCELL (z);
  SCM_SETCAR (z, type);
  SCM_SETCDR (z, ptr);
  SCM_ALLOW_INTS;
  return z;
}

static void *
get_tptr_ptr (SCM obj)
{
  if (obj == SCM_BOOL_F)
    return NULL;
  return TPTR_PTR (obj);
}

static int
is_tptr_type (SCM obj, SCM type)
{
  return obj == SCM_BOOL_F || SCM_CAR (obj) == type;
}

/* GtkObjects */

static long tc16_gtkobj;

#define GTKOBJP(x) (SCM_NIMP(x) && SCM_CAR(x) == tc16_gtkobj)
#define GTKOBJ(x)  ((GtkObject *)SCM_CDR(x))

static void apply_signal_thunk (GtkObject *, gpointer thunk);

static void mark_handler (GtkSignalFunc func, gpointer data)
{
  if (func == (GtkSignalFunc)apply_signal_thunk)
    scm_gc_mark ((SCM)data);
}

static SCM
gtkobj_mark (SCM exp)
{
  GtkObject *obj;

  SCM_SETGC8MARK (exp);
  obj = GTKOBJ (exp);
  if (obj)
    gtk_signal_for_each_handler (obj, mark_handler);
  return SCM_BOOL_F;
}

static scm_sizet
gtkobj_free (SCM exp)
{
  assert (GTKOBJ (exp) == NULL);
  return 0;
}

static int
gtkobj_print (SCM exp, SCM port, scm_print_state *pstate)
{
  scm_gen_puts (scm_regular_string, "#<", port);
  if (GTKOBJ (exp) == NULL)
    scm_gen_puts (scm_regular_string, "gtk-destroyed-object", port);
  else
    {
      guint tid = GTK_OBJECT_TYPE (GTKOBJ (exp));
      scm_gen_puts (scm_regular_string, gtk_type_name (tid), port);
    }
  scm_gen_puts (scm_regular_string, ">", port);
  return 1;
}

struct scm_smobfuns gtkobj_smob = {
  gtkobj_mark,
  gtkobj_free,
  gtkobj_print,
  NULL
};

static SCM all_gtkobjs;

static void
destroy_gtkobj (GtkObject *obj, SCM cell)
{
  SCM_SETCDR(cell, NULL);
  SCM_SETCAR (all_gtkobjs, scm_delq_x (cell, SCM_CAR (all_gtkobjs)));
}

static char *handle_key = "guile_scm_handle";

static SCM
make_gtkobj (GtkObject *obj)
{
  SCM z;

  SCM_DEFER_INTS;
  SCM_NEWCELL (z);
  SCM_SETCAR (z, tc16_gtkobj);
  SCM_SETCDR (z, obj);
  gtk_signal_connect (obj, "destroy",
		      (GtkSignalFunc)destroy_gtkobj, (gpointer)z);
  gtk_object_set_data (obj, handle_key, (gpointer)z);
  SCM_SETCAR (all_gtkobjs, scm_cons (z, SCM_CAR (all_gtkobjs)));
  SCM_ALLOW_INTS;

  return z;
}

static SCM
wrap_gtkobj (GtkObject *obj)
{
  SCM handle;

  handle = (SCM) gtk_object_get_data (obj, handle_key);
  if (handle == (SCM) NULL)
    handle = make_gtkobj (obj);
  return handle;
}

static int
is_a_gtkobj (guint type, SCM obj)
{
  GtkObject *gobj;

  if (obj == SCM_BOOL_F)
    return 1;
  if (!(SCM_NIMP (obj) && GTKOBJP (obj)))
    return 0;
  gobj = GTKOBJ (obj);
  return gobj && gtk_type_is_a (GTK_OBJECT_TYPE (gobj), type);
}

static GtkObject*
get_gtkobj (SCM obj)
{
  if (obj == SCM_BOOL_F)
    return NULL;
  else
    return GTKOBJ (obj);
}

SCM_PROC (s_gtk_destroyed_p, "gtk-destroyed?", 1, 0, 0, sgtk_destroyed_p);

SCM
sgtk_destroyed_p (SCM obj)
{
  SCM_ASSERT (SCM_NIMP (obj) && GTKOBJP (obj), obj,
	      SCM_ARG1, s_gtk_destroyed_p);
  return GTKOBJ (obj) == NULL? SCM_BOOL_T : SCM_BOOL_F;
}

/* signals */

SCM_PROC (s_gtk_signal_connect, "gtk-signal-connect", 3, 0, 0, sgtk_signal_connect);

static void
apply_signal_thunk (GtkObject *obj, gpointer thunk)
{
  scm_apply ((SCM) thunk, SCM_EOL, SCM_EOL);
}

SCM
sgtk_signal_connect (SCM obj, SCM tag, SCM thunk)
{
  int ret;

  SCM_ASSERT (is_a_gtkobj (gtk_object_get_type(), obj), obj,
	      SCM_ARG1, s_gtk_signal_connect);
  SCM_ASSERT (SCM_NIMP (tag) && SCM_STRINGP (tag), tag,
	      SCM_ARG2, s_gtk_signal_connect);
  SCM_ASSERT (scm_procedure_p (thunk) == SCM_BOOL_T, thunk,
	      SCM_ARG3, s_gtk_signal_connect);

  SCM_DEFER_INTS;
  ret = gtk_signal_connect (GTKOBJ (obj), SCM_CHARS (tag),
			    (GtkSignalFunc)apply_signal_thunk,
			    (gpointer)thunk);
  SCM_ALLOW_INTS;

  return scm_long2num (ret);
}

/* timeouts */

static SCM all_timeouts;

static SCM
apply_timeout_thunk (SCM thunk)
{
  return scm_apply (thunk, SCM_EOL, SCM_EOL);
}

SCM_PROC (s_gtk_timeout_add, "gtk-timeout-add", 2, 0, 0, sgtk_timeout_add);

SCM
sgtk_timeout_add (SCM timeout, SCM thunk)
{
  guint t = scm_num2ulong (timeout, (char *)SCM_ARG1, s_gtk_timeout_add);
  SCM id;
  SCM_ASSERT (scm_procedure_p (thunk) == SCM_BOOL_T, thunk,
	      SCM_ARG2, s_gtk_timeout_add);
  
  SCM_DEFER_INTS;
  id = SCM_MAKINUM (gtk_timeout_add (t, (GtkFunction)apply_timeout_thunk,
				     (gpointer)thunk));
  SCM_SETCAR (all_timeouts, scm_acons (id, thunk, SCM_CAR (all_timeouts)));
  SCM_ALLOW_INTS;
  return id;
}

SCM_PROC (s_gtk_timeout_remove, "gtk-timeout-remove", 1, 0, 0, sgtk_timeout_remove);

SCM
sgtk_timeout_remove (SCM id)
{
  SCM_ASSERT (SCM_INUMP (id), id, SCM_ARG1, s_gtk_timeout_remove);

  SCM_SETCAR (all_timeouts, scm_assq_remove_x (SCM_CAR (all_timeouts), id));
  SCM_DEFER_INTS;
  gtk_timeout_remove (SCM_INUM (id));
  SCM_ALLOW_INTS;
  return SCM_UNSPECIFIED;
}

typedef GtkList GtkListWidget;
#define GTK_LIST_WIDGET(x) GTK_LIST(x)
#define gtk_list_widget_get_type gtk_list_get_type

#include "guile_gtk_defs.c"

void
scm_gtk_init ()
{
  tc16_tptr = scm_newsmob (&tptr_smob);
  tptr_infos = NULL;
  n_tptr_infos = max_tptr_infos = 0;

  tc16_gtkobj = scm_newsmob (&gtkobj_smob);
  all_gtkobjs = scm_permanent_object (scm_cons (SCM_EOL, SCM_BOOL_F));
  all_timeouts = scm_permanent_object (scm_cons (SCM_EOL, SCM_BOOL_F));

#include "guile-gtk.x"

  init_guile_gtk_defs ();
}

      
