/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import org.gnu.glib.Type;

/**
 * The MenuShell class is an abstract base class used to derive the Menu
 * and MenuBar subclasses.
 * <p>
 * A MenuShell is a container of MenuItem objects arranged in a list which can be
 * navigated, selected, and activated by the user to perform application functions. A
 * MenuItem can have a submenu associated with it, allowing for nested hierarchical
 * menus.
 */
public abstract class MenuShell extends Container {

	protected MenuShell(int handle) {
	    setHandle(handle);
	    initializeEventHandlers();
	}

    protected MenuShell() {}

	protected void noopInit() {}
	
	/**
	 * Append a new MenuItem to the end of the MenuShell's item list.
	 * 
	 * @param child The MenuItem to add
	 */
	public void append(MenuItem child) {
		MenuShell.gtk_menu_shell_append(getHandle(), child.getHandle());
	}
	
	/**
	 * Adds a new MenuItem to the beginning of the MenuShell's item list.
	 * 
	 * @param child The MenuItem to add.
	 */
	public void prepend(MenuItem child) {
		MenuShell.gtk_menu_shell_prepend(getHandle(), child.getHandle());
	}
	
	/**
	 * Adds a new MenuItem to the MenuShell's item list at the requested position.
	 * 
	 * @param child The MenuItem to add.
	 * @param position The position in the item list where <code>child</code> is added.  Positions
	 * are zero based.
	 */
	public void insert(MenuItem child, int position) {
		MenuShell.gtk_menu_shell_insert(getHandle(), child.getHandle(), position);
	}
	
	/**
	 * Deactivates the MenuShell.  Typically this results in the MenuShell being erased from
	 * the screen.
	 */
	public void deactivate() {
		MenuShell.gtk_menu_shell_deactivate(getHandle());
	}
	
	/**
	 * Selects the MenuItem from the MenuShell.
	 * 
	 * @param menuItem The MenuItem to select.
	 */
	public void selectItem(MenuItem menuItem) {
		MenuShell.gtk_menu_shell_select_item(getHandle(), menuItem.getHandle());
	}
	
	/**
	 * Deselects the currently selected item from the MenuShell.
	 */
	public void deselect() {
		MenuShell.gtk_menu_shell_deselect(getHandle());
	}
	
	/**
	 * Activates the MenuItem within the MenuShell.
	 * 
	 * @param menuItem The MenuItem to activate.
	 * @param forceDeactivate If <code>true</code>, force the deactivation of the MenuShell
	 * after the MenuItem is activated.
	 */
	public void activateItem(MenuItem menuItem, boolean forceDeactivate)  {
		MenuShell.gtk_menu_shell_activate_item(getHandle(), menuItem.getHandle(), forceDeactivate);
	}
	
	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static Type getType() {
		return new Type(gtk_menu_shell_get_type());
	}

	/****************************************
	 * BEGINNING OF GENERATED CODE
	 ****************************************/
	native static final protected int gtk_menu_shell_get_type();
	native static final protected void gtk_menu_shell_append(int menu_shell, int child);
	native static final protected void gtk_menu_shell_prepend(int menu_shell, int child);
	native static final protected void gtk_menu_shell_insert(int menu_shell, int child, int position);
	native static final protected void gtk_menu_shell_deactivate(int menu_shell);
	native static final protected void gtk_menu_shell_select_item(int menu_shell, int menuItem);
	native static final protected void gtk_menu_shell_deselect(int menu_shell);
	native static final protected void gtk_menu_shell_activate_item(int menu_shell, int menuItem, boolean forceDeactivate);
	/****************************************
	 * END OF GENERATED CODE
	 ****************************************/
}
