/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gdk;

import org.gnu.glib.GObject;
import org.gnu.glib.Type;
import org.gnu.glib.Handle;

/**
 *  Screen objects are the GDK representation of a physical screen.
 *  It is used throughout GDK and GTK+ to specify which screen the top
 *  level windows are to be displayed on. It is also used to query the
 *  screen specification and default settings such as the default
 *  colormap, the screen width, etc.
 *  
 *  <p>Note that a Screen may consist of multiple monitors which are
 *  merged to form a large screen area.
 */
public class Screen extends GObject {
	
    /**
     * Creates and object representing the default
     * Screen for the default Display.
     */
    public Screen() {
        super(gdk_screen_get_default());
    }
	
    public Screen(Handle handle) {
        super(handle);
    }

    /**
     * Returns the default Colormap for this screen.
     */
    public Colormap getDefaultColormap() {
        return 
            Colormap.
            getColormapFromHandle(
                gdk_screen_get_default_colormap(getHandle()));
    }
	
    /**
     * Sets the default Colormap for this screen.
     * @param colormap
     */
    public void setDefaultColormap(Colormap colormap) {
        gdk_screen_set_default_colormap(getHandle(), colormap.getHandle());
    }
	
    /**
     * Returns the system's default Colormap for Screens
     */
    public Colormap getSystemColormap() {
        return 
            Colormap.
            getColormapFromHandle( 
                gdk_screen_get_system_colormap(getHandle()) );
    }
	
    /**
     * Returns the system's default Visual for Screens.  This is the
     * Visual for the root window of the Display.  
     */
    public Visual getSystemVisual() {
        return 
            Visual.
            getVisualFromHandle( gdk_screen_get_system_visual(getHandle()) );
    }
	
    /**
     * Gets the preferred colormap for rendering image data on screen. 
     */
    public Colormap getRGBColormap() {
        return 
            Colormap.
            getColormapFromHandle( gdk_screen_get_rgb_colormap(getHandle()) );
    }
	
    public Colormap getRGBAColormap() {
        return 
            Colormap.
            getColormapFromHandle( gdk_screen_get_rgba_colormap(getHandle()) );
    }

    /**
     * Gets a "preferred visual" chosen by RGB for rendering image data 
     * on screen. 
     */
    public Visual getRGBVisual() {
        return 
            Visual.
            getVisualFromHandle( gdk_screen_get_rgb_visual(getHandle()) );
    }

    public Visual getRGBAVisual() {
        return 
            Visual.
            getVisualFromHandle( gdk_screen_get_rgba_visual(getHandle()) );
    }

    /**
     * Returns the root window of the Screen.
     */
    public Window getRootWindow() {
        return 
            Window.
            getWindowFromHandle( gdk_screen_get_root_window(getHandle()) );
    }
	
    /**
     * Returns the display that belongs to this Screen.
     */
    public Display getDisplay() {
        return Display.getDisplayFromHandle( gdk_screen_get_display(getHandle()) );
    }

    /**
     * Gets the index of this Screen among the Screens in the Display
     * to which it belongs. 
     */
    public int getNumber() {
        return gdk_screen_get_number(getHandle());
    }
	
    /**
     * Gets the height of the Screen in pixels.
     */
    public int getHeight() {
        return gdk_screen_get_height(getHandle());
    }
	
    /**
     * Gets the width of the Screen in pixels.
     */
    public int getWidth() {
        return gdk_screen_get_width(getHandle());
    }
	
    /**
     * Gets the height of the Screen in milimeters.
     */
    public int getHeightMM() {
        return gdk_screen_get_height_mm(getHandle());
    }
	
    /**
     * Gets the width of the Screen in milimeters.
     */
    public int getWidthMM() {
        return gdk_screen_get_width_mm(getHandle());
    }
	
    /**
     * List the Visuals for this Screen.  A Visual describes a 
     * hardware image data format. For example, a visual might 
     * support 24-bit color, or 8-bit color, and might expect 
     * pixels to be in a certain format.
     */
    public Visual[] listVisuals() {
        Handle[] hndls = gdk_screen_list_visuals(getHandle());
        Visual[] vis = new Visual[hndls.length];
        for (int i = 0; i < hndls.length; i++) {
            vis[i] = Visual.getVisualFromHandle( hndls[i] );
        }
        return vis;
    }
	
    /**
     * Returns a list of all known toplevel Windows on the Screen.
     * A toplevel Window is a child of the root window. 
     */
    public Window[] getToplevelWindows() {
        Handle[] hndls = gdk_screen_get_toplevel_windows(getHandle());
        Window[] win = new Window[hndls.length];
        for (int i = 0; i < hndls.length; i++) {
            win[ i ] = Window.getWindowFromHandle( hndls[ i ] );
        }
        return win;
    }

    /**
     * Returns the name to pass to the Display constructor to 
     * create a Display with this Screen as the default Screen.
     */
    public String makeDisplayName() {
        return gdk_screen_make_display_name(getHandle());
    }

    /**
     * Returns the number of monitors which Screen consists of.
     */
    public int getNumMonitors() {
        return gdk_screen_get_n_monitors(getHandle());
    }
	
    /**
     * Returns a Rectangle representing the size and position of the
     * individual monitor within the entire Screen area.
     * @param monitorNum
     */
    public Rectangle getMonitorGeometry(int monitorNum) {
        Handle hndl = gdk_screen_get_monitor_geometry(getHandle(), monitorNum);
        return Rectangle.getRectangleFromHandle(hndl);
    }
	
    /**
     * Returns the monitor number in which the point (x,y) is located.
     * @param x
     * @param y
     */
    public int getMonitorAtPoint(int x, int y) {
        return gdk_screen_get_monitor_at_point(getHandle(), x, y);
    }
	
    /**
     * Returns the number of the monitor in which the largest area
     * of the bounding rectangle of the Window resides.
     * @param win
     */
    public int getMonitorAtWindow(Window win) {
        return gdk_screen_get_monitor_at_window(getHandle(), win.getHandle());
    }
	
    /**
     * On X11, sends an X ClientMessage event to all toplevel windows 
     * on screen.
     * 
     * Toplevel windows are determined by checking for the WM_STATE 
     * property, as described in the Inter-Client Communication Conventions 
     * Manual (ICCCM). If no windows are found with the WM_STATE property 
     * set, the message is sent to all children of the root window.
     * 
     * On Windows, broadcasts a message registered with the name 
     * GDK_WIN32_CLIENT_MESSAGE to all top-level windows. The amount of 
     * data is limited to one long, i.e. four bytes.
     * 
     * @param evt
     */
    public void broadcastClientMessage(Event evt) {
        gdk_screen_broadcast_client_message(getHandle(), evt.getHandle());
    }

    public Type getType() {
        return new Type(gdk_screen_get_type());
    }

    /**
     * Internal static factory method to be used only internally by Java-Gnome.
     */
    public static Screen getScreenFromHandle( Handle hndl ) {
        if ( hndl != null ) {
            GObject obj = GObject.getGObjectFromHandle( hndl );
            return ( obj != null ) ? (Screen)obj : new Screen(hndl);
        }
        return null;
    }

    native static final protected int gdk_screen_get_type();
    native static final protected Handle gdk_screen_get_default_colormap(Handle screen);
    native static final protected void gdk_screen_set_default_colormap(Handle screen, Handle colormap);
    native static final protected Handle gdk_screen_get_system_colormap(Handle screen);
    native static final protected Handle gdk_screen_get_system_visual(Handle screen);
    native static final protected Handle gdk_screen_get_rgb_colormap(Handle screen);
    native static final protected Handle gdk_screen_get_rgba_colormap(Handle screen);
    native static final protected Handle gdk_screen_get_rgb_visual(Handle screen);
    native static final protected Handle gdk_screen_get_rgba_visual(Handle screen);
    native static final protected Handle gdk_screen_get_root_window(Handle screen);
    native static final protected Handle gdk_screen_get_display(Handle screen);
    native static final protected int gdk_screen_get_number(Handle screen);
    native static final int gdk_screen_get_height(Handle screen);
    native static final int gdk_screen_get_width(Handle screen);
    native static final protected int gdk_screen_get_height_mm(Handle screen);
    native static final protected int gdk_screen_get_width_mm(Handle screen);
    native static final protected Handle[] gdk_screen_list_visuals(Handle screen);
    native static final protected Handle[] gdk_screen_get_toplevel_windows(Handle screen);
    native static final protected String gdk_screen_make_display_name(Handle screen);
    native static final protected int gdk_screen_get_n_monitors(Handle screen);
    native static final protected Handle gdk_screen_get_monitor_geometry(Handle screen, int monitorNum);
    native static final protected int gdk_screen_get_monitor_at_point(Handle screen, int x, int y);
    native static final protected int gdk_screen_get_monitor_at_window(Handle screen, Handle window);
    native static final protected void gdk_screen_broadcast_client_message(Handle screen, Handle event);
    native static final Handle gdk_screen_get_default();
    //	native static final protected boolean gdk_screen_get_setting(Handle screen, String name, Handle value);
}
