/*
 * Remote Laboratory Instrumentation Server
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * (c) 2009 Timothy Pearson
 * Raptor Engineering
 * http://www.raptorengineeringinc.com
 */

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <getopt.h>
#include "parameters.h"
#include "gpib_functions.h"
#include "gpib/ib.h"

#include <time.h>

#include <tqimage.h>

#define SCOPE_TRACE_MAX_POINTS 131072

extern char falpha[1024];
unsigned char scope_raw_screenshot_data[4194304];
double scope_raw_trace_data[SCOPE_TRACE_MAX_POINTS];
double scope_raw_position_data[SCOPE_TRACE_MAX_POINTS];

char scope_segarray[4194304];

unsigned long scopeScreenWidth (const char * scopeType) {
	if (strcmp("HP54600OS", scopeType) == 0) {
		return 512;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		return 640;
	}
	else {
		return 1;
	}
}

unsigned long scopeScreenHeight (const char * scopeType) {
	if (strcmp("HP54600OS", scopeType) == 0) {
		return 280;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		return 480;
	}
	else {
		return 1;
	}
}

unsigned long scopeScreenSize (const char * scopeType) {
	if (strcmp("HP54600OS", scopeType) == 0) {
		return scopeScreenWidth(scopeType)*scopeScreenHeight(scopeType)*3;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		return scopeScreenWidth(scopeType)*scopeScreenHeight(scopeType)*3;
	}
	else {
		return 1;
	}
}

int gpib_read_binblock(int ud, int max_num_bytes, const char * scopeType)
{
	long array_pointer;
	long ai;

	int x;
	int y;

	array_pointer = 0;

	#ifdef ENABLE_EXTRA_DEBUGGING
	printf("[DEBG] Trying to read %i bytes from GPIB device...\n", max_num_bytes);
	#endif

	ibtmo(ud, T30s);
	ibeos(ud, 0x0);

	ai = gpib_read_array(ud, max_num_bytes, scope_segarray);
	if (ai == -1) {
		return 1;
	}
	else {
		if (strcmp("HP54600OS", scopeType) == 0) {
			ai = 0;
			for (x=0;x<scopeScreenWidth(scopeType);x++) {
				for (y=0;y<scopeScreenHeight(scopeType);y++) {
					if ((x & 0x7) == 0) scope_raw_screenshot_data[y+(x*scopeScreenHeight(scopeType))] = (scope_segarray[((x >> 3)+(y*(560/8)))+17] & 0x80) << 0;
					if ((x & 0x7) == 1) scope_raw_screenshot_data[y+(x*scopeScreenHeight(scopeType))] = (scope_segarray[((x >> 3)+(y*(560/8)))+17] & 0x40) << 1;
					if ((x & 0x7) == 2) scope_raw_screenshot_data[y+(x*scopeScreenHeight(scopeType))] = (scope_segarray[((x >> 3)+(y*(560/8)))+17] & 0x20) << 2;
					if ((x & 0x7) == 3) scope_raw_screenshot_data[y+(x*scopeScreenHeight(scopeType))] = (scope_segarray[((x >> 3)+(y*(560/8)))+17] & 0x10) << 3;
					if ((x & 0x7) == 4) scope_raw_screenshot_data[y+(x*scopeScreenHeight(scopeType))] = (scope_segarray[((x >> 3)+(y*(560/8)))+17] & 0x08) << 4;
					if ((x & 0x7) == 5) scope_raw_screenshot_data[y+(x*scopeScreenHeight(scopeType))] = (scope_segarray[((x >> 3)+(y*(560/8)))+17] & 0x04) << 5;
					if ((x & 0x7) == 6) scope_raw_screenshot_data[y+(x*scopeScreenHeight(scopeType))] = (scope_segarray[((x >> 3)+(y*(560/8)))+17] & 0x02) << 6;
					if ((x & 0x7) == 7) scope_raw_screenshot_data[y+(x*scopeScreenHeight(scopeType))] = (scope_segarray[((x >> 3)+(y*(560/8)))+17] & 0x01) << 7;
					if (scope_raw_screenshot_data[y+(x*scopeScreenHeight(scopeType))] == 0x0) scope_raw_screenshot_data[y+(x*scopeScreenHeight(scopeType))] = 255;
					if (scope_raw_screenshot_data[y+(x*scopeScreenHeight(scopeType))] == 0x80) scope_raw_screenshot_data[y+(x*scopeScreenHeight(scopeType))] = 0;
				}
			}
		}
		else {
			return 2;
		}
	}

	ibtmo(ud, T10s);

	#ifdef ENABLE_EXTRA_DEBUGGING
	printf("[DEBG] Read %li bytes from GPIB device\n", array_pointer);
	#endif

	return 0;
}

int scope_get_screenshot_stage2(const char * scopeType, int gpibDevice) {
	long bytestosend;
	int k;
	int m;

	if ((strcmp("HP54600OS", scopeType) == 0) || (strcmp("TDS744AOS", scopeType) == 0) || (strcmp("TDS744COS", scopeType) == 0)) {
		printf("[INFO] Getting scope screenshot [Stage 2]\n\r");
		if (strcmp("HP54600OS", scopeType) == 0) {
			if (gpib_read_binblock(gpibDevice, 19768, scopeType) == 0) {
				return 0;
			}
			else {
				return 1;
			}
		}
		else if ((strcmp("TDS744AOS", scopeType) == 0)
			|| (strcmp("TDS744COS", scopeType) == 0)) {
			gpib_read_binary(gpibDevice, scopeScreenSize(scopeType));

			int bpp;
			TQImage bmp;
			bmp.load("/tmp/current_scope_screenshot.bmp");
			if (bmp.isNull()) {
				printf("[WARN] Unable to load screenshot bitmap\n\r");
				return 1;
			}
			bpp = bmp.depth();
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Found BPP of %d\n\r", bpp);
			#endif
			bytestosend = 0;
			for (m=0;m<scopeScreenWidth(scopeType);m++) {
				for (k=0;k<scopeScreenHeight(scopeType);k++) {
					TQRgb pixel = bmp.pixel(m, k);
					scope_raw_screenshot_data[bytestosend] = tqRed(pixel);
					bytestosend = bytestosend + 1;
					scope_raw_screenshot_data[bytestosend] = tqGreen(pixel);
					bytestosend = bytestosend + 1;
					scope_raw_screenshot_data[bytestosend] = tqBlue(pixel);
					bytestosend = bytestosend + 1;
				}
			}
			return 0;
		}
	}
	return 1;
}

int scope_reset(const char * funcgenType, int gpibDevice) {
	// FIXME
	// GNDN
	return 0;
}

int scope_get_screenshot(const char * scopeType, int gpibDevice) {
	if ((strcmp("HP54600OS", scopeType) == 0) || (strcmp("TDS744AOS", scopeType) == 0) || (strcmp("TDS744COS", scopeType) == 0)) {
		printf("[INFO] Getting scope screenshot [Stage 1]\n\r");
		if (strcmp("HP54600OS", scopeType) == 0) {
			sprintf(falpha, "PRINT?");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else if ((strcmp("TDS744AOS", scopeType) == 0)
			|| (strcmp("TDS744COS", scopeType) == 0)) {
			sprintf(falpha, "HARDCOPY:FORMAT BMPCOLOR");
			if (gpib_write(gpibDevice, falpha) == 0) {
				#ifdef ENABLE_EXTRA_DEBUGGING
				printf("[DEBG] Wrote: %s\n\r", falpha);
				#endif
				sprintf(falpha, "HARDCOPY:LAYOUT PORTRAIT");
				if (gpib_write(gpibDevice, falpha) == 0) {
					#ifdef ENABLE_EXTRA_DEBUGGING
					printf("[DEBG] Wrote: %s\n\r", falpha);
					#endif
					sprintf(falpha, "HARDCOPY:PALETTE HARDCOPY");
					if (gpib_write(gpibDevice, falpha) == 0) {
						#ifdef ENABLE_EXTRA_DEBUGGING
						printf("[DEBG] Wrote: %s\n\r", falpha);
						#endif
						sprintf(falpha, "HARDCOPY:PORT GPIB");
						if (gpib_write(gpibDevice, falpha) == 0) {
							#ifdef ENABLE_EXTRA_DEBUGGING
							printf("[DEBG] Wrote: %s\n\r", falpha);
							#endif
							sprintf(falpha, "HARDCOPY START");
							if (gpib_write(gpibDevice, falpha) == 0) {
								#ifdef ENABLE_EXTRA_DEBUGGING
								printf("[DEBG] Wrote: %s\n\r", falpha);
								#endif
								return 0;
							}
						}
					}
				}
			}
		}
	}
	return 1;
}

int scope_set_timebase(float desired_timebase,const char * scopeType, int gpibDevice) {
	if ((strcmp("HP54600OS", scopeType) == 0) || (strcmp("TDS744AOS", scopeType) == 0) || (strcmp("TDS744COS", scopeType) == 0)) {
		printf("[INFO] Setting scope timebase to %E\n\r", desired_timebase);
		if (strcmp("HP54600OS", scopeType) == 0) {
			sprintf(falpha, "TIM:RANG %E", desired_timebase);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else if ((strcmp("TDS744AOS", scopeType) == 0)
			|| (strcmp("TDS744COS", scopeType) == 0)) {
			sprintf(falpha, "HORIZONTAL:MAIN:SCALE %E", desired_timebase);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return -1;
		}
	}
	else {
		return 1;
	}
}

int scope_get_timebase(double * retval, const char * scopeType, int gpibDevice) {
	char floatstring[1024];
	long ai;
	int max_num_bytes = 0;

	if (strcmp("HP54600OS", scopeType) == 0) {
		// FIXME
		// Not supported (yet)
		return -1;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		// Send request
		printf("[INFO] Getting scope timebase\n\r");
		sprintf(falpha,"HORIZONTAL:MAIN:SCALE?");
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Writing: %s\n\r", falpha);
		#endif
		if (gpib_write(gpibDevice, falpha) == 0) {
			max_num_bytes = 24;	// Request more bytes than are possible to ensure no bytes are left behind
		}
		else {
			return 2;
		}

		// Read response
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Trying to read %i bytes from GPIB device...\n", max_num_bytes);
		#endif

		ai = gpib_read_array(gpibDevice, max_num_bytes, floatstring);
		if (ai == -1) {
			return 1;
		}
		else {
			floatstring[ai]=0;
			*retval = atoi(floatstring);
		}

		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Read %li bytes from GPIB device\n", ai);
		#endif

		return 0;
	}
	else {
		return -1;
	}
}

int scope_set_volts_div(int desired_channel, double desired_volts, const char * scopeType, int gpibDevice) {
	if ((strcmp("HP54600OS", scopeType) == 0) || (strcmp("TDS744AOS", scopeType) == 0) || (strcmp("TDS744COS", scopeType) == 0)) {
		printf("[INFO] Setting scope volts/div on channel %d to %E\n\r", desired_channel, desired_volts);
		if (strcmp("HP54600OS", scopeType) == 0) {
			sprintf(falpha, "CHAN%d:RANG %E", desired_channel, desired_volts);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else if ((strcmp("TDS744AOS", scopeType) == 0)
			|| (strcmp("TDS744COS", scopeType) == 0)) {
			sprintf(falpha, "CH%d:SCALE %E", desired_channel, desired_volts);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return -1;
		}
	}
	else {
		return 1;
	}
}

int scope_set_acquisition(int status,const char * scopeType, int gpibDevice) {
	if ((strcmp("HP54600OS", scopeType) == 0) || (strcmp("TDS744AOS", scopeType) == 0) || (strcmp("TDS744COS", scopeType) == 0)) {
		printf("[INFO] Setting scope run status to %d\n\r", status);
		if (strcmp("HP54600OS", scopeType) == 0) {
			if (status == 0) {
				sprintf(falpha, "STOP");
			}
			else {
				sprintf(falpha, "RUN");
			}
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else if ((strcmp("TDS744AOS", scopeType) == 0)
			|| (strcmp("TDS744COS", scopeType) == 0)) {
			sprintf(falpha, "ACQUIRE:STATE %d", status);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return -1;
		}
	}
	else {
		return 1;
	}
}

int scope_get_acquisition(int * retval, const char * scopeType, int gpibDevice) {
	char floatstring[1024];
	long ai;
	int max_num_bytes = 0;

	if (strcmp("HP54600OS", scopeType) == 0) {
		// FIXME
		// Not supported (yet)
		return -1;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		// Send request
		printf("[INFO] Getting run state\n\r");
		sprintf(falpha,"ACQUIRE:STATE?");
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Writing: %s\n\r", falpha);
		#endif
		if (gpib_write(gpibDevice, falpha) == 0) {
			max_num_bytes = 24;	// Request more bytes than are possible to ensure no bytes are left behind
		}
		else {
			return 2;
		}

		// Read response
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Trying to read %i bytes from GPIB device...\n", max_num_bytes);
		#endif

		ai = gpib_read_array(gpibDevice, max_num_bytes, floatstring);
		if (ai == -1) {
			return 1;
		}
		else {
			floatstring[ai]=0;
			*retval = atoi(floatstring);
		}

		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Read %li bytes from GPIB device\n", ai);
		#endif

		return 0;
	}
	else {
		return -1;
	}
}

int scope_set_channel_state(int desired_channel, int status,const char * scopeType, int gpibDevice) {
	if ((strcmp("HP54600OS", scopeType) == 0) || (strcmp("TDS744AOS", scopeType) == 0) || (strcmp("TDS744COS", scopeType) == 0)) {
		printf("[INFO] Setting channel %d state to %i\n\r", desired_channel, status);
		if (strcmp("HP54600OS", scopeType) == 0) {
			if (status == 0) {
				sprintf(falpha, "BLAN CHAN%d", desired_channel);
			}
			else if (status == 1) {
				sprintf(falpha, "VIEW CHAN%d", desired_channel);
			}
			else {
				return 2;
			}
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else if ((strcmp("TDS744AOS", scopeType) == 0)
			|| (strcmp("TDS744COS", scopeType) == 0)) {
			if (status == 0) {
				sprintf(falpha, "SELECT:CH%d OFF", desired_channel);
			}
			else if (status == 1) {
				sprintf(falpha, "SELECT:CH%d ON", desired_channel);
			}
			else {
				return 2;
			}
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				usleep(2*1000000);	// The scope is slow to respond, and also blind to commands during the update window!  This makes sure that the scope will actually process subsequent commands
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return -1;
		}
	}
	else {
		return 1;
	}
}

int scope_get_channel_state(int * retval, int desired_channel, const char * scopeType, int gpibDevice) {
	char floatstring[1024];
	long ai;
	int max_num_bytes = 0;

	printf("[INFO] Getting channel %d state\n\r", desired_channel);
	if (strcmp("HP54600OS", scopeType) == 0) {
		// FIXME
		// Not supported (yet)
		return -1;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		sprintf(falpha, "SELECT:CH%d?", desired_channel);
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Writing: %s\n\r", falpha);
		#endif
		if (gpib_write(gpibDevice, falpha) == 0) {
			max_num_bytes = 24;	// Request more bytes than are possible to ensure no bytes are left behind
		}
		else {
			return 2;
		}

		// Read response
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Trying to read %i bytes from GPIB device...\n", max_num_bytes);
		#endif

		ai = gpib_read_array(gpibDevice, max_num_bytes, floatstring);
		if (ai == -1) {
			return 1;
		}
		else {
			floatstring[ai]=0;
			*retval = atoi(floatstring);
		}

		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Read %li bytes from GPIB device\n", ai);
		#endif

		return 0;
	}
	else {
		return -1;
	}
}

int scope_set_trigger_channel(int desired_channel,const char * scopeType, int gpibDevice) {
	if ((strcmp("HP54600OS", scopeType) == 0) || (strcmp("TDS744AOS", scopeType) == 0) || (strcmp("TDS744COS", scopeType) == 0)) {
		printf("[INFO] Setting scope trigger channel to %d\n\r", desired_channel);
		if (strcmp("HP54600OS", scopeType) == 0) {
			sprintf(falpha, "TRIG:SOUR CHAN%d", desired_channel);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else if ((strcmp("TDS744AOS", scopeType) == 0)
			|| (strcmp("TDS744COS", scopeType) == 0)) {
			sprintf(falpha, "TRIGGER:MAIN:EDGE:SOURCE CH%d", desired_channel);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return -1;
		}
	}
	else {
		return 1;
	}
}

int scope_get_trigger_channel(int * retval, const char * scopeType, int gpibDevice) {
	char floatstring[1024];
	long ai;
	int max_num_bytes = 0;

	printf("[INFO] Getting trigger channel\n\r");
	if (strcmp("HP54600OS", scopeType) == 0) {
		// FIXME
		// Not supported (yet)
		return -1;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		sprintf(falpha, "TRIGGER:MAIN:EDGE:SOURCE?");
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Writing: %s\n\r", falpha);
		#endif
		if (gpib_write(gpibDevice, falpha) == 0) {
			max_num_bytes = 24;	// Request more bytes than are possible to ensure no bytes are left behind
		}
		else {
			return 2;
		}

		// Read response
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Trying to read %i bytes from GPIB device...\n", max_num_bytes);
		#endif

		ai = gpib_read_array(gpibDevice, max_num_bytes, floatstring);
		if (ai == -1) {
			return 1;
		}
		else {
			floatstring[ai]=0;
			TQString retchan(floatstring);
			if (retchan.startsWith("CH1")) *retval = 1;
			else if (retchan.startsWith("CH2")) *retval = 2;
			else if (retchan.startsWith("CH3")) *retval = 3;
			else if (retchan.startsWith("CH4")) *retval = 4;
			else if (retchan.startsWith("AUX")) *retval = -1;
			else if (retchan.startsWith("LINE")) *retval = -2;
			else {
				*retval = 0;
				return -1;
			}
		}

		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Read %li bytes from GPIB device\n", ai);
		#endif

		return 0;
	}
	else {
		return -1;
	}
}

int scope_set_trigger_level(float desired_level,const char * scopeType, int gpibDevice) {
	if ((strcmp("HP54600OS", scopeType) == 0) || (strcmp("TDS744AOS", scopeType) == 0) || (strcmp("TDS744COS", scopeType) == 0)) {
		printf("[INFO] Setting scope trigger level to %f\n\r", desired_level);
		if (strcmp("HP54600OS", scopeType) == 0) {
			sprintf(falpha, "TRIG:LEV %E", desired_level);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else if ((strcmp("TDS744AOS", scopeType) == 0)
			|| (strcmp("TDS744COS", scopeType) == 0)) {
			sprintf(falpha, "TRIGGER:MAIN:LEVEL %f", desired_level);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return -1;
		}
	}
	else {
		return 1;
	}
}

int scope_get_trigger_level(double * retval, const char * scopeType, int gpibDevice) {
	char floatstring[1024];
	long ai;
	int max_num_bytes = 0;

	if (strcmp("HP54600OS", scopeType) == 0) {
		// FIXME
		// Not supported (yet)
		return -1;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		// Send request
		printf("[INFO] Getting trigger level\n\r");
		sprintf(falpha,"TRIGGER:MAIN:LEVEL?");
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Writing: %s\n\r", falpha);
		#endif
		if (gpib_write(gpibDevice, falpha) == 0) {
			max_num_bytes = 24;	// Request more bytes than are possible to ensure no bytes are left behind
		}
		else {
			return 2;
		}

		// Read response
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Trying to read %i bytes from GPIB device...\n", max_num_bytes);
		#endif

		ai = gpib_read_array(gpibDevice, max_num_bytes, floatstring);
		if (ai == -1) {
			return 1;
		}
		else {
			floatstring[ai]=0;
			*retval = atof(floatstring);
		}

		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Read %li bytes from GPIB device\n", ai);
		#endif

		return 0;
	}
	else {
		return -1;
	}
}

int scope_set_channel_position(int desired_channel, float desired_level,const char * scopeType, int gpibDevice) {
	if ((strcmp("HP54600OS", scopeType) == 0) || (strcmp("TDS744AOS", scopeType) == 0) || (strcmp("TDS744COS", scopeType) == 0)) {
		printf("[INFO] Setting scope channel %d level to %f\n\r", desired_channel, desired_level);
		if (strcmp("HP54600OS", scopeType) == 0) {
			sprintf(falpha, "CHAN%d:OFFS %E", desired_channel, desired_level);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else if ((strcmp("TDS744AOS", scopeType) == 0)
			|| (strcmp("TDS744COS", scopeType) == 0)) {
			sprintf(falpha, "CH%d:POSITION %f", desired_channel, desired_level);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return -1;
		}
	}
	else {
		return 1;
	}
}

int scope_perform_initial_setup(const char * scopeType, int gpibDevice) {
	// Send request
	printf("[INFO] Configuring oscilloscope\n\r");
	if (strcmp("HP54600OS", scopeType) == 0) {
		// FIXME
		// Not supported (yet)
		return -1;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		sprintf(falpha,"HEAD OFF");
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Writing: %s\n\r", falpha);
		#endif
		if (gpib_write(gpibDevice, falpha) == 0) {
			sprintf(falpha,"DATA:ENCDG RIBINARY");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				sprintf(falpha,"DATA:WIDTH 2");
				#ifdef ENABLE_EXTRA_DEBUGGING
				printf("[DEBG] Writing: %s\n\r", falpha);
				#endif
				if (gpib_write(gpibDevice, falpha) == 0) {
					sprintf(falpha,"DATA:START 1");
					#ifdef ENABLE_EXTRA_DEBUGGING
					printf("[DEBG] Writing: %s\n\r", falpha);
					#endif
					if (gpib_write(gpibDevice, falpha) == 0) {
						sprintf(falpha,"DATA:STOP %d", SCOPE_TRACE_MAX_POINTS-1);
						#ifdef ENABLE_EXTRA_DEBUGGING
						printf("[DEBG] Writing: %s\n\r", falpha);
						#endif
						return 0;
					}
					else {
						return -2;
					}
				}
				else {
					return -2;
				}
			}
			else {
				return -2;
			}
		}
		else {
			return -2;
		}
	}
	else {
		return -1;
	}
}

#define ENABLE_PROFILING

long scope_get_channel_trace(int desired_channel, const char * scopeType, int gpibDevice) {
	long array_pointer;
	long ai;

#ifdef ENABLE_PROFILING
struct timespec tp1, tp2, tp3, tp4, tp5, tp6, tp7;
#endif

#ifdef ENABLE_PROFILING
clock_gettime(CLOCK_REALTIME, &tp1);
#endif

	// Send request
	printf("[INFO] Getting oscilloscope trace for channel %d [Stage 1]\n\r", desired_channel);
	if (strcmp("HP54600OS", scopeType) == 0) {
		// FIXME
		// Not supported (yet)
		return -1;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		// We need to get/parse the preamble, then obtain and adjust the trace data
		sprintf(falpha,"DATA:SOURCE CH%d", desired_channel);
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Writing: %s\n\r", falpha);
		#endif
		if (gpib_write(gpibDevice, falpha) != 0) {
			return -1;
		}
		sprintf(falpha,"DATA:STOP %d", SCOPE_TRACE_MAX_POINTS-1);
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Writing: %s\n\r", falpha);
		#endif
		if (gpib_write(gpibDevice, falpha) == 0) {
#ifdef ENABLE_PROFILING
clock_gettime(CLOCK_REALTIME, &tp2);
printf("[PROFILE] %f s\n\r", ((tp2.tv_nsec+(tp2.tv_sec*1e9))-(tp1.tv_nsec+(tp1.tv_sec*1e9)))/1e9);
#endif
			sprintf(falpha,"WFMPRE?");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
#ifdef ENABLE_PROFILING
clock_gettime(CLOCK_REALTIME, &tp3);
printf("[PROFILE] %f s\n\r", ((tp3.tv_nsec+(tp3.tv_sec*1e9))-(tp2.tv_nsec+(tp2.tv_sec*1e9)))/1e9);
#endif
				// Read response
				#ifdef ENABLE_EXTRA_DEBUGGING
				printf("[DEBG] Trying to read %i bytes from GPIB device...\n", 65535);
				#endif
				ai = gpib_read_array(gpibDevice, 65535, scope_segarray);
				if (ai == -1) {
					return -1;
				}
				else {
					#ifdef ENABLE_EXTRA_DEBUGGING
					printf("[DEBG] Read %li bytes from GPIB device\n", ai);
					#endif
#ifdef ENABLE_PROFILING
clock_gettime(CLOCK_REALTIME, &tp4);
printf("[PROFILE] %f s\n\r", ((tp4.tv_nsec+(tp4.tv_sec*1e9))-(tp3.tv_nsec+(tp3.tv_sec*1e9)))/1e9);
#endif
					TQString preamble(scope_segarray);
					#ifdef ENABLE_EXTRA_TRACE_DEBUGGING
					printf("[DEBG] Preamble: %s\n", preamble.ascii());
					#endif
					TQStringList resultPairs = TQStringList::split(";", preamble, FALSE);
					// Find/initialize critical data values
					double ymult = 0;
					double yoffset = 0;
					double yposition = 0;
					char* yunits = NULL;
					double xincr = 0;
					double xposition = 0;
					char* xunits = NULL;

					if (strcmp("TDS744AOS", scopeType) == 0) {
						if (resultPairs.count() > 15) {
							ymult = resultPairs[13].toDouble();
							yoffset = resultPairs[14].toDouble()*ymult;
							yposition = resultPairs[15].toDouble()*ymult;
							yunits = strdup(resultPairs[12]);
							xincr = resultPairs[9].toDouble();
							xposition = resultPairs[10].toDouble();
							xunits = strdup(resultPairs[8]);
						}
					}
					else if (strcmp("TDS744COS", scopeType) == 0) {
						if (resultPairs.count() > 15) {
							ymult = resultPairs[13].toDouble();
							yoffset = resultPairs[14].toDouble()*ymult;
							yposition = resultPairs[15].toDouble()*ymult;
							yunits = strdup(resultPairs[12]);
							xincr = resultPairs[9].toDouble();
							xposition = resultPairs[10].toDouble();
							xunits = strdup(resultPairs[8]);
						}
					}

					#ifdef ENABLE_EXTRA_TRACE_DEBUGGING
					printf("[DEBG] Decoded preamble: xincr: %f ymult: %f\n", xincr, ymult);
					#endif

					// If the units are desired, comment out these lines...
					if (yunits) {
						free(yunits);
					}
					if (xunits) {
						free(xunits);
					}

					// Get the curve data now
					sprintf(falpha,"CURVE?");
					#ifdef ENABLE_EXTRA_DEBUGGING
					printf("[DEBG] Writing: %s\n\r", falpha);
					#endif
					if (gpib_write(gpibDevice, falpha) == 0) {
#ifdef ENABLE_PROFILING
clock_gettime(CLOCK_REALTIME, &tp5);
printf("[PROFILE] %f s\n\r", ((tp5.tv_nsec+(tp5.tv_sec*1e9))-(tp4.tv_nsec+(tp4.tv_sec*1e9)))/1e9);
#endif
						#ifdef ENABLE_EXTRA_DEBUGGING
						printf("[DEBG] Trying to read %i bytes from GPIB device...\n", (SCOPE_TRACE_MAX_POINTS-1)*2);
						#endif
						ai = gpib_read_array(gpibDevice, (SCOPE_TRACE_MAX_POINTS-1)*2, scope_segarray);
						if (ai == -1) {
							return -1;
						}
						else {
#ifdef ENABLE_PROFILING
clock_gettime(CLOCK_REALTIME, &tp6);
printf("[PROFILE] %f s\n\r", ((tp6.tv_nsec+(tp6.tv_sec*1e9))-(tp5.tv_nsec+(tp5.tv_sec*1e9)))/1e9);
#endif
							#ifdef ENABLE_EXTRA_DEBUGGING
							printf("[DEBG] Read %li bytes from GPIB device\n", ai);
							#endif
							// Interpret the results
							long pointCount = ai/2;
							double horizPos = 0.0;
							char yheaderlen[2];
							#ifdef ENABLE_EXTRA_TRACE_DEBUGGING
							printf("[DEBG] Point count: %d\n", pointCount);
							#endif
							yheaderlen[0] = scope_segarray[1];
							yheaderlen[1] = 0;
							int data_offset = atoi(yheaderlen) + 2;
							for (array_pointer=0; array_pointer<pointCount; array_pointer++) {
								TQ_INT16 tempvalue;
								tempvalue = (unsigned char)scope_segarray[(array_pointer*2)+1+data_offset];			// LSB
								tempvalue = tempvalue | ((unsigned char)scope_segarray[(array_pointer*2)+0+data_offset] << 8);	// MSB
								scope_raw_trace_data[array_pointer] = tempvalue;
								scope_raw_trace_data[array_pointer] = (scope_raw_trace_data[array_pointer] * ymult)-yoffset;
								#ifdef ENABLE_EXTRA_TRACE_DEBUGGING
								if (array_pointer < 512) {
									printf("[DEBG] Array [%d]: %04x (%f) [%02x/%02x]\n", array_pointer, tempvalue, scope_raw_trace_data[array_pointer], (unsigned char)scope_segarray[(array_pointer*2)+0+data_offset], (unsigned char)scope_segarray[(array_pointer*2)+1+data_offset]);
								}
								#endif
								scope_raw_position_data[array_pointer] = horizPos;
								horizPos = horizPos + xincr;
							}
						}
					}
					else {
						return -2;
					}
				}
			}
			else {
				return -2;
			}
		}
		else {
			return -2;
		}

#ifdef ENABLE_PROFILING
clock_gettime(CLOCK_REALTIME, &tp7);
printf("[PROFILE] %f s\n\r", ((tp7.tv_nsec+(tp7.tv_sec*1e9))-(tp6.tv_nsec+(tp6.tv_sec*1e9)))/1e9);
#endif
		return array_pointer;
	}
	else {
		return -1;
	}
}

int scope_get_number_of_horizontal_divisions(const char * scopeType, int gpibDevice) {
	if (strcmp("HP54600OS", scopeType) == 0) {
		// FIXME
		// Not supported (yet)
		return -1;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		return 8;
	}
	else {
		return -1;
	}
}

int scope_get_number_of_vertical_divisions(const char * scopeType, int gpibDevice) {
	if (strcmp("HP54600OS", scopeType) == 0) {
		// FIXME
		// Not supported (yet)
		return -1;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		return 10;
	}
	else {
		return -1;
	}
}

int scope_get_number_of_channels(const char * scopeType, int gpibDevice) {
	if (strcmp("HP54600OS", scopeType) == 0) {
		return 2;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		return 4;
	}
	else {
		return -1;
	}
}

int scope_get_channel_volts_div(double * retval, int desired_channel, const char * scopeType, int gpibDevice) {
	char floatstring[1024];
	long ai;
	int max_num_bytes = 0;

	if (strcmp("HP54600OS", scopeType) == 0) {
		// FIXME
		// Not supported (yet)
		return -1;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		// Send request
		printf("[INFO] Getting scope volts per division for channel %d\n\r", desired_channel);
		sprintf(falpha,"CH%d:SCALE?", desired_channel);
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Writing: %s\n\r", falpha);
		#endif
		if (gpib_write(gpibDevice, falpha) == 0) {
			max_num_bytes = 24;	// Request more bytes than are possible to ensure no bytes are left behind
		}
		else {
			return 2;
		}

		// Read response
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Trying to read %i bytes from GPIB device...\n", max_num_bytes);
		#endif

		ai = gpib_read_array(gpibDevice, max_num_bytes, floatstring);
		if (ai == -1) {
			return 1;
		}
		else {
			floatstring[ai]=0;
			*retval = atof(floatstring);
		}

		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Read %li bytes from GPIB device\n", ai);
		#endif

		return 0;
	}
	else {
		return -1;
	}
}

int scope_get_channel_seconds_div(double * retval, int desired_channel, const char * scopeType, int gpibDevice) {
	char floatstring[1024];
	long ai;
	int max_num_bytes = 0;

	if (strcmp("HP54600OS", scopeType) == 0) {
		// FIXME
		// Not supported (yet)
		return -1;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		double xincr;

		// Send request
		printf("[INFO] Getting scope seconds per division for channel %d\n\r", desired_channel);
		sprintf(falpha,"DATA:SOURCE CH%d", desired_channel);
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Writing: %s\n\r", falpha);
		#endif
		if (gpib_write(gpibDevice, falpha) != 0) {
			return 2;
		}

		sprintf(falpha,"DATA:STOP %d", SCOPE_TRACE_MAX_POINTS-1);
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Writing: %s\n\r", falpha);
		#endif
		if (gpib_write(gpibDevice, falpha) != 0) {
			return 2;
		}

		sprintf(falpha,"WFMPRE:CH%d:XINCR?",desired_channel);
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Writing: %s\n\r", falpha);
		#endif
		if (gpib_write(gpibDevice, falpha) == 0) {
			max_num_bytes = 24;	// Request more bytes than are possible to ensure no bytes are left behind
		}
		else {
			return 2;
		}

		// Read response
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Trying to read %i bytes from GPIB device...\n", max_num_bytes);
		#endif

		ai = gpib_read_array(gpibDevice, max_num_bytes, floatstring);
		if (ai == -1) {
			return 1;
		}
		else {
			floatstring[ai]=0;
			xincr = atof(floatstring);
		}

		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Read %li bytes from GPIB device\n", ai);
		#endif

		sprintf(falpha,"WFMPRE:CH%d:NR_P?", desired_channel);
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Writing: %s\n\r", falpha);
		#endif
		if (gpib_write(gpibDevice, falpha) == 0) {
			max_num_bytes = 24;	// Request more bytes than are possible to ensure no bytes are left behind
		}
		else {
			return 2;
		}

		// Read response
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Trying to read %i bytes from GPIB device...\n", max_num_bytes);
		#endif

		ai = gpib_read_array(gpibDevice, max_num_bytes, floatstring);
		if (ai == -1) {
			return 1;
		}
		else {
			floatstring[ai]=0;
			*retval = ((atof(floatstring)*xincr)/scope_get_number_of_vertical_divisions(scopeType, gpibDevice));
		}

		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Read %li bytes from GPIB device\n", ai);
		#endif

		return 0;
	}
	else {
		return -1;
	}
}

int scope_get_channel_sample_count(unsigned long * retval, int desired_channel, const char * scopeType, int gpibDevice) {
	char floatstring[1024];
	long ai;
	int max_num_bytes = 0;

	if (strcmp("HP54600OS", scopeType) == 0) {
		// FIXME
		// Not supported (yet)
		return -1;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		// Send request
		printf("[INFO] Getting number of samples in trace for channel %d\n\r", desired_channel);
		sprintf(falpha,"DATA:SOURCE CH%d", desired_channel);
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Writing: %s\n\r", falpha);
		#endif
		if (gpib_write(gpibDevice, falpha) != 0) {
			return 2;
		}

		sprintf(falpha,"DATA:STOP %d", SCOPE_TRACE_MAX_POINTS-1);
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Writing: %s\n\r", falpha);
		#endif
		if (gpib_write(gpibDevice, falpha) != 0) {
			return 2;
		}

		sprintf(falpha,"WFMPRE:CH%d:NR_P?", desired_channel);
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Writing: %s\n\r", falpha);
		#endif
		if (gpib_write(gpibDevice, falpha) == 0) {
			max_num_bytes = 24;	// Request more bytes than are possible to ensure no bytes are left behind
		}
		else {
			return 2;
		}

		// Read response
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Trying to read %i bytes from GPIB device...\n", max_num_bytes);
		#endif

		ai = gpib_read_array(gpibDevice, max_num_bytes, floatstring);
		if (ai == -1) {
			return 1;
		}
		else {
			floatstring[ai]=0;
			*retval = atof(floatstring);
		}

		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Read %li bytes from GPIB device\n", ai);
		#endif

		return 0;
	}
	else {
		return -1;
	}
}

int scope_get_probe_attenuation_multiplier(double * retval, int desired_channel, const char * scopeType, int gpibDevice) {
	char floatstring[1024];
	long ai;
	int max_num_bytes = 0;

	if (strcmp("HP54600OS", scopeType) == 0) {
		// FIXME
		// Not supported (yet)
		return -1;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		// Send request
		printf("[INFO] Getting trigger level for channel %d\n\r", desired_channel);
		sprintf(falpha,"CH%d:PROBE?", desired_channel);
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Writing: %s\n\r", falpha);
		#endif
		if (gpib_write(gpibDevice, falpha) == 0) {
			max_num_bytes = 24;	// Request more bytes than are possible to ensure no bytes are left behind
		}
		else {
			return 2;
		}

		// Read response
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Trying to read %i bytes from GPIB device...\n", max_num_bytes);
		#endif

		ai = gpib_read_array(gpibDevice, max_num_bytes, floatstring);
		if (ai == -1) {
			return 1;
		}
		else {
			floatstring[ai]=0;
			*retval = atof(floatstring);
		}

		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Read %li bytes from GPIB device\n", ai);
		#endif

		return 0;
	}
	else {
		return -1;
	}
}

int scope_get_permitted_volts_div_settings_at_1x(int * number_of_values, double ** retarray, const char * scopeType, int gpibDevice) {
	if (strcmp("HP54600OS", scopeType) == 0) {
		// FIXME
		// Not supported (yet)
		return -1;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		*number_of_values = 13;
		double* values = (double*)malloc(sizeof(double)*(*number_of_values));
		values[0] = 1e-3;
		values[1] = 2e-3;
		values[2] = 5e-3;
		values[3] = 1e-2;
		values[4] = 2e-2;
		values[5] = 5e-2;
		values[6] = 1e-1;
		values[7] = 2e-1;
		values[8] = 5e-1;
		values[9] = 1e0;
		values[10] = 2e0;
		values[11] = 5e0;
		values[12] = 1e1;
		*retarray = values;
		return 0;
	}
	else {
		return -1;
	}
}

int scope_get_permitted_seconds_div_settings(int * number_of_values, double ** retarray, const char * scopeType, int gpibDevice) {
	if (strcmp("HP54600OS", scopeType) == 0) {
		// FIXME
		// Not supported (yet)
		return -1;
	}
	else if ((strcmp("TDS744AOS", scopeType) == 0)
		|| (strcmp("TDS744COS", scopeType) == 0)) {
		*number_of_values = 28;
		double* values = (double*)malloc(sizeof(double)*(*number_of_values));
		values[0] = 1.25e-8;
		values[1] = 2.5e-8;
		values[2] = 5e-8;
		values[3] = 1e-7;
		values[4] = 2e-7;
		values[5] = 5e-7;
		values[6] = 1e-6;
		values[7] = 2e-6;
		values[8] = 5e-6;
		values[9] = 1e-5;
		values[10] = 2e-5;
		values[11] = 5e-5;
		values[12] = 1e-4;
		values[13] = 2e-4;
		values[14] = 5e-4;
		values[15] = 1e-3;
		values[16] = 2e-3;
		values[17] = 5e-3;
		values[18] = 1e-2;
		values[19] = 2e-2;
		values[20] = 5e-2;
		values[21] = 1e-1;
		values[22] = 2e-1;
		values[23] = 5e-1;
		values[24] = 1e0;
		values[25] = 2e0;
		values[26] = 5e0;
		values[27] = 1e1;
		*retarray = values;
		return 0;
	}
	else {
		return -1;
	}
}
