/***************************************************************************
    File:         renderer.h
    Project:      Kio-Sword -- An ioslave for SWORD and KDE
    Copyright:    Copyright (C) 2004-2005 Luke Plant
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef KS_RENDERER_H
#define KS_RENDERER_H

#include "filter.h"

#include <swmgr.h>
#include <swmodule.h>
#include <versekey.h>

#include <tqstring.h>
#include <tqstringlist.h>

#include <functional>
#include <vector>
#include <set>


/** Handles sword backend and prints modules 
 *
 * Renderer inherits from sword::SWMgr and handles the majority of
 * the sword stuff.  It also adds functions to 'print' text from
 * a specified module and to list available modules.
 *
 * All output is in HTML, and unicode characters are not converted
 * to HTML entities (this allows the calling function to decide
 * what encoding to use in the web page e.g. utf 8 could be used
 * and no transformation would need to be done).
 *
 */

namespace KioSword {
	class SwordOptions;
	class Template;
	
	class Renderer : public sword::SWMgr {
	
	public:
		Renderer();
		virtual ~Renderer();
	
		typedef enum {  SEARCH_WORDS,
				SEARCH_PHRASE,
				SEARCH_REGEX } SearchType;
				
		/** Return an HTML snippet of specified key from module
		*
		* @param module The sword module to print from
		* @param key    The text description of the key from the users URL
		* @param options Options for rendering text
		* @param tmplt Output parameter that should be filled with title and content
		*/
		void moduleQuery(const TQString &module, const TQString &ref, const SwordOptions &options, Template* tmplt);
		
		TQString search(const TQString &module, const TQString &query, SearchType stype, const SwordOptions &options);
		
		/** Return an HTML snippet containing a hyperlinked table of modules
		*/
		TQString listModules(const SwordOptions &options);
		void setOptions(const SwordOptions &options);
		TQStringList moduleList();
		TQStringList availableLocales();
		
		
	protected:
		typedef enum { BIBLE, COMMENTARY, LEXDICT, GENERIC, NUM_MODULE_TYPES } ModuleType ;
		enum KeyType { SWKEY, VERSEKEY, TREEKEY } ;
		void setModuleFilter(sword::SWModule *module, const SwordOptions* options);
	
		ModuleType getModuleType(sword::SWModule *module);
		TQString indexBible(sword::SWModule *module, const SwordOptions& options);
		TQString indexBook(sword::SWModule *module, const SwordOptions& options);
		TQString indexTree(sword::SWModule *module, const SwordOptions& options, bool fromTop, const int depth = -1);
		
		void verseQuery(sword::SWModule *module, const TQString &query, const SwordOptions &options,
				ModuleType modtype, Template* tmplt, TQString &navlinks);
		void treeQuery(sword::SWModule *module, const TQString &query, const SwordOptions &options,
				ModuleType modtype, Template* tmplt, TQString &navlinks);
		void normalQuery(sword::SWModule *module, const TQString &query, const SwordOptions &options,
				ModuleType modtype, Template* tmplt, TQString &navlinks);
		
		static TQString renderText(sword::SWModule *module);
		static TQString chapterList(const TQString &modname, const sword::VerseKey *vk, const SwordOptions& options);
		
		static TQString chapterLink(const TQString &modname, const sword::VerseKey *vk, const SwordOptions& options);
		static TQString chapterLink(const TQString &modname, const sword::SWKey *sk, const SwordOptions& options);
		
		static TQString bookLink(const TQString &modname, const sword::VerseKey *vk, const SwordOptions& options);
		static TQString bookLink(const TQString &modname, const sword::SWKey *sk, const SwordOptions& options);
		
		static TQString bookChapter(const sword::SWKey *sk);
		static TQString bookChapter(const sword::VerseKey *vk);
		
		static TQString bookName(const sword::SWKey *sk);
		static TQString bookName(const sword::VerseKey *vk);
		
		FilterBase *m_osisfilter;
		FilterBase *m_gbffilter;
		FilterBase *m_thmlfilter;
		sword::SWFilter *m_plainfilter;
		sword::SWFilter *m_rtffilter;
		
		std::set<sword::SWModule *, std::less<sword::SWModule *> > m_modset;
		std::vector<const char *> m_moduleTypes;
		std::vector<TQString> m_moduleTypeNames;
	
	};
}
#endif // KS_RENDERER_H

