/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.
 
    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>
 
    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.
 
    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.
 
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/

#ifdef HAVE_LIBLO

#include "AudioPluginOSCGUI.h"

#include "misc/Debug.h"
#include "misc/Strings.h"
#include "base/AudioPluginInstance.h"
#include "base/Exception.h"
#include "sound/PluginIdentifier.h"
#include <tdeprocess.h>
#include <tqdir.h>
#include <tqfileinfo.h>
#include <tqstring.h>
#include <algorithm>


namespace Rosegarden
{

AudioPluginOSCGUI::AudioPluginOSCGUI(AudioPluginInstance *instance,
                                     TQString serverURL, TQString friendlyName) :
        m_gui(0),
        m_address(0),
        m_basePath(""),
        m_serverUrl(serverURL)
{
    TQString identifier = strtoqstr(instance->getIdentifier());

    TQString filePath = getGUIFilePath(identifier);
    if (!filePath) {
        throw Exception("No GUI found");
    }

    TQString type, soName, label;
    PluginIdentifier::parseIdentifier(identifier, type, soName, label);
    TQFileInfo soInfo(soName);

    // arguments: osc url, dll name, label, instance tag

    m_gui = new TDEProcess();

    *m_gui << filePath
    << m_serverUrl
    << soInfo.fileName()
    << label
    << friendlyName;

    RG_DEBUG << "AudioPluginOSCGUI::AudioPluginOSCGUI: Starting process "
    << filePath << " " << m_serverUrl << " "
    << soInfo.fileName() << " " << label << " " << friendlyName << endl;

    if (!m_gui->start(TDEProcess::NotifyOnExit, TDEProcess::NoCommunication)) {
        RG_DEBUG << "AudioPluginOSCGUI::AudioPluginOSCGUI: Couldn't start process " << filePath << endl;
        delete m_gui;
        m_gui = 0;
        throw Exception("Failed to start GUI");
    }
}

AudioPluginOSCGUI::~AudioPluginOSCGUI()
{
    quit();
}

TQString
AudioPluginOSCGUI::getGUIFilePath(TQString identifier)
{
    TQString type, soName, label;
    PluginIdentifier::parseIdentifier(identifier, type, soName, label);

    RG_DEBUG << "AudioPluginOSCGUI::getGUIFilePath(" << identifier << ")" << endl;

    TQFileInfo soInfo(soName);
    if (soInfo.isRelative()) {
        //!!!
        RG_DEBUG << "AudioPluginOSCGUI::AudioPluginOSCGUI: Unable to deal with relative .so path \"" << soName << "\" in identifier \"" << identifier << "\" yet" << endl;
        throw Exception("Can't deal with relative .soname");
    }

    TQDir dir(soInfo.dir());
    TQString fileBase(soInfo.baseName(true));

    if (!dir.cd(fileBase)) {
        RG_DEBUG << "AudioPluginOSCGUI::AudioPluginOSCGUI: No GUI subdir for plugin .so " << soName << endl;
        throw Exception("No GUI subdir available");
    }

    const TQFileInfoList *list = dir.entryInfoList();

    // in order of preference:
    const char *suffixes[] = { "_rg", "_kde", "_qt", "_gtk2", "_gtk", "_x11", "_gui"
                             };
    int nsuffixes = sizeof(suffixes) / sizeof(suffixes[0]);

    for (int k = 0; k <= nsuffixes; ++k) {

        for (int fuzzy = 0; fuzzy <= 1; ++fuzzy) {

            TQFileInfoListIterator i(*list);
            TQFileInfo *info;

            while ((info = i.current()) != 0) {

                RG_DEBUG << "Looking at " << info->fileName() << " in path "
                << info->filePath() << " for suffix " << (k == nsuffixes ? "(none)" : suffixes[k]) << ", fuzzy " << fuzzy << endl;

                ++i;

                if (!(info->isFile() || info->isSymLink())
                        || !info->isExecutable()) {
                    RG_DEBUG << "(not executable)" << endl;
                    continue;
                }

                if (fuzzy) {
                    if (info->fileName().left(fileBase.length()) != fileBase)
                        continue;
                    RG_DEBUG << "(is file base)" << endl;
                } else {
                    if (info->fileName().left(label.length()) != label)
                        continue;
                    RG_DEBUG << "(is label)" << endl;
                }

                if (k == nsuffixes || info->fileName().lower().endsWith(suffixes[k])) {
                    RG_DEBUG << "(ends with suffix " << (k == nsuffixes ? "(none)" : suffixes[k]) << " or out of suffixes)" << endl;
                    return info->filePath();
                }
                RG_DEBUG << "(doesn't end with suffix " << (k == nsuffixes ? "(none)" : suffixes[k]) << ")" << endl;
            }
        }
    }

    return TQString();
}

void
AudioPluginOSCGUI::setGUIUrl(TQString url)
{
    if (m_address)
        lo_address_free(m_address);

    char *host = lo_url_get_hostname(url.ascii());
    char *port = lo_url_get_port(url.ascii());
    m_address = lo_address_new(host, port);
    free(host);
    free(port);

    m_basePath = lo_url_get_path(url.ascii());
}

void
AudioPluginOSCGUI::show()
{
    RG_DEBUG << "AudioPluginOSCGUI::show" << endl;

    if (!m_address)
        return ;
    TQString path = m_basePath + "/show";
    lo_send(m_address, path.ascii(), "");
}

void
AudioPluginOSCGUI::hide()
{
    if (!m_address)
        return ;
    TQString path = m_basePath + "/hide";
    lo_send(m_address, path.ascii(), "");
}

void
AudioPluginOSCGUI::quit()
{
    if (!m_address)
        return ;
    TQString path = m_basePath + "/quit";
    lo_send(m_address, path.ascii(), "");
}

void
AudioPluginOSCGUI::sendProgram(int bank, int program)
{
    if (!m_address)
        return ;
    TQString path = m_basePath + "/program";
    lo_send(m_address, path.ascii(), "ii", bank, program);
}

void
AudioPluginOSCGUI::sendPortValue(int port, float value)
{
    if (!m_address)
        return ;
    TQString path = m_basePath + "/control";
    lo_send(m_address, path.ascii(), "if", port, value);
}

void
AudioPluginOSCGUI::sendConfiguration(TQString key, TQString value)
{
    if (!m_address)
        return ;
    TQString path = m_basePath + "/configure";
    lo_send(m_address, path.ascii(), "ss", key.ascii(), value.ascii());
}

}

#endif
