/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.
 
    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>
 
    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.
 
    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.
 
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/


#include "AudioInstrumentParameterPanel.h"
#include <tqlayout.h>
#include <tdeapplication.h>

#include <tdelocale.h>
#include "misc/Debug.h"
#include "misc/Strings.h"
#include "base/AudioPluginInstance.h"
#include "base/Instrument.h"
#include "base/MidiProgram.h"
#include "document/RosegardenGUIDoc.h"
#include "gui/studio/AudioPluginManager.h"
#include "gui/studio/AudioPlugin.h"
#include "gui/studio/StudioControl.h"
#include "gui/widgets/AudioFaderBox.h"
#include "gui/widgets/AudioVUMeter.h"
#include "gui/widgets/Fader.h"
#include "gui/widgets/Rotary.h"
#include "gui/widgets/AudioRouteMenu.h"
#include "InstrumentParameterPanel.h"
#include "sound/MappedCommon.h"
#include "sound/MappedStudio.h"
#include <tqcolor.h>
#include <tqframe.h>
#include <tqlabel.h>
#include <tqpalette.h>
#include <tqpixmap.h>
#include <tqpushbutton.h>
#include <tqstring.h>
#include <tqtooltip.h>
#include <tqwidget.h>
#include <tqsignalmapper.h>


namespace Rosegarden
{

void
AudioInstrumentParameterPanel::slotSelectAudioLevel(float dB)
{
    if (m_selectedInstrument == 0)
        return ;

    if (m_selectedInstrument->getType() == Instrument::Audio ||
            m_selectedInstrument->getType() == Instrument::SoftSynth) {
        m_selectedInstrument->setLevel(dB);

        StudioControl::setStudioObjectProperty
        (MappedObjectId(m_selectedInstrument->getMappedId()),
         MappedAudioFader::FaderLevel,
         MappedObjectValue(dB));
    }

    emit updateAllBoxes();
    emit instrumentParametersChanged(m_selectedInstrument->getId());
}

void
AudioInstrumentParameterPanel::slotSelectAudioRecordLevel(float dB)
{
    if (m_selectedInstrument == 0)
        return ;

    //    std::cerr << "AudioInstrumentParameterPanel::slotSelectAudioRecordLevel("
    //	      << dB << ")" << std::endl;

    if (m_selectedInstrument->getType() == Instrument::Audio) {
        m_selectedInstrument->setRecordLevel(dB);

        StudioControl::setStudioObjectProperty
        (MappedObjectId(m_selectedInstrument->getMappedId()),
         MappedAudioFader::FaderRecordLevel,
         MappedObjectValue(dB));

        emit updateAllBoxes();
        emit instrumentParametersChanged(m_selectedInstrument->getId());
    }
}

void
AudioInstrumentParameterPanel::slotPluginSelected(InstrumentId instrumentId,
        int index, int plugin)
{
    if (!m_selectedInstrument ||
            instrumentId != m_selectedInstrument->getId())
        return ;

    RG_DEBUG << "AudioInstrumentParameterPanel::slotPluginSelected - "
    << "instrument = " << instrumentId
    << ", index = " << index
    << ", plugin = " << plugin << endl;

    TQColor pluginBackgroundColour = TQt::black;
    bool bypassed = false;

    TQPushButton *button = 0;
    TQString noneText;

    // updates synth gui button &c:
    m_audioFader->slotSetInstrument(&m_doc->getStudio(), m_selectedInstrument);

    if (index == (int)Instrument::SYNTH_PLUGIN_POSITION) {
        button = m_audioFader->m_synthButton;
        noneText = i18n("<no synth>");
    } else {
        button = m_audioFader->m_plugins[index];
        noneText = i18n("<no plugin>");
    }

    if (!button)
        return ;

    if (plugin == -1) {

        button->setText(noneText);
        TQToolTip::add
            (button, noneText);

    } else {

        AudioPlugin *pluginClass = m_doc->getPluginManager()->getPlugin(plugin);

        if (pluginClass) {
            button->setText(pluginClass->getLabel());

            TQToolTip::add
                (button, pluginClass->getLabel());

            pluginBackgroundColour = pluginClass->getColour();
        }
    }

    AudioPluginInstance *inst =
        m_selectedInstrument->getPlugin(index);

    if (inst)
        bypassed = inst->isBypassed();

    setButtonColour(index, bypassed, pluginBackgroundColour);

    if (index == (int)Instrument::SYNTH_PLUGIN_POSITION) {
        emit changeInstrumentLabel(instrumentId, button->text());
    }
}

void
AudioInstrumentParameterPanel::slotPluginBypassed(InstrumentId instrumentId,
        int pluginIndex, bool bp)
{
    if (!m_selectedInstrument ||
            instrumentId != m_selectedInstrument->getId())
        return ;

    AudioPluginInstance *inst =
        m_selectedInstrument->getPlugin(pluginIndex);

    TQColor backgroundColour = TQt::black; // default background colour

    if (inst && inst->isAssigned()) {
        AudioPlugin *pluginClass
        = m_doc->getPluginManager()->getPlugin(
              m_doc->getPluginManager()->
              getPositionByIdentifier(inst->getIdentifier().c_str()));

        /// Set the colour on the button
        //
        if (pluginClass)
            backgroundColour = pluginClass->getColour();
    }

    setButtonColour(pluginIndex, bp, backgroundColour);
}

void
AudioInstrumentParameterPanel::setButtonColour(
    int pluginIndex, bool bypassState, const TQColor &colour)
{
    RG_DEBUG << "AudioInstrumentParameterPanel::setButtonColour "
    << "pluginIndex = " << pluginIndex
    << ", bypassState = " << bypassState
    << ", rgb = " << TQString(colour.name()).ascii() << endl;

    TQPushButton *button = 0;

    if (pluginIndex == Instrument::SYNTH_PLUGIN_POSITION) {
        button = m_audioFader->m_synthButton;
    } else {
        button = m_audioFader->m_plugins[pluginIndex];
    }

    if (!button)
        return ;

    // Set the bypass colour on the plugin button
    if (bypassState) {
        button->
        setPaletteForegroundColor(tdeApp->palette().
                                  color(TQPalette::Active, TQColorGroup::Button));

        button->
        setPaletteBackgroundColor(tdeApp->palette().
                                  color(TQPalette::Active, TQColorGroup::ButtonText));
    } else if (colour == TQt::black) {
        button->
        setPaletteForegroundColor(tdeApp->palette().
                                  color(TQPalette::Active, TQColorGroup::ButtonText));

        button->
        setPaletteBackgroundColor(tdeApp->palette().
                                  color(TQPalette::Active, TQColorGroup::Button));
    } else {
        button->
        setPaletteForegroundColor(TQt::white);

        button->
        setPaletteBackgroundColor(colour);
    }
}

AudioInstrumentParameterPanel::AudioInstrumentParameterPanel(RosegardenGUIDoc* doc, TQWidget* parent)
        : InstrumentParameterPanel(doc, parent),
        m_audioFader(new AudioFaderBox(this))
{
    TQGridLayout *gridLayout = new TQGridLayout(this, 3, 2, 5, 5);

    // Instrument label : first row, all cols
    gridLayout->addMultiCellWidget(m_instrumentLabel, 0, 0, 0, 1, AlignCenter);

    // fader and connect it
    gridLayout->addMultiCellWidget(m_audioFader, 1, 1, 0, 1);

    gridLayout->setRowStretch(2, 1);

    connect(m_audioFader, TQ_SIGNAL(audioChannelsChanged(int)),
            this, TQ_SLOT(slotAudioChannels(int)));

    connect(m_audioFader->m_signalMapper, TQ_SIGNAL(mapped(int)),
            this, TQ_SLOT(slotSelectPlugin(int)));

    connect(m_audioFader->m_fader, TQ_SIGNAL(faderChanged(float)),
            this, TQ_SLOT(slotSelectAudioLevel(float)));

    connect(m_audioFader->m_recordFader, TQ_SIGNAL(faderChanged(float)),
            this, TQ_SLOT(slotSelectAudioRecordLevel(float)));

    connect(m_audioFader->m_pan, TQ_SIGNAL(valueChanged(float)),
            this, TQ_SLOT(slotSetPan(float)));

    connect(m_audioFader->m_audioOutput, TQ_SIGNAL(changed()),
            this, TQ_SLOT(slotAudioRoutingChanged()));

    connect(m_audioFader->m_audioInput, TQ_SIGNAL(changed()),
            this, TQ_SLOT(slotAudioRoutingChanged()));

    connect(m_audioFader->m_synthButton, TQ_SIGNAL(clicked()),
            this, TQ_SLOT(slotSynthButtonClicked()));

    connect(m_audioFader->m_synthGUIButton, TQ_SIGNAL(clicked()),
            this, TQ_SLOT(slotSynthGUIButtonClicked()));
}

void
AudioInstrumentParameterPanel::slotSynthButtonClicked()
{
    slotSelectPlugin(Instrument::SYNTH_PLUGIN_POSITION);
}

void
AudioInstrumentParameterPanel::slotSynthGUIButtonClicked()
{
    emit showPluginGUI(m_selectedInstrument->getId(),
                       Instrument::SYNTH_PLUGIN_POSITION);
}

void
AudioInstrumentParameterPanel::slotSetPan(float pan)
{
    RG_DEBUG << "AudioInstrumentParameterPanel::slotSetPan - "
    << "pan = " << pan << endl;

    StudioControl::setStudioObjectProperty
    (MappedObjectId(m_selectedInstrument->getMappedId()),
     MappedAudioFader::Pan,
     MappedObjectValue(pan));

    m_selectedInstrument->setPan(MidiByte(pan + 100.0));
    emit instrumentParametersChanged(m_selectedInstrument->getId());
}

void
AudioInstrumentParameterPanel::setAudioMeter(float dBleft, float dBright,
        float recDBleft, float recDBright)
{
    //    RG_DEBUG << "AudioInstrumentParameterPanel::setAudioMeter: (" << dBleft
    //	     << "," << dBright << ")" << endl;

    if (m_selectedInstrument) {
        // Always set stereo, because we have to reflect what's happening
        // with the pan setting even on mono tracks
        m_audioFader->m_vuMeter->setLevel(dBleft, dBright);
        m_audioFader->m_vuMeter->setRecordLevel(recDBleft, recDBright);
    }
}

void
AudioInstrumentParameterPanel::setupForInstrument(Instrument* instrument)
{
    blockSignals(true);

    m_selectedInstrument = instrument;

    m_instrumentLabel->setText(strtoqstr(instrument->getName()));

    m_audioFader->m_recordFader->setFader(instrument->getRecordLevel());
    m_audioFader->m_fader->setFader(instrument->getLevel());

    m_audioFader->slotSetInstrument(&m_doc->getStudio(), instrument);

    int start = 0;

    if (instrument->getType() == Instrument::SoftSynth)
        start = -1;

    for (int i = start; i < int(m_audioFader->m_plugins.size()); i++) {
        int index;
        TQPushButton *button;
        TQString noneText;

        if (i == -1) {
            index = Instrument::SYNTH_PLUGIN_POSITION;
            button = m_audioFader->m_synthButton;
            noneText = i18n("<no synth>");
        } else {
            index = i;
            button = m_audioFader->m_plugins[i];
            noneText = i18n("<no plugin>");
        }

        button->show();

        AudioPluginInstance *inst = instrument->getPlugin(index);

        if (inst && inst->isAssigned()) {
            AudioPlugin *pluginClass
            = m_doc->getPluginManager()->getPlugin(
                  m_doc->getPluginManager()->
                  getPositionByIdentifier(inst->getIdentifier().c_str()));

            if (pluginClass) {
                button->setText(pluginClass->getLabel());
                TQToolTip::add
                    (button, pluginClass->getLabel());
                setButtonColour(index, inst->isBypassed(),
                                pluginClass->getColour());
            }
        } else {
            button->setText(noneText);
            TQToolTip::add
                (button, noneText);
            setButtonColour(index, inst ? inst->isBypassed() : false, TQt::black);
        }
    }

    // Set the number of channels on the fader widget
    //
    m_audioFader->setAudioChannels(instrument->getAudioChannels());

    // Pan - adjusted backwards
    //
    m_audioFader->m_pan->setPosition(instrument->getPan() - 100);

    // Tell fader box whether to include e.g. audio input selection
    //
    m_audioFader->setIsSynth(instrument->getType() == Instrument::SoftSynth);

    blockSignals(false);
}

void
AudioInstrumentParameterPanel::slotAudioChannels(int channels)
{
    RG_DEBUG << "AudioInstrumentParameterPanel::slotAudioChannels - "
    << "channels = " << channels << endl;

    m_selectedInstrument->setAudioChannels(channels);

    StudioControl::setStudioObjectProperty
    (MappedObjectId(m_selectedInstrument->getMappedId()),
     MappedAudioFader::Channels,
     MappedObjectValue(channels));

    emit instrumentParametersChanged(m_selectedInstrument->getId());

}

void
AudioInstrumentParameterPanel::slotAudioRoutingChanged()
{
    if (m_selectedInstrument)
        emit instrumentParametersChanged(m_selectedInstrument->getId());
}

void
AudioInstrumentParameterPanel::slotSelectPlugin(int index)
{
    if (m_selectedInstrument) {
        emit selectPlugin(0, m_selectedInstrument->getId(), index);
    }
}

}
#include "AudioInstrumentParameterPanel.moc"
