/* This file is part of the KDE project
   Copyright (C) 2005 Ariya Hidayat <ariya@kde.org>
             (C) 2003 Norbert Andres <nandres@web.de>
             (C) 2002 Philipp Mueller <philipp.mueller@gmx.de>
             (C) 1999-2002 Laurent Montel <montel@kde.org>
             (C) 1999 Stephan Kulow <coolo@kde.org>
             (C) 1998-1999 Torben Weis <weis@kde.org>
   
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "link.h"

#include <tqcombobox.h>
#include <tqframe.h>
#include <tqlabel.h>
#include <tqlayout.h>

#include <kdesktopfile.h>
#include <kdialogbase.h>
#include <kiconloader.h>
#include <klineedit.h>
#include <tdemessagebox.h>
#include <tdelocale.h>
#include <tderecentdocument.h>
#include <kurlrequester.h>

using namespace KSpread;

class LinkDialog::Private
{
public:
    TQString text;
    TQFrame* internetPage;
    KLineEdit* internetText;
    KLineEdit* internetLink;
    TQFrame* mailPage;
    KLineEdit* mailText;
    KLineEdit* mailLink;
    TQFrame* filePage;
    KLineEdit* fileText;
    KURLRequester* fileLink;
    TQFrame* cellPage;
    KLineEdit* cellText;
    KLineEdit* cellLink;
};

LinkDialog::LinkDialog( TQWidget*, const char* )
  :  KDialogBase( KDialogBase::IconList,i18n( "Insert Link") ,
                  KDialogBase::Ok | KDialogBase::Cancel,
                  KDialogBase::Ok )
{
    d = new Private;
    
    // link for web or ftp
    d->internetPage = addPage( i18n( "Internet" ), TQString(), 
        BarIcon( "text-html",TDEIcon::SizeMedium ) );
    TQVBoxLayout* iLayout = new TQVBoxLayout( d->internetPage, marginHint(), spacingHint() );
    iLayout->add( new TQLabel( i18n("Text to display:" ), d->internetPage ) );
    d->internetText = new KLineEdit( d->internetPage );
    iLayout->add( d->internetText );
    iLayout->add( new TQLabel( i18n("Internet address:" ), d->internetPage ) );
    d->internetLink = new KLineEdit( d->internetPage );
    iLayout->add( d->internetLink );
    iLayout->addItem( new TQSpacerItem( 0, 0, TQSizePolicy::Minimum, TQSizePolicy::Expanding ) );
    connect( d->internetText, TQ_SIGNAL( textChanged( const TQString& ) ), this,
        TQ_SLOT( setText( const TQString& ) ) );
    
    // link for e-mail
    d->mailPage = addPage( i18n( "Mail" ), TQString(), 
        BarIcon( "mail_generic",TDEIcon::SizeMedium ) );
    TQVBoxLayout* mLayout = new TQVBoxLayout( d->mailPage, marginHint(), spacingHint() );
    mLayout->add( new TQLabel( i18n("Text to display:" ), d->mailPage ) );
    d->mailText = new KLineEdit( d->mailPage );
    mLayout->add( d->mailText );
    mLayout->add( new TQLabel( i18n("Email:" ), d->mailPage ) );
    d->mailLink = new KLineEdit( d->mailPage );
    mLayout->add( d->mailLink );
    mLayout->addItem( new TQSpacerItem( 0, 0, TQSizePolicy::Minimum, TQSizePolicy::Expanding ) );
    connect( d->mailText, TQ_SIGNAL( textChanged( const TQString& ) ), this,
        TQ_SLOT( setText( const TQString& ) ) );
    
    // link for external file
    d->filePage = addPage( i18n( "File" ), TQString(), 
        BarIcon( "document-new",TDEIcon::SizeMedium ) );
    TQVBoxLayout* fLayout = new TQVBoxLayout( d->filePage, marginHint(), spacingHint() );
    fLayout->add( new TQLabel( i18n("Text to display:" ), d->filePage ) );
    d->fileText = new KLineEdit( d->filePage );
    fLayout->add( d->fileText );
    fLayout->add( new TQLabel( i18n("File location:" ), d->filePage ) );
    d->fileLink = new KURLRequester( d->filePage );
    fLayout->add( d->fileLink );
    fLayout->add( new TQLabel( i18n("Recent file:" ), d->filePage ) );
    TQComboBox* recentFile = new TQComboBox( d->filePage );
    recentFile->setSizePolicy( TQSizePolicy::Expanding, TQSizePolicy::Fixed );
    fLayout->add( recentFile );
    fLayout->addItem( new TQSpacerItem( 0, 40, TQSizePolicy::Minimum, TQSizePolicy::MinimumExpanding ) );
    connect( d->fileText, TQ_SIGNAL( textChanged( const TQString& ) ), this,
        TQ_SLOT( setText( const TQString& ) ) );
    TQObject::connect( recentFile, TQ_SIGNAL( highlighted ( const TQString &) ),
        d->fileLink->lineEdit(), TQ_SLOT( setText( const TQString & ) ) );
    
    
    // populate recent files
    TQStringList fileList = TDERecentDocument::recentDocuments();
    for( TQStringList::ConstIterator it = fileList.begin();it != fileList.end(); ++it )
    {
        KDesktopFile f(*it, true /* read only */);
        if ( !f.readURL().isEmpty() )
            recentFile->insertItem( f.readURL() );
    }
    if( recentFile->count()==0 )
    {
        recentFile->insertItem( i18n("No Entries") );
        recentFile->setEnabled( false );
    }    
    
    // link to another cell
    d->cellPage =  addPage( i18n( "Cell" ), TQString(), 
        BarIcon( "misc",TDEIcon::SizeMedium ) );
    TQVBoxLayout* cLayout = new TQVBoxLayout( d->cellPage, marginHint(), spacingHint() );
    cLayout->add( new TQLabel( i18n("Text to display:" ), d->cellPage ) );
    d->cellText = new KLineEdit( d->cellPage );
    cLayout->add( d->cellText );
    cLayout->add( new TQLabel( i18n("Cell:" ), d->cellPage ) );
    d->cellLink = new KLineEdit( d->cellPage );
    cLayout->add( d->cellLink );
    cLayout->addItem( new TQSpacerItem( 0, 0, TQSizePolicy::Minimum, TQSizePolicy::Expanding ) );
    connect( d->cellText, TQ_SIGNAL( textChanged( const TQString& ) ), this,
        TQ_SLOT( setText( const TQString& ) ) );
    
    enableButtonSeparator( true );
    d->internetText->setFocus();
    resize( 400,300 );
}

LinkDialog::~LinkDialog()
{
    delete d;
}

TQString LinkDialog::text() const
{
    return d->text;
}

TQString LinkDialog::link() const
{
    TQString str;
    switch( activePageIndex() )
    {
        case 0:  
          str = d->internetLink->text();  
          if( !str.isEmpty() )
          if( str.find( "http://" )==-1 )
          if( str.find( "https://" )==-1 )
          if( str.find( "ftp://" )==-1 )
              str.prepend( "http://" );
          break;
          
        case 1:  
           str = d->mailLink->text(); 
           if( !str.isEmpty() )
           if( str.find( "mailto:" )==-1 )
               str.prepend( "mailto:" );
           break;
           
        case 2:  
           str = d->fileLink->lineEdit()->text(); 
           if( !str.isEmpty() )
           if( str.find( "file:/" )==-1 )
               str.prepend( "file://" );
           break;
           
        case 3:  
            str = d->cellLink->text(); 
            break;
            
        break;
    }
    return str;
}

void LinkDialog::setText( const TQString& text )
{
    d->text = text;
    
    d->internetText->blockSignals( true );
    d->internetText->setText( text );
    d->internetText->blockSignals( false );
    
    d->mailText->blockSignals( true );
    d->mailText->setText( text );
    d->mailText->blockSignals( false );

    d->fileText->blockSignals( true );
    d->fileText->setText( text );
    d->fileText->blockSignals( false );
    
    d->cellText->blockSignals( true );
    d->cellText->setText( text );
    d->cellText->blockSignals( false );
}

// link must be complete, e.g. "http://www.koffice.org" instead of 
// "www.koffice.org" only, since protocol is used to decide which page to show
void LinkDialog::setLink( const TQString& link )
{
    if( link.startsWith( "https://" ) )
    {
      d->internetLink->setText( link.mid( TQString("https://").length() ) );
      showPage( 0 );
      return;
    }
    
    if( link.startsWith( "http://" ) )
    {
      d->internetLink->setText( link.mid( TQString("http://").length() ) );
      showPage( 0 );
      return;
    }
    
    if( link.startsWith( "ftp://" ) )
    {
      d->internetLink->setText( link.mid( TQString("ftp://").length() ) );
      showPage( 0 );
      return;
    }
    
    if( link.startsWith( "mailto:" ) )
    {
      d->mailLink->setText( link.mid( TQString("mailto:").length() ) );
      showPage( 1 );
      return;
    }
    
    if( link.startsWith( "file:/" ) )
    {
      TQString s = link.mid( TQString("file:/").length() );
      while(s.startsWith("//")) s.remove(0,1);
      d->fileLink->lineEdit()->setText(s);
      showPage( 2 );
      return;
    }
    
    // assume cell reference
    d->cellLink->setText( link );
    showPage( 3 );
}

void LinkDialog::slotOk()
{
    TQString str;
    switch( activePageIndex() )
    {
        case 0:  str = i18n( "Internet address is empty" );  break;
        case 1:  str = i18n( "Mail address is empty" ); break;
        case 2:  str = i18n( "File name is empty" ); break;
        case 3:  str = i18n( "Destination cell is empty" ); break;
        break;
    }
    
    if( link().isEmpty() )
    {
       KMessageBox::error( this, str );
       return;
    } 
    
    if( d->text.isEmpty() )
        d->text = link();
    
    accept();
}

#include "link.moc"
