package ztls

import (
	"github.com/projectdiscovery/tlsx/pkg/tlsx/clients"
	errorutil "github.com/projectdiscovery/utils/errors"
	"github.com/zmap/zcrypto/tls"
	"github.com/zmap/zcrypto/x509"
)

var (
	AllCiphers           []uint16
	AllCiphersNames      []string
	SupportedTlsVersions []string
)

func init() {
	for name, cipher := range ztlsCiphers {
		AllCiphersNames = append(AllCiphersNames, name)
		AllCiphers = append(AllCiphers, cipher)
	}
	for name := range versionStringToTLSVersion {
		SupportedTlsVersions = append(SupportedTlsVersions, name)
	}
}

func toZTLSCiphers(items []string) ([]uint16, error) {
	var convertedCiphers []uint16
	for _, item := range items {
		zcipher, ok := ztlsCiphers[item]
		if !ok {
			return nil, errorutil.NewWithTag("ztls", "cipher suite %v not supported", item)
		}
		convertedCiphers = append(convertedCiphers, zcipher)
	}
	return convertedCiphers, nil
}

// ConvertCertificateToResponse using zcrypto x509
func ConvertCertificateToResponse(options *clients.Options, hostname string, cert *x509.Certificate) *clients.CertificateResponse {
	if cert == nil {
		return nil
	}
	response := &clients.CertificateResponse{
		SubjectAN:    cert.DNSNames,
		Emails:       cert.EmailAddresses,
		NotBefore:    cert.NotBefore,
		NotAfter:     cert.NotAfter,
		Expired:      clients.IsExpired(cert.NotAfter),
		SelfSigned:   clients.IsSelfSigned(cert.AuthorityKeyId, cert.SubjectKeyId, cert.DNSNames),
		MisMatched:   clients.IsMisMatchedCert(hostname, append(cert.DNSNames, cert.Subject.CommonName)),
		Revoked:      clients.IsZTLSRevoked(options, cert),
		WildCardCert: clients.IsWildCardCert(append(cert.DNSNames, cert.Subject.CommonName)),
		IssuerDN:     cert.Issuer.String(),
		IssuerCN:     cert.Issuer.CommonName,
		IssuerOrg:    cert.Issuer.Organization,
		SubjectDN:    cert.Subject.String(),
		SubjectCN:    cert.Subject.CommonName,
		SubjectOrg:   cert.Subject.Organization,
		FingerprintHash: clients.CertificateResponseFingerprintHash{
			MD5:    clients.MD5Fingerprint(cert.Raw),
			SHA1:   clients.SHA1Fingerprint(cert.Raw),
			SHA256: clients.SHA256Fingerprint(cert.Raw),
		},
		Serial: clients.FormatToSerialNumber(cert.SerialNumber),
	}
	if options.DisplayDns {
		response.Domains = clients.GetUniqueDomainsFromCert(response)
	}
	if options.Cert {
		response.Certificate = clients.PemEncode(cert.Raw)
	}
	return response
}

// ParseSimpleTLSCertificate using zcrypto x509
func ParseSimpleTLSCertificate(cert tls.SimpleCertificate) *x509.Certificate {
	parsed, _ := x509.ParseCertificate(cert.Raw)
	return parsed
}

var ztlsCiphers = map[string]uint16{
	"TLS_NULL_WITH_NULL_NULL":                           tls.TLS_NULL_WITH_NULL_NULL,
	"TLS_RSA_WITH_NULL_MD5":                             tls.TLS_RSA_WITH_NULL_MD5,
	"TLS_RSA_WITH_NULL_SHA":                             tls.TLS_RSA_WITH_NULL_SHA,
	"TLS_RSA_EXPORT_WITH_RC4_40_MD5":                    tls.TLS_RSA_EXPORT_WITH_RC4_40_MD5,
	"TLS_RSA_WITH_RC4_128_MD5":                          tls.TLS_RSA_WITH_RC4_128_MD5,
	"TLS_RSA_WITH_RC4_128_SHA":                          tls.TLS_RSA_WITH_RC4_128_SHA,
	"TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5":                tls.TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5,
	"TLS_RSA_WITH_IDEA_CBC_SHA":                         tls.TLS_RSA_WITH_IDEA_CBC_SHA,
	"TLS_RSA_EXPORT_WITH_DES40_CBC_SHA":                 tls.TLS_RSA_EXPORT_WITH_DES40_CBC_SHA,
	"TLS_RSA_WITH_DES_CBC_SHA":                          tls.TLS_RSA_WITH_DES_CBC_SHA,
	"TLS_RSA_WITH_3DES_EDE_CBC_SHA":                     tls.TLS_RSA_WITH_3DES_EDE_CBC_SHA,
	"TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA":              tls.TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA,
	"TLS_DH_DSS_WITH_DES_CBC_SHA":                       tls.TLS_DH_DSS_WITH_DES_CBC_SHA,
	"TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA":                  tls.TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA,
	"TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA":              tls.TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA,
	"TLS_DH_RSA_WITH_DES_CBC_SHA":                       tls.TLS_DH_RSA_WITH_DES_CBC_SHA,
	"TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA":                  tls.TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA,
	"TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA":             tls.TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA,
	"TLS_DHE_DSS_WITH_DES_CBC_SHA":                      tls.TLS_DHE_DSS_WITH_DES_CBC_SHA,
	"TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA":                 tls.TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA,
	"TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA":             tls.TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA,
	"TLS_DHE_RSA_WITH_DES_CBC_SHA":                      tls.TLS_DHE_RSA_WITH_DES_CBC_SHA,
	"TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA":                 tls.TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA,
	"TLS_DH_ANON_EXPORT_WITH_RC4_40_MD5":                tls.TLS_DH_ANON_EXPORT_WITH_RC4_40_MD5,
	"TLS_DH_ANON_WITH_RC4_128_MD5":                      tls.TLS_DH_ANON_WITH_RC4_128_MD5,
	"TLS_DH_ANON_EXPORT_WITH_DES40_CBC_SHA":             tls.TLS_DH_ANON_EXPORT_WITH_DES40_CBC_SHA,
	"TLS_DH_ANON_WITH_DES_CBC_SHA":                      tls.TLS_DH_ANON_WITH_DES_CBC_SHA,
	"TLS_DH_ANON_WITH_3DES_EDE_CBC_SHA":                 tls.TLS_DH_ANON_WITH_3DES_EDE_CBC_SHA,
	"SSL_FORTEZZA_KEA_WITH_NULL_SHA":                    tls.SSL_FORTEZZA_KEA_WITH_NULL_SHA,
	"SSL_FORTEZZA_KEA_WITH_FORTEZZA_CBC_SHA":            tls.SSL_FORTEZZA_KEA_WITH_FORTEZZA_CBC_SHA,
	"TLS_KRB5_WITH_DES_CBC_SHA":                         tls.TLS_KRB5_WITH_DES_CBC_SHA,
	"TLS_KRB5_WITH_3DES_EDE_CBC_SHA":                    tls.TLS_KRB5_WITH_3DES_EDE_CBC_SHA,
	"TLS_KRB5_WITH_RC4_128_SHA":                         tls.TLS_KRB5_WITH_RC4_128_SHA,
	"TLS_KRB5_WITH_IDEA_CBC_SHA":                        tls.TLS_KRB5_WITH_IDEA_CBC_SHA,
	"TLS_KRB5_WITH_DES_CBC_MD5":                         tls.TLS_KRB5_WITH_DES_CBC_MD5,
	"TLS_KRB5_WITH_3DES_EDE_CBC_MD5":                    tls.TLS_KRB5_WITH_3DES_EDE_CBC_MD5,
	"TLS_KRB5_WITH_RC4_128_MD5":                         tls.TLS_KRB5_WITH_RC4_128_MD5,
	"TLS_KRB5_WITH_IDEA_CBC_MD5":                        tls.TLS_KRB5_WITH_IDEA_CBC_MD5,
	"TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA":               tls.TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA,
	"TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA":               tls.TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA,
	"TLS_KRB5_EXPORT_WITH_RC4_40_SHA":                   tls.TLS_KRB5_EXPORT_WITH_RC4_40_SHA,
	"TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5":               tls.TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5,
	"TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5":               tls.TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5,
	"TLS_KRB5_EXPORT_WITH_RC4_40_MD5":                   tls.TLS_KRB5_EXPORT_WITH_RC4_40_MD5,
	"TLS_PSK_WITH_NULL_SHA":                             tls.TLS_PSK_WITH_NULL_SHA,
	"TLS_DHE_PSK_WITH_NULL_SHA":                         tls.TLS_DHE_PSK_WITH_NULL_SHA,
	"TLS_RSA_PSK_WITH_NULL_SHA":                         tls.TLS_RSA_PSK_WITH_NULL_SHA,
	"TLS_RSA_WITH_AES_128_CBC_SHA":                      tls.TLS_RSA_WITH_AES_128_CBC_SHA,
	"TLS_DH_DSS_WITH_AES_128_CBC_SHA":                   tls.TLS_DH_DSS_WITH_AES_128_CBC_SHA,
	"TLS_DH_RSA_WITH_AES_128_CBC_SHA":                   tls.TLS_DH_RSA_WITH_AES_128_CBC_SHA,
	"TLS_DHE_DSS_WITH_AES_128_CBC_SHA":                  tls.TLS_DHE_DSS_WITH_AES_128_CBC_SHA,
	"TLS_DHE_RSA_WITH_AES_128_CBC_SHA":                  tls.TLS_DHE_RSA_WITH_AES_128_CBC_SHA,
	"TLS_DH_ANON_WITH_AES_128_CBC_SHA":                  tls.TLS_DH_ANON_WITH_AES_128_CBC_SHA,
	"TLS_RSA_WITH_AES_256_CBC_SHA":                      tls.TLS_RSA_WITH_AES_256_CBC_SHA,
	"TLS_DH_DSS_WITH_AES_256_CBC_SHA":                   tls.TLS_DH_DSS_WITH_AES_256_CBC_SHA,
	"TLS_DH_RSA_WITH_AES_256_CBC_SHA":                   tls.TLS_DH_RSA_WITH_AES_256_CBC_SHA,
	"TLS_DHE_DSS_WITH_AES_256_CBC_SHA":                  tls.TLS_DHE_DSS_WITH_AES_256_CBC_SHA,
	"TLS_DHE_RSA_WITH_AES_256_CBC_SHA":                  tls.TLS_DHE_RSA_WITH_AES_256_CBC_SHA,
	"TLS_DH_ANON_WITH_AES_256_CBC_SHA":                  tls.TLS_DH_ANON_WITH_AES_256_CBC_SHA,
	"TLS_RSA_WITH_NULL_SHA256":                          tls.TLS_RSA_WITH_NULL_SHA256,
	"TLS_RSA_WITH_AES_128_CBC_SHA256":                   tls.TLS_RSA_WITH_AES_128_CBC_SHA256,
	"TLS_RSA_WITH_AES_256_CBC_SHA256":                   tls.TLS_RSA_WITH_AES_256_CBC_SHA256,
	"TLS_DH_DSS_WITH_AES_128_CBC_SHA256":                tls.TLS_DH_DSS_WITH_AES_128_CBC_SHA256,
	"TLS_DH_RSA_WITH_AES_128_CBC_SHA256":                tls.TLS_DH_RSA_WITH_AES_128_CBC_SHA256,
	"TLS_DHE_DSS_WITH_AES_128_CBC_SHA256":               tls.TLS_DHE_DSS_WITH_AES_128_CBC_SHA256,
	"TLS_RSA_WITH_CAMELLIA_128_CBC_SHA":                 tls.TLS_RSA_WITH_CAMELLIA_128_CBC_SHA,
	"TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA":              tls.TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA,
	"TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA":              tls.TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA,
	"TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA":             tls.TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA,
	"TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA":             tls.TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA,
	"TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA":             tls.TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA,
	"TLS_RSA_EXPORT1024_WITH_RC4_56_MD5":                tls.TLS_RSA_EXPORT1024_WITH_RC4_56_MD5,
	"TLS_RSA_EXPORT1024_WITH_RC2_CBC_56_MD5":            tls.TLS_RSA_EXPORT1024_WITH_RC2_CBC_56_MD5,
	"TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA":               tls.TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA,
	"TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA":           tls.TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA,
	"TLS_RSA_EXPORT1024_WITH_RC4_56_SHA":                tls.TLS_RSA_EXPORT1024_WITH_RC4_56_SHA,
	"TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA":            tls.TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA,
	"TLS_DHE_DSS_WITH_RC4_128_SHA":                      tls.TLS_DHE_DSS_WITH_RC4_128_SHA,
	"TLS_DHE_RSA_WITH_AES_128_CBC_SHA256":               tls.TLS_DHE_RSA_WITH_AES_128_CBC_SHA256,
	"TLS_DH_DSS_WITH_AES_256_CBC_SHA256":                tls.TLS_DH_DSS_WITH_AES_256_CBC_SHA256,
	"TLS_DH_RSA_WITH_AES_256_CBC_SHA256":                tls.TLS_DH_RSA_WITH_AES_256_CBC_SHA256,
	"TLS_DHE_DSS_WITH_AES_256_CBC_SHA256":               tls.TLS_DHE_DSS_WITH_AES_256_CBC_SHA256,
	"TLS_DHE_RSA_WITH_AES_256_CBC_SHA256":               tls.TLS_DHE_RSA_WITH_AES_256_CBC_SHA256,
	"TLS_DH_ANON_WITH_AES_128_CBC_SHA256":               tls.TLS_DH_ANON_WITH_AES_128_CBC_SHA256,
	"TLS_DH_ANON_WITH_AES_256_CBC_SHA256":               tls.TLS_DH_ANON_WITH_AES_256_CBC_SHA256,
	"TLS_GOSTR341094_WITH_28147_CNT_IMIT":               tls.TLS_GOSTR341094_WITH_28147_CNT_IMIT,
	"TLS_GOSTR341001_WITH_28147_CNT_IMIT":               tls.TLS_GOSTR341001_WITH_28147_CNT_IMIT,
	"TLS_GOSTR341094_WITH_NULL_GOSTR3411":               tls.TLS_GOSTR341094_WITH_NULL_GOSTR3411,
	"TLS_GOSTR341001_WITH_NULL_GOSTR3411":               tls.TLS_GOSTR341001_WITH_NULL_GOSTR3411,
	"TLS_RSA_WITH_CAMELLIA_256_CBC_SHA":                 tls.TLS_RSA_WITH_CAMELLIA_256_CBC_SHA,
	"TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA":              tls.TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA,
	"TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA":              tls.TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA,
	"TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA":             tls.TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA,
	"TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA":             tls.TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA,
	"TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA":             tls.TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA,
	"TLS_PSK_WITH_RC4_128_SHA":                          tls.TLS_PSK_WITH_RC4_128_SHA,
	"TLS_PSK_WITH_3DES_EDE_CBC_SHA":                     tls.TLS_PSK_WITH_3DES_EDE_CBC_SHA,
	"TLS_PSK_WITH_AES_128_CBC_SHA":                      tls.TLS_PSK_WITH_AES_128_CBC_SHA,
	"TLS_PSK_WITH_AES_256_CBC_SHA":                      tls.TLS_PSK_WITH_AES_256_CBC_SHA,
	"TLS_DHE_PSK_WITH_RC4_128_SHA":                      tls.TLS_DHE_PSK_WITH_RC4_128_SHA,
	"TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA":                 tls.TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA,
	"TLS_DHE_PSK_WITH_AES_128_CBC_SHA":                  tls.TLS_DHE_PSK_WITH_AES_128_CBC_SHA,
	"TLS_DHE_PSK_WITH_AES_256_CBC_SHA":                  tls.TLS_DHE_PSK_WITH_AES_256_CBC_SHA,
	"TLS_RSA_PSK_WITH_RC4_128_SHA":                      tls.TLS_RSA_PSK_WITH_RC4_128_SHA,
	"TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA":                 tls.TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA,
	"TLS_RSA_PSK_WITH_AES_128_CBC_SHA":                  tls.TLS_RSA_PSK_WITH_AES_128_CBC_SHA,
	"TLS_RSA_PSK_WITH_AES_256_CBC_SHA":                  tls.TLS_RSA_PSK_WITH_AES_256_CBC_SHA,
	"TLS_RSA_WITH_SEED_CBC_SHA":                         tls.TLS_RSA_WITH_SEED_CBC_SHA,
	"TLS_DH_DSS_WITH_SEED_CBC_SHA":                      tls.TLS_DH_DSS_WITH_SEED_CBC_SHA,
	"TLS_DH_RSA_WITH_SEED_CBC_SHA":                      tls.TLS_DH_RSA_WITH_SEED_CBC_SHA,
	"TLS_DHE_DSS_WITH_SEED_CBC_SHA":                     tls.TLS_DHE_DSS_WITH_SEED_CBC_SHA,
	"TLS_DHE_RSA_WITH_SEED_CBC_SHA":                     tls.TLS_DHE_RSA_WITH_SEED_CBC_SHA,
	"TLS_DH_ANON_WITH_SEED_CBC_SHA":                     tls.TLS_DH_ANON_WITH_SEED_CBC_SHA,
	"TLS_RSA_WITH_AES_128_GCM_SHA256":                   tls.TLS_RSA_WITH_AES_128_GCM_SHA256,
	"TLS_RSA_WITH_AES_256_GCM_SHA384":                   tls.TLS_RSA_WITH_AES_256_GCM_SHA384,
	"TLS_DHE_RSA_WITH_AES_128_GCM_SHA256":               tls.TLS_DHE_RSA_WITH_AES_128_GCM_SHA256,
	"TLS_DHE_RSA_WITH_AES_256_GCM_SHA384":               tls.TLS_DHE_RSA_WITH_AES_256_GCM_SHA384,
	"TLS_DH_RSA_WITH_AES_128_GCM_SHA256":                tls.TLS_DH_RSA_WITH_AES_128_GCM_SHA256,
	"TLS_DH_RSA_WITH_AES_256_GCM_SHA384":                tls.TLS_DH_RSA_WITH_AES_256_GCM_SHA384,
	"TLS_DHE_DSS_WITH_AES_128_GCM_SHA256":               tls.TLS_DHE_DSS_WITH_AES_128_GCM_SHA256,
	"TLS_DHE_DSS_WITH_AES_256_GCM_SHA384":               tls.TLS_DHE_DSS_WITH_AES_256_GCM_SHA384,
	"TLS_DH_DSS_WITH_AES_128_GCM_SHA256":                tls.TLS_DH_DSS_WITH_AES_128_GCM_SHA256,
	"TLS_DH_DSS_WITH_AES_256_GCM_SHA384":                tls.TLS_DH_DSS_WITH_AES_256_GCM_SHA384,
	"TLS_DH_ANON_WITH_AES_128_GCM_SHA256":               tls.TLS_DH_ANON_WITH_AES_128_GCM_SHA256,
	"TLS_DH_ANON_WITH_AES_256_GCM_SHA384":               tls.TLS_DH_ANON_WITH_AES_256_GCM_SHA384,
	"TLS_PSK_WITH_AES_128_GCM_SHA256":                   tls.TLS_PSK_WITH_AES_128_GCM_SHA256,
	"TLS_PSK_WITH_AES_256_GCM_SHA384":                   tls.TLS_PSK_WITH_AES_256_GCM_SHA384,
	"TLS_DHE_PSK_WITH_AES_128_GCM_SHA256":               tls.TLS_DHE_PSK_WITH_AES_128_GCM_SHA256,
	"TLS_DHE_PSK_WITH_AES_256_GCM_SHA384":               tls.TLS_DHE_PSK_WITH_AES_256_GCM_SHA384,
	"TLS_RSA_PSK_WITH_AES_128_GCM_SHA256":               tls.TLS_RSA_PSK_WITH_AES_128_GCM_SHA256,
	"TLS_RSA_PSK_WITH_AES_256_GCM_SHA384":               tls.TLS_RSA_PSK_WITH_AES_256_GCM_SHA384,
	"TLS_PSK_WITH_AES_128_CBC_SHA256":                   tls.TLS_PSK_WITH_AES_128_CBC_SHA256,
	"TLS_PSK_WITH_AES_256_CBC_SHA384":                   tls.TLS_PSK_WITH_AES_256_CBC_SHA384,
	"TLS_PSK_WITH_NULL_SHA256":                          tls.TLS_PSK_WITH_NULL_SHA256,
	"TLS_PSK_WITH_NULL_SHA384":                          tls.TLS_PSK_WITH_NULL_SHA384,
	"TLS_DHE_PSK_WITH_AES_128_CBC_SHA256":               tls.TLS_DHE_PSK_WITH_AES_128_CBC_SHA256,
	"TLS_DHE_PSK_WITH_AES_256_CBC_SHA384":               tls.TLS_DHE_PSK_WITH_AES_256_CBC_SHA384,
	"TLS_DHE_PSK_WITH_NULL_SHA256":                      tls.TLS_DHE_PSK_WITH_NULL_SHA256,
	"TLS_DHE_PSK_WITH_NULL_SHA384":                      tls.TLS_DHE_PSK_WITH_NULL_SHA384,
	"TLS_RSA_PSK_WITH_AES_128_CBC_SHA256":               tls.TLS_RSA_PSK_WITH_AES_128_CBC_SHA256,
	"TLS_RSA_PSK_WITH_AES_256_CBC_SHA384":               tls.TLS_RSA_PSK_WITH_AES_256_CBC_SHA384,
	"TLS_RSA_PSK_WITH_NULL_SHA256":                      tls.TLS_RSA_PSK_WITH_NULL_SHA256,
	"TLS_RSA_PSK_WITH_NULL_SHA384":                      tls.TLS_RSA_PSK_WITH_NULL_SHA384,
	"TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256":              tls.TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256,
	"TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256":           tls.TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256,
	"TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256":           tls.TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256,
	"TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256":          tls.TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256,
	"TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256":          tls.TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256,
	"TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA256":          tls.TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA256,
	"TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256":              tls.TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256,
	"TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256":           tls.TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256,
	"TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256":           tls.TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256,
	"TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256":          tls.TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256,
	"TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256":          tls.TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256,
	"TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA256":          tls.TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA256,
	"TLS_RENEGO_PROTECTION_REQUEST":                     tls.TLS_RENEGO_PROTECTION_REQUEST,
	"TLS_FALLBACK_SCSV":                                 tls.TLS_FALLBACK_SCSV,
	"TLS_ECDH_ECDSA_WITH_NULL_SHA":                      tls.TLS_ECDH_ECDSA_WITH_NULL_SHA,
	"TLS_ECDH_ECDSA_WITH_RC4_128_SHA":                   tls.TLS_ECDH_ECDSA_WITH_RC4_128_SHA,
	"TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA":              tls.TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA,
	"TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA":               tls.TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA,
	"TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA":               tls.TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA,
	"TLS_ECDHE_ECDSA_WITH_NULL_SHA":                     tls.TLS_ECDHE_ECDSA_WITH_NULL_SHA,
	"TLS_ECDHE_ECDSA_WITH_RC4_128_SHA":                  tls.TLS_ECDHE_ECDSA_WITH_RC4_128_SHA,
	"TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA":             tls.TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA,
	"TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA":              tls.TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA,
	"TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA":              tls.TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA,
	"TLS_ECDH_RSA_WITH_NULL_SHA":                        tls.TLS_ECDH_RSA_WITH_NULL_SHA,
	"TLS_ECDH_RSA_WITH_RC4_128_SHA":                     tls.TLS_ECDH_RSA_WITH_RC4_128_SHA,
	"TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA":                tls.TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA,
	"TLS_ECDH_RSA_WITH_AES_128_CBC_SHA":                 tls.TLS_ECDH_RSA_WITH_AES_128_CBC_SHA,
	"TLS_ECDH_RSA_WITH_AES_256_CBC_SHA":                 tls.TLS_ECDH_RSA_WITH_AES_256_CBC_SHA,
	"TLS_ECDHE_RSA_WITH_NULL_SHA":                       tls.TLS_ECDHE_RSA_WITH_NULL_SHA,
	"TLS_ECDHE_RSA_WITH_RC4_128_SHA":                    tls.TLS_ECDHE_RSA_WITH_RC4_128_SHA,
	"TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA":               tls.TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA,
	"TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA":                tls.TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA,
	"TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA":                tls.TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA,
	"TLS_ECDH_ANON_WITH_NULL_SHA":                       tls.TLS_ECDH_ANON_WITH_NULL_SHA,
	"TLS_ECDH_ANON_WITH_RC4_128_SHA":                    tls.TLS_ECDH_ANON_WITH_RC4_128_SHA,
	"TLS_ECDH_ANON_WITH_3DES_EDE_CBC_SHA":               tls.TLS_ECDH_ANON_WITH_3DES_EDE_CBC_SHA,
	"TLS_ECDH_ANON_WITH_AES_128_CBC_SHA":                tls.TLS_ECDH_ANON_WITH_AES_128_CBC_SHA,
	"TLS_ECDH_ANON_WITH_AES_256_CBC_SHA":                tls.TLS_ECDH_ANON_WITH_AES_256_CBC_SHA,
	"TLS_SRP_SHA_WITH_3DES_EDE_CBC_SHA":                 tls.TLS_SRP_SHA_WITH_3DES_EDE_CBC_SHA,
	"TLS_SRP_SHA_RSA_WITH_3DES_EDE_CBC_SHA":             tls.TLS_SRP_SHA_RSA_WITH_3DES_EDE_CBC_SHA,
	"TLS_SRP_SHA_DSS_WITH_3DES_EDE_CBC_SHA":             tls.TLS_SRP_SHA_DSS_WITH_3DES_EDE_CBC_SHA,
	"TLS_SRP_SHA_WITH_AES_128_CBC_SHA":                  tls.TLS_SRP_SHA_WITH_AES_128_CBC_SHA,
	"TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA":              tls.TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA,
	"TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA":              tls.TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA,
	"TLS_SRP_SHA_WITH_AES_256_CBC_SHA":                  tls.TLS_SRP_SHA_WITH_AES_256_CBC_SHA,
	"TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA":              tls.TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA,
	"TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA":              tls.TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA,
	"TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256":           tls.TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256,
	"TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384":           tls.TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384,
	"TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256":            tls.TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256,
	"TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384":            tls.TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384,
	"TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256":             tls.TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256,
	"TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384":             tls.TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384,
	"TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256":              tls.TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256,
	"TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384":              tls.TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384,
	"TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256":           tls.TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256,
	"TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384":           tls.TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384,
	"TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256":            tls.TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256,
	"TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384":            tls.TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384,
	"TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256":             tls.TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256,
	"TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384":             tls.TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384,
	"TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256":              tls.TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256,
	"TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384":              tls.TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384,
	"TLS_ECDHE_PSK_WITH_RC4_128_SHA":                    tls.TLS_ECDHE_PSK_WITH_RC4_128_SHA,
	"TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA":               tls.TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA,
	"TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA":                tls.TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA,
	"TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA":                tls.TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA,
	"TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256":             tls.TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256,
	"TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384":             tls.TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384,
	"TLS_ECDHE_PSK_WITH_NULL_SHA":                       tls.TLS_ECDHE_PSK_WITH_NULL_SHA,
	"TLS_ECDHE_PSK_WITH_NULL_SHA256":                    tls.TLS_ECDHE_PSK_WITH_NULL_SHA256,
	"TLS_ECDHE_PSK_WITH_NULL_SHA384":                    tls.TLS_ECDHE_PSK_WITH_NULL_SHA384,
	"TLS_RSA_WITH_ARIA_128_CBC_SHA256":                  tls.TLS_RSA_WITH_ARIA_128_CBC_SHA256,
	"TLS_RSA_WITH_ARIA_256_CBC_SHA384":                  tls.TLS_RSA_WITH_ARIA_256_CBC_SHA384,
	"TLS_DH_DSS_WITH_ARIA_128_CBC_SHA256":               tls.TLS_DH_DSS_WITH_ARIA_128_CBC_SHA256,
	"TLS_DH_DSS_WITH_ARIA_256_CBC_SHA384":               tls.TLS_DH_DSS_WITH_ARIA_256_CBC_SHA384,
	"TLS_DH_RSA_WITH_ARIA_128_CBC_SHA256":               tls.TLS_DH_RSA_WITH_ARIA_128_CBC_SHA256,
	"TLS_DH_RSA_WITH_ARIA_256_CBC_SHA384":               tls.TLS_DH_RSA_WITH_ARIA_256_CBC_SHA384,
	"TLS_DHE_DSS_WITH_ARIA_128_CBC_SHA256":              tls.TLS_DHE_DSS_WITH_ARIA_128_CBC_SHA256,
	"TLS_DHE_DSS_WITH_ARIA_256_CBC_SHA384":              tls.TLS_DHE_DSS_WITH_ARIA_256_CBC_SHA384,
	"TLS_DHE_RSA_WITH_ARIA_128_CBC_SHA256":              tls.TLS_DHE_RSA_WITH_ARIA_128_CBC_SHA256,
	"TLS_DHE_RSA_WITH_ARIA_256_CBC_SHA384":              tls.TLS_DHE_RSA_WITH_ARIA_256_CBC_SHA384,
	"TLS_DH_ANON_WITH_ARIA_128_CBC_SHA256":              tls.TLS_DH_ANON_WITH_ARIA_128_CBC_SHA256,
	"TLS_DH_ANON_WITH_ARIA_256_CBC_SHA384":              tls.TLS_DH_ANON_WITH_ARIA_256_CBC_SHA384,
	"TLS_ECDHE_ECDSA_WITH_ARIA_128_CBC_SHA256":          tls.TLS_ECDHE_ECDSA_WITH_ARIA_128_CBC_SHA256,
	"TLS_ECDHE_ECDSA_WITH_ARIA_256_CBC_SHA384":          tls.TLS_ECDHE_ECDSA_WITH_ARIA_256_CBC_SHA384,
	"TLS_ECDH_ECDSA_WITH_ARIA_128_CBC_SHA256":           tls.TLS_ECDH_ECDSA_WITH_ARIA_128_CBC_SHA256,
	"TLS_ECDH_ECDSA_WITH_ARIA_256_CBC_SHA384":           tls.TLS_ECDH_ECDSA_WITH_ARIA_256_CBC_SHA384,
	"TLS_ECDHE_RSA_WITH_ARIA_128_CBC_SHA256":            tls.TLS_ECDHE_RSA_WITH_ARIA_128_CBC_SHA256,
	"TLS_ECDHE_RSA_WITH_ARIA_256_CBC_SHA384":            tls.TLS_ECDHE_RSA_WITH_ARIA_256_CBC_SHA384,
	"TLS_ECDH_RSA_WITH_ARIA_128_CBC_SHA256":             tls.TLS_ECDH_RSA_WITH_ARIA_128_CBC_SHA256,
	"TLS_ECDH_RSA_WITH_ARIA_256_CBC_SHA384":             tls.TLS_ECDH_RSA_WITH_ARIA_256_CBC_SHA384,
	"TLS_RSA_WITH_ARIA_128_GCM_SHA256":                  tls.TLS_RSA_WITH_ARIA_128_GCM_SHA256,
	"TLS_RSA_WITH_ARIA_256_GCM_SHA384":                  tls.TLS_RSA_WITH_ARIA_256_GCM_SHA384,
	"TLS_DHE_RSA_WITH_ARIA_128_GCM_SHA256":              tls.TLS_DHE_RSA_WITH_ARIA_128_GCM_SHA256,
	"TLS_DHE_RSA_WITH_ARIA_256_GCM_SHA384":              tls.TLS_DHE_RSA_WITH_ARIA_256_GCM_SHA384,
	"TLS_DH_RSA_WITH_ARIA_128_GCM_SHA256":               tls.TLS_DH_RSA_WITH_ARIA_128_GCM_SHA256,
	"TLS_DH_RSA_WITH_ARIA_256_GCM_SHA384":               tls.TLS_DH_RSA_WITH_ARIA_256_GCM_SHA384,
	"TLS_DHE_DSS_WITH_ARIA_128_GCM_SHA256":              tls.TLS_DHE_DSS_WITH_ARIA_128_GCM_SHA256,
	"TLS_DHE_DSS_WITH_ARIA_256_GCM_SHA384":              tls.TLS_DHE_DSS_WITH_ARIA_256_GCM_SHA384,
	"TLS_DH_DSS_WITH_ARIA_128_GCM_SHA256":               tls.TLS_DH_DSS_WITH_ARIA_128_GCM_SHA256,
	"TLS_DH_DSS_WITH_ARIA_256_GCM_SHA384":               tls.TLS_DH_DSS_WITH_ARIA_256_GCM_SHA384,
	"TLS_DH_ANON_WITH_ARIA_128_GCM_SHA256":              tls.TLS_DH_ANON_WITH_ARIA_128_GCM_SHA256,
	"TLS_DH_ANON_WITH_ARIA_256_GCM_SHA384":              tls.TLS_DH_ANON_WITH_ARIA_256_GCM_SHA384,
	"TLS_ECDHE_ECDSA_WITH_ARIA_128_GCM_SHA256":          tls.TLS_ECDHE_ECDSA_WITH_ARIA_128_GCM_SHA256,
	"TLS_ECDHE_ECDSA_WITH_ARIA_256_GCM_SHA384":          tls.TLS_ECDHE_ECDSA_WITH_ARIA_256_GCM_SHA384,
	"TLS_ECDH_ECDSA_WITH_ARIA_128_GCM_SHA256":           tls.TLS_ECDH_ECDSA_WITH_ARIA_128_GCM_SHA256,
	"TLS_ECDH_ECDSA_WITH_ARIA_256_GCM_SHA384":           tls.TLS_ECDH_ECDSA_WITH_ARIA_256_GCM_SHA384,
	"TLS_ECDHE_RSA_WITH_ARIA_128_GCM_SHA256":            tls.TLS_ECDHE_RSA_WITH_ARIA_128_GCM_SHA256,
	"TLS_ECDHE_RSA_WITH_ARIA_256_GCM_SHA384":            tls.TLS_ECDHE_RSA_WITH_ARIA_256_GCM_SHA384,
	"TLS_ECDH_RSA_WITH_ARIA_128_GCM_SHA256":             tls.TLS_ECDH_RSA_WITH_ARIA_128_GCM_SHA256,
	"TLS_ECDH_RSA_WITH_ARIA_256_GCM_SHA384":             tls.TLS_ECDH_RSA_WITH_ARIA_256_GCM_SHA384,
	"TLS_PSK_WITH_ARIA_128_CBC_SHA256":                  tls.TLS_PSK_WITH_ARIA_128_CBC_SHA256,
	"TLS_PSK_WITH_ARIA_256_CBC_SHA384":                  tls.TLS_PSK_WITH_ARIA_256_CBC_SHA384,
	"TLS_DHE_PSK_WITH_ARIA_128_CBC_SHA256":              tls.TLS_DHE_PSK_WITH_ARIA_128_CBC_SHA256,
	"TLS_DHE_PSK_WITH_ARIA_256_CBC_SHA384":              tls.TLS_DHE_PSK_WITH_ARIA_256_CBC_SHA384,
	"TLS_RSA_PSK_WITH_ARIA_128_CBC_SHA256":              tls.TLS_RSA_PSK_WITH_ARIA_128_CBC_SHA256,
	"TLS_RSA_PSK_WITH_ARIA_256_CBC_SHA384":              tls.TLS_RSA_PSK_WITH_ARIA_256_CBC_SHA384,
	"TLS_PSK_WITH_ARIA_128_GCM_SHA256":                  tls.TLS_PSK_WITH_ARIA_128_GCM_SHA256,
	"TLS_PSK_WITH_ARIA_256_GCM_SHA384":                  tls.TLS_PSK_WITH_ARIA_256_GCM_SHA384,
	"TLS_DHE_PSK_WITH_ARIA_128_GCM_SHA256":              tls.TLS_DHE_PSK_WITH_ARIA_128_GCM_SHA256,
	"TLS_DHE_PSK_WITH_ARIA_256_GCM_SHA384":              tls.TLS_DHE_PSK_WITH_ARIA_256_GCM_SHA384,
	"TLS_RSA_PSK_WITH_ARIA_128_GCM_SHA256":              tls.TLS_RSA_PSK_WITH_ARIA_128_GCM_SHA256,
	"TLS_RSA_PSK_WITH_ARIA_256_GCM_SHA384":              tls.TLS_RSA_PSK_WITH_ARIA_256_GCM_SHA384,
	"TLS_ECDHE_PSK_WITH_ARIA_128_CBC_SHA256":            tls.TLS_ECDHE_PSK_WITH_ARIA_128_CBC_SHA256,
	"TLS_ECDHE_PSK_WITH_ARIA_256_CBC_SHA384":            tls.TLS_ECDHE_PSK_WITH_ARIA_256_CBC_SHA384,
	"TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256":      tls.TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256,
	"TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384":      tls.TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384,
	"TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256":       tls.TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256,
	"TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384":       tls.TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384,
	"TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256":        tls.TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256,
	"TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384":        tls.TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384,
	"TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256":         tls.TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256,
	"TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384":         tls.TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384,
	"TLS_RSA_WITH_CAMELLIA_128_GCM_SHA256":              tls.TLS_RSA_WITH_CAMELLIA_128_GCM_SHA256,
	"TLS_RSA_WITH_CAMELLIA_256_GCM_SHA384":              tls.TLS_RSA_WITH_CAMELLIA_256_GCM_SHA384,
	"TLS_DHE_RSA_WITH_CAMELLIA_128_GCM_SHA256":          tls.TLS_DHE_RSA_WITH_CAMELLIA_128_GCM_SHA256,
	"TLS_DHE_RSA_WITH_CAMELLIA_256_GCM_SHA384":          tls.TLS_DHE_RSA_WITH_CAMELLIA_256_GCM_SHA384,
	"TLS_DH_RSA_WITH_CAMELLIA_128_GCM_SHA256":           tls.TLS_DH_RSA_WITH_CAMELLIA_128_GCM_SHA256,
	"TLS_DH_RSA_WITH_CAMELLIA_256_GCM_SHA384":           tls.TLS_DH_RSA_WITH_CAMELLIA_256_GCM_SHA384,
	"TLS_DHE_DSS_WITH_CAMELLIA_128_GCM_SHA256":          tls.TLS_DHE_DSS_WITH_CAMELLIA_128_GCM_SHA256,
	"TLS_DHE_DSS_WITH_CAMELLIA_256_GCM_SHA384":          tls.TLS_DHE_DSS_WITH_CAMELLIA_256_GCM_SHA384,
	"TLS_DH_DSS_WITH_CAMELLIA_128_GCM_SHA256":           tls.TLS_DH_DSS_WITH_CAMELLIA_128_GCM_SHA256,
	"TLS_DH_DSS_WITH_CAMELLIA_256_GCM_SHA384":           tls.TLS_DH_DSS_WITH_CAMELLIA_256_GCM_SHA384,
	"TLS_DH_ANON_WITH_CAMELLIA_128_GCM_SHA256":          tls.TLS_DH_ANON_WITH_CAMELLIA_128_GCM_SHA256,
	"TLS_DH_ANON_WITH_CAMELLIA_256_GCM_SHA384":          tls.TLS_DH_ANON_WITH_CAMELLIA_256_GCM_SHA384,
	"TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_GCM_SHA256":      tls.TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_GCM_SHA256,
	"TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_GCM_SHA384":      tls.TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_GCM_SHA384,
	"TLS_ECDH_ECDSA_WITH_CAMELLIA_128_GCM_SHA256":       tls.TLS_ECDH_ECDSA_WITH_CAMELLIA_128_GCM_SHA256,
	"TLS_ECDH_ECDSA_WITH_CAMELLIA_256_GCM_SHA384":       tls.TLS_ECDH_ECDSA_WITH_CAMELLIA_256_GCM_SHA384,
	"TLS_ECDHE_RSA_WITH_CAMELLIA_128_GCM_SHA256":        tls.TLS_ECDHE_RSA_WITH_CAMELLIA_128_GCM_SHA256,
	"TLS_ECDHE_RSA_WITH_CAMELLIA_256_GCM_SHA384":        tls.TLS_ECDHE_RSA_WITH_CAMELLIA_256_GCM_SHA384,
	"TLS_ECDH_RSA_WITH_CAMELLIA_128_GCM_SHA256":         tls.TLS_ECDH_RSA_WITH_CAMELLIA_128_GCM_SHA256,
	"TLS_ECDH_RSA_WITH_CAMELLIA_256_GCM_SHA384":         tls.TLS_ECDH_RSA_WITH_CAMELLIA_256_GCM_SHA384,
	"TLS_PSK_WITH_CAMELLIA_128_GCM_SHA256":              tls.TLS_PSK_WITH_CAMELLIA_128_GCM_SHA256,
	"TLS_PSK_WITH_CAMELLIA_256_GCM_SHA384":              tls.TLS_PSK_WITH_CAMELLIA_256_GCM_SHA384,
	"TLS_DHE_PSK_WITH_CAMELLIA_128_GCM_SHA256":          tls.TLS_DHE_PSK_WITH_CAMELLIA_128_GCM_SHA256,
	"TLS_DHE_PSK_WITH_CAMELLIA_256_GCM_SHA384":          tls.TLS_DHE_PSK_WITH_CAMELLIA_256_GCM_SHA384,
	"TLS_RSA_PSK_WITH_CAMELLIA_128_GCM_SHA256":          tls.TLS_RSA_PSK_WITH_CAMELLIA_128_GCM_SHA256,
	"TLS_RSA_PSK_WITH_CAMELLIA_256_GCM_SHA384":          tls.TLS_RSA_PSK_WITH_CAMELLIA_256_GCM_SHA384,
	"TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256":              tls.TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256,
	"TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384":              tls.TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384,
	"TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256":          tls.TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256,
	"TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384":          tls.TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384,
	"TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256":          tls.TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256,
	"TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384":          tls.TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384,
	"TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256":        tls.TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256,
	"TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384":        tls.TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384,
	"TLS_RSA_WITH_AES_128_CCM":                          tls.TLS_RSA_WITH_AES_128_CCM,
	"TLS_RSA_WITH_AES_256_CCM":                          tls.TLS_RSA_WITH_AES_256_CCM,
	"TLS_DHE_RSA_WITH_AES_128_CCM":                      tls.TLS_DHE_RSA_WITH_AES_128_CCM,
	"TLS_DHE_RSA_WITH_AES_256_CCM":                      tls.TLS_DHE_RSA_WITH_AES_256_CCM,
	"TLS_RSA_WITH_AES_128_CCM_8":                        tls.TLS_RSA_WITH_AES_128_CCM_8,
	"TLS_RSA_WITH_AES_256_CCM_8":                        tls.TLS_RSA_WITH_AES_256_CCM_8,
	"TLS_DHE_RSA_WITH_AES_128_CCM_8":                    tls.TLS_DHE_RSA_WITH_AES_128_CCM_8,
	"TLS_DHE_RSA_WITH_AES_256_CCM_8":                    tls.TLS_DHE_RSA_WITH_AES_256_CCM_8,
	"TLS_PSK_WITH_AES_128_CCM":                          tls.TLS_PSK_WITH_AES_128_CCM,
	"TLS_PSK_WITH_AES_256_CCM":                          tls.TLS_PSK_WITH_AES_256_CCM,
	"TLS_DHE_PSK_WITH_AES_128_CCM":                      tls.TLS_DHE_PSK_WITH_AES_128_CCM,
	"TLS_DHE_PSK_WITH_AES_256_CCM":                      tls.TLS_DHE_PSK_WITH_AES_256_CCM,
	"TLS_PSK_WITH_AES_128_CCM_8":                        tls.TLS_PSK_WITH_AES_128_CCM_8,
	"TLS_PSK_WITH_AES_256_CCM_8":                        tls.TLS_PSK_WITH_AES_256_CCM_8,
	"TLS_PSK_DHE_WITH_AES_128_CCM_8":                    tls.TLS_PSK_DHE_WITH_AES_128_CCM_8,
	"TLS_PSK_DHE_WITH_AES_256_CCM_8":                    tls.TLS_PSK_DHE_WITH_AES_256_CCM_8,
	"TLS_ECDHE_ECDSA_WITH_AES_128_CCM":                  tls.TLS_ECDHE_ECDSA_WITH_AES_128_CCM,
	"TLS_ECDHE_ECDSA_WITH_AES_256_CCM":                  tls.TLS_ECDHE_ECDSA_WITH_AES_256_CCM,
	"TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8":                tls.TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8,
	"TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8":                tls.TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8,
	"TLS_ECDHE_PSK_WITH_AES_128_GCM_SHA256":             tls.TLS_ECDHE_PSK_WITH_AES_128_GCM_SHA256,
	"TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256":       tls.TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256,
	"TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256":     tls.TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256,
	"TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256":         tls.TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256,
	"TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256_OLD":   tls.TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256_OLD,
	"TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256_OLD": tls.TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256_OLD,
	"TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256_OLD":     tls.TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256_OLD,
	"SSL_RSA_WITH_RC2_CBC_MD5":                          tls.SSL_RSA_WITH_RC2_CBC_MD5,
	"SSL_RSA_WITH_IDEA_CBC_MD5":                         tls.SSL_RSA_WITH_IDEA_CBC_MD5,
	"SSL_RSA_WITH_DES_CBC_MD5":                          tls.SSL_RSA_WITH_DES_CBC_MD5,
	"SSL_RSA_WITH_3DES_EDE_CBC_MD5":                     tls.SSL_RSA_WITH_3DES_EDE_CBC_MD5,
	"SSL_EN_RC2_128_CBC_WITH_MD5":                       tls.SSL_EN_RC2_128_CBC_WITH_MD5,
	"OP_PCL_TLS10_AES_128_CBC_SHA512":                   tls.OP_PCL_TLS10_AES_128_CBC_SHA512,
}
