"""
pyspartn Performance benchmarking utility

Usage (kwargs optional): python3 benchmark.py cycles=10000

Created on 18 Feb 2022

:author: semuadmin
:copyright: SEMU Consulting © 2022
:license: BSD 3-Clause
"""

# pylint: disable=line-too-long

from io import BytesIO
from datetime import datetime
from platform import python_version
from platform import version as osver
from sys import argv
from time import process_time_ns

from pyspartn import SPARTNReader
from pyspartn import version as spartnver

KEY = "930d847b779b126863c8b3b2766ae7cc"
BASEDATE = datetime(2024, 4, 18, 20, 48, 29, 977255)

SPARTNMESSAGES = [
    b"s\x00\x0fl\x03mX[\x19HQ\x19W\x92\x98\xed\xbd\x17\xe1\xbbQHI\xd5\xf4\xe55i\xc3V\xc0\xd1\xce\x88)]\xf9:\x1blBB\xfb",
    b"s\x00\x10\xe3\x14\xbeH[\x17\x08dL\x1aF3\xa0\x05\x0eO\xf7)\xfa\xe2jF\x80\xb4\xab\xbe\x85\x9f\xb6\xd1\xcc\x99\xa9B*\xd4\xceQ~\xbd?:\xed",
    b"s\x00\x12k3l\xe8[\x1f\x08\x9c&\xca\x15\xb1\x01\xa0R\\w\x02\x8c\xd5\xbd/U\xa1\xe4U\xc9\xd7\x03\xdc\x9e\xe9L\xc0f\xd8\xdc\x85\x1fB\xebM\xbd\xc9`\x1e",
    b's\x00\x11c#mX[\x17\x08\xcd\xa1}\xe8n`>#\xbe\xf6\x10\x15\xd6\x97\xdd\x8b\xb7\xa1\x8d>\xf2za\x81AJ\x11V\xfe\x14v\xd3\n*QT"',
    b's\x04_\xee\x03m\x80[\x18\x88=\xd5j\x7f\xf9\xbc\xc3\xbd]=\x1b\x96\xc9\xdb\xd7t\x0bx\x0b\xad,\xb4e\xef\x90\x1fo\xc1qy\xa8\x7f\xf8\xf6\x8c\xb0bh\x04#\xa7\x0b\xd0\x89\xf9\x95\x1f!\xc3\xab\xdc\x0c\x90n%S\xe5\xb1\xeaH\xae\xe7\x99\xb29\xee(_\x04\x80\xeb\xa0\x9f\x0fS@J\x88JF\xd7\x00~\xbah\x10f&\x9e\x98\x8f\x1a\xa1\xcd\x06\xac\xf2j2\xfc\xae@\xd5\xc4\xefb\xcd@K\xc8\xf2\xf9\xf0\x8c\x04hyHK\x9an\xe4\xf5\xfa<\xde2e\xf8nT\xc4Mz\x91`\xd3\xb9\x17\xf1\x1f\x94qr$\xfc\xf3\xbb\x14\xd6\xf1\xd4P\x8e\xa0\xd98\xb6\x15\xc37Y\xd5g\x10\x87\t\xe6a(\xb9F\x80\xa6\x9f\xdb\x02\x116kn@6\x16\x84\x19"\xf58\xd2\xd5\x13\xe2\xe4',
    b's\x04\x1fa\x03m\x80[\x18\xc8\x03M\xaebT\xc6"a\xefG\x00\xcc\xe5:\x1f\xef\xc0\xa0@mW\xe4\xcc\xce\xa1\xdc\x07Z}h\xcf\x8a&\xd6u\xaf\xb2x\x84\x9c\xdb\n\xc0\xb5&^\x89[\x9a\xe9V#\x9b\x08\xbes?\x92\xa17\x12\xd5\xf340',
    b"s\x02\xf7\xeb\x08\xd7!\xef\x80[\x17\x88\xc2?\x0f\xe8\x9a\xd2\xcd\xcb\xcdd\x08\xfb\x15\x9e\x9d\x18Z\xd9'\x1f (\xbb\xce@6\x85\xe0\xa1B\xdc\xd0\x8d\xa8/V\xaa}\xde\xd2\xc0\xe8cU\x8a(\x90\xa8\xe5\xf2s\xf9\x83F\xf3D\x81I\xe3\xda\x93\x1b&\x05\x83\xd4[R\x89\xd8\xfbOU\x99\xc2AQN\x86j\xecD\x98\xaa\xa6l\x98\x8dk\x9b\x9cr\xad\xc3\xef\xa7\xb5\xfd\xc2\x8bY\xda\xd7\xc4\xa6D\xd80r\xcd\x7f\xd7\xfa\xae\xcf\x9fh\x8di'U\xb3\xbe#\x7f\x83\xdcx\xddN\xd1\xccw\x80x\x13\xc1\x1c^\xdf~\xe2\xe9\x00\xf5\x83\xb1\x14Y\xa2\xcf\x82\xba\x1f\x88\xd7\xc7\xae/`\xc3\xad\xc4\x0bAh\x0b#\x14b97\x0cw$\xf7\xad\xae\xce\x17\x15q\x1da\x84IY'\x8f\xe7\x11\x9a9+\x01\xe4\xe5\xaa\xb3[\xb4\x89\xc2\t\x8cn\xb7\xac\x86Z \x17\xf5?j\xe5\xca3\x08+\x8eU\xd2J\xa02\x99\xf7T\xdd\xd2\x8av\xa5(\xc8\xf1y\xff:\xe7\x84\xbc\xd6JC\x9e\xd1\xc6\xeb\x85\xcfa\x1d\x1a\xd4\xd7\xf4\xb5\xa1M\xf9\xd1t\x1e\\\xdf~\x1d\x17G\x84\x1d\xe5\xefcy\x7f\xf7tM\x0b\xda\xb1'23\x07\x01\xcb\xd4\x18ZT\xbe.B\xf8*\xbc\xb3n\xb1\xc3\x04!\x88\xe6^\x9e\xb9\xa8\x84\xbd\x0c\x9dO,\x93\xce\xe1[\xd6\x0e\x98n\xb7\x14\xa7[.C>Q9$\xddO4\x18\xbf\xf3\xd5\xaa\x8eV\x0c\x7f\xa4\x00:*\xeb5\xa8\xb9\xee\xf3_0\x03\x16\x07\xa5\xc6\n\x1et\xbb\xdf]7\x95=,\xf7(y\xc5Z\xf8\x97\x9d\xcdG\x9e\x1a\xf6Dr\xbb\xeaM\xacv\x14\x8b#\xe1x\xc3]L*\x88\xa9?\xbb\t5\x19k]p\xf8\xee\xeb\x81,7\xf86F\x8a\xc1!\xe5\x86\xab\x95\xa1\x88\x8c\xa5\xa6\x84\xd8(\x11\xabU\xf0\t\xc5\xbdQ\xae(\x02\xad\xdee\nI=\x85\x96\x86\xa9\x1d8\x86\xc6w,3\xb0\xe3'\xcf\xc3w\xa4\xc4\xa8\xbaMI\xc0\xa0j&\xf6\xa7-\xf5\xb4\xd3\xcb\xe0M\xc4#fFy\xb9\xd5",
    b's\x02\xefn\x08\xd7!\xef\x80[\x17\xc8\xdb0\xbe\xeb\x15\x90s\xbe\t#\xb7w\x12\xf2\x9a40,\xde\x13\xce\xb2\x8a\x12*\xf0\xc8rr\x87\xaeP\xa4\x83,{p\xfa\xfc}>\xa8s\xb4\xa4\x14\xfb\x10\xcf\x1e\x07\x1e\\\x8c\xcc\xd6\x89\x03C\xa5\x10\x83j!f+\xa1=]o\x1bS\x06Xn\x10\xcf\xa6\xf9\xa8\xbe%o\'\x0c\r\x8f\xdb\x1b\xd7z\xf2\xb4#\x0c\x8fF\x18\x9a\x01\xf0g+\xfe\xec%h\xb69\x1cs\r\xc4\x1cS\xc6\x95\xea\x99Ay\xc2;?\x02\xce\xe7\x87Hj\x17"\x94\x06\xffm\x01u\xa1\x97p\xf0\xcc*\xba\xed:\xbf\xd8\xff\xd1*\x1c\x1b?0YO\x01LK\x81i9\xc2@\xf0\xd0\xff\x9f8\xa3\x07#\xd5\xa4A\xa6Y\xf2\xbfN\x14\x0c~{\x7f\x1dXu\x15\xfdK\xf1\xa33\x9d\xb5!\xa7+\xd7\xf9L\x07\xa6u\x8aV\xe7\xe7\xf4\xa5\xf2Q;\x17\x1a\xc1\x8d\x94W\xfdc\xe25g"\x8a\xd3\xe3\x19\xc3\xfdR\x13\xa7\x08\xb5\xc2\xc8f\xa5\xf5\xef\x0f\xa6i\xf2\x95\x8aL)\xe6\xe2\x87\xd2\xa7\xe9\xa6\x05\xb8}\x9c\x05\xf4\xe3X]|@N\x16\x8b!0\xc62\xca\x92\xaa/m\xc9\x83\xf2\x1f9(1c\xb07t%3\xadJ \x85\x8c\xdb\x19\xca\xa5W\x17\x94|\x98\xae\x93\xa5\x8f\xdc\xc1\xe6\x13\xe1)\x8c\xff\xdc\x9d\x90}v\xba\x12\xca\x8bk\xec9Z\x10\x11\xd8\x1f\xcb\xb6\xde\xab\x9c<\xc0\x96\xddXo\x11\x9dD\x83\xb3\xed\xaa\xe7\xbd\xff\x12\xa4\xbam\xf4\xa1)\xa0\xc6\x08\x1e\xa0\x1d0\xa8\x19Q\xf3\xd7\xf3\x91\xe4\x82\xbc#\xf5 \xc2\x15T\x10\xcb>y\xa8\xbcm\xf0\xf3\xb6\xd8o\x0c\x1d\x11\xdbD\xb04a>\xcbV\xc4B\xe8:I\xe1b.S\x01\xbbkTy\xa3\xdb~b\x857=/\x8f\x11\xd5q\x89\x01w6\x14\xbe\xf3\xb0\xbc\xef\x87\x14\xc0\x15\x19\xb2-\x0e\x9eY\xfb\xb4\x95\x00\xd5_\xe96\x04\x07\x93\x80\xa07t\x98\x0cF\xf7\x88\xder',
    b"s\x02\xefn\x08\xd7!\xef\x80[\x18\x08\x01\xc1\xdd\xeb\xc08\xbd\xd6</$N\x1d\xa9\xc4\xa2\xed\xe4|\x12R\xf4\xe6:f\x1fn\x88z\xf3&o\xee\xf1\xc3\x1aw\x0f\x9e\xb5v\xbeI\x97\xdd\xb2\x0eI\xd0\xf1v\xdb%\x1e\x10\xa0`}m\xa3Dc\xf6\xbc\x95\xc8\xed\xb4A*\x8e{\xe6\xaa\x1e\x86\x92*3\x96\x94\xe5\x1b\xf0\x0f\x1a\x94m\xdc\x10\xaa,4\x1d\xaf\xee\xc8\x98\"\xb6U=\xb0\xcd'\x1f\xb5V\xe1s5\xf2\xab\xe3\xba\xf7\\/\xe8\xbf@\x97i\xa5\x82\x0fU\xd6ch\t\x81\xf2\xcf\x91\xbal]q\xc5-`\xd4\xeeh\xf6h\xc0R\x17\x04\xec\xd1\xae|\xb4\xe4,v\x8a\x1b\xd1\x10\xcc\xa7}8b0\x07\xe8\x18\x1ct\xeb\xe4\xf3h~\xd6\x1aff\xbf\xf2\xfc\xa3\x8f\xf0\xcd\xc9\xb3C \xfc\xdf=P\x8a\x98\x11\x0c\x80\x91\x9b\x88QzQ\xa1\x9f\xd1\x88\x02\xe2\xfd5-\x18O\\g\xda\x0b\x02\x9e\xb8\xdd!n\x90\x81}\xb4\xb05\x97\xb0\xa2\xd9;j\xe0C\x9c6\xb7-\x97\xa1\x03\xca\xbd \xca\xa2\x8f\x05\xac\x8b\xa2By3H1\xb5/\x8c\xed$\xbe\xff\x16\xd1\xdb\xf3\xfd|]\xa1\x8a}Af\xa17\x15,@\xae{T~\xe6D\x9et\xb1\x84\xde\xa9\x9de\x1b\xa1\x0c\x1b\x80\xb6_\xef\xe5D\xdbH|\xcf\x0f\x1b'\x0b\xefv$\x930fR \xfc\xa16\x825\x12x>\xc5\x9b\x0c\x15[`)z\xd0Rp\xce\xa0\xbf\x7f\xba\x06n\x97\xe6\xd4{}\xe2\xacm\xddyb?\xce\xc0\x04\x0el\xffoT%\x05\xa6u\xd9\t\x89\x9a\xdd\x03*O\xe8\xc4\x81G{\xf5\xe1\x83\xd3\xce!\xda\x19\xf7\x19\xff\x03\xa2\xfd\xeb\xfe\xfe\xe1\x98l\xba\xd6\xcd:\xab\xa8m\xd1\xec\xde\xec\x15G\x92\x08#n#\x95L\x83\x8bO\xcf\xf6\x02\x03\x87h(qa\xec\xe8\xe962K8\xeb\x8c\xc0E_\x97\xbf\x00\xb4\xc9\xc4X/\xc9\x0fI\xba\xa2\xff\xa6*l\xa9q\xa49\x01\xbf\xc1",
    b's\x02\xecf\x08\xd7!\xef\x80[\x18H\xc8h\xa9`\xd3\x9e6\xe9g@\x0ei\xa4\x1d[\xda\xba\xcaL\xc1\xe1AD\\\x0co\xa4\xc7a\xf4k\xdd!\xbfb\x04\x01\xdb\xc7*\xd5T%\x1a\x13\xa6\x05Q\x884\xfb\x1e\x8a\xd5W}\xcd\xebN~\xaf\xf8v[m\xab^G\x99\x1b\xee\xaa\xf1\x13S\xd5\x7fp\xc0y\x8d\x8b\xf0\xcf`\xb2O\x94\xf8cT\xc7\xb2\xbdZ\xc8\xe1_b_8\xb6~\x1fz\xde\xfbG\x7fnD\xb7&!\xdd\xea\xa0\\o\rG(\x8b\x8eKY\xa2Pew14\n\x1f\x88+Y\xb0\xd0?\x88\xcd\xc5\x7f\xc5\xf3\\R\x96?\xa8o\xa5)\x9bLz(n)\xf1\xc0\x1a\xdd\x1drU\x07\xf5rZU\xc0\xdes\x1ar\xb0\x91\xaa\xa6\x80\xc1_\x18.\xaf=\x86\xd4\xd5\x15\xc1\xb5$\xc3\xec\xc8\x0b:G\xa5\xaew)\xf6\x16\xd1\xf1Lv\xcb\x14\x91y\x7f\xea-\xb4\xd7\xad\xd1\xbf\xa50\xcf\xae\xfe\x81[vPTF\xc8\x92R\xde\x12\xb06\xa9!\xc1\xefl\xfanx\x88[\xceQ!\xe1\xa6\xedp\xc7\x92s\xf7\x81\x06\xe4\xdd\xdfs0M&\xeb+&\t\xacX\n\xba^\xd1\xd1\n\xeb\xb71\xb0]\xd0!t\xd5\xa4z^\xfc,\tup\xc0\x0f\x8e\x95\xc2\xd2O\xf0\xb9\x08\xe4JtR\x0c-\x8c\x814\xac\x93rI\x86s\xa1\xefg\x13y\x95\x19\x9f{\x98j\xdd\xddOCA\xae\xc7\x82\x813\tp\xd9\x8e\xc2%\xa0#f\x85\x95Y\xc2]\x05\xb2\xb2\x1c\xe3p`\x84\x9f\xd2\xc7\x80\xfb\xdbl\r[b\xec\x88\xfd&\xa6\xdbB\xe5\x84\xcb\x1a$\x8c\x04\xa0\xe0\x8e\xcb\xc4o\xf0\xcbq\x1a\x08\x9f[A@Kb\x1f\xa7\x16\xab\x01\x81"\x1e#1\xec+d\xe3\xf4.\xee\x98\x14p\xeb\xfcq\xa4\x12\xcb\n\xbf\xcb\xe9\x81&\x17\xd7\xabz\\\x1a\x9a\x18P&\xe0\xf70\xf1\x7f\x84\xfe\x0b\x90\x95\xabA\x89%\x84\xc4\x81\xdb\x1a\xadT\x89\t)',
    b"s\x021k\x08\xd7!\xef\x80[\x18\x88\xc8\xae\x98\x96\xc9\x81\xddaJv\xb4\x81Hy\xe8\xc3\xa58\xa9%V2U\xf0a\x1d\xf9>~\xfaG\xa6\xa6~\x84:\xca\xf0r\xdb\x11}\xc7'*L?\x07@\xc1\xc2\x9b\xc5D\x90\xf1\xbc\xd2\xcbx\x90\x9a\x83+\xe5\xa4\xef g\x91\xc5\xea\xba\xab\xab\xd4\x8d\x9bK\xd5\xcf\xc4\x08\x14*q\xb4\x82\"\xd9\x07\x92\xd1IT/`\xe0p\xdb-",
    b"s\x02\xd2\xe3\x18\xd7#@p[\x1eH\xb2`~\xd6!,r\xc7\x0eE2\x11\xaabT&AA\xb3\x9b\x12\x0c\xe8\xc6\x1f&\xbe}\xc1\x0cf\xa2\xa643>\xc4o\x8d\xea\x04(*\xfb\xe2U`\xc5\xe9#\x18r\xb4\xaf}\xe57s0\xd21^\x87>r%\xe58q\x7fPk\xf8;\xa2\x9e9\xbc\xd0\x03\x00\xbb\\k\x15=\x0bx\x90\x8fN\x99\xc2{\xc0\x95\x10\xe3\xe2$\xbc\xa9\xa1\xd8\xbd\x93{\xdf\xdcA=m\xd7w\r\xb2$\x9c\xd2\xa5kS\x87\xfb\xbe\x03\xbcnB\x00\x82\xa4\x112e\xca\xf5`J\xe6q\x0e\x17\x05BMn\x1e{@*`N\xdc\xbb\xc6\x15\x9f\tc\xac\x84\x93\xbaPx\x08\x82r\xc9\x0b\xec\x99\xac\xeae\x10\xdd\xc6\xb1\xbf|k\x08%n\xf0\x94\xa5\xe62()$\xbc\xc9\xf6\xd8i]q\xe2\xe0\xcd\x1dW\xa1_\xa7\x06\xa2\x90d\x07I!\xb2\xb8\xb0\xf0f\xa8\x85\xbeE1\xab:\x97\xaen:\xa1\x96\xc7P\xd3\x15{H\x0c~\x10\x1f\xca\xe5\x1eg\x16M\xbf=\xbb\x1d\xdd\xbe\x8b\xfe\x0c\x18\xa1<\x0b?\x1b\xad\xad\xa7TR1\xbb\xfb\t\x1e\x82\xd6#AL\xe29nN\x067\xbb\xd9\x90\xd6;&3\xe9\x95\x1a\x01Y\xb8\xed-\x19\xf8m~\xccl\xa7\xb5\x95\xa0\xfd;/\x072\xfe\x99*\x83\x97\x84\xa2\xa2\x8c\x07aI\n\xfd\xfdv\x0f\xe6\xedA\x93\xde\xac\x10\x07\xa6\xf5A\x94\xb4zlv\xa0\x87Iw\xbd\xbf\xdc\x0e\x13b\\H\xe3;\xb8\x82\x86\xee\x0c\xb0\xdf\x16\xd5\xfe\xd9\xb2WNh\xfd\x82c\xa1\xe8\xd8\x1f\x0f\xf9\x9e\x0eO~C\x8b]i\xa9bT\x9a\x9c\xdc\x0cCnR\x1a\xd7\xf4\xc8\xec\xaa\xaa8\xc3X\xc9\xcf\x05c\x9dU\xe6\xff\x1a",
    b"s\x02\xe4b\x18\xd7#@p[\x1e\x883\t[a+\xc2\xe5\xc5\xfa~\xae\x08\x83\x15>\xc7Ph\xdf\xfb\xe8\xb0\x97yQ[\xf0i#U\x96\x0c\xec\xc1\x19m\x8b\x87\xa8S\xde\x9f[\xaf\xf2\x9a\x11\xc2\x88\xbb\xc5\xc0\x9f\xcf\xfd\xab,\x9d\x12\xca\xc3\x1b\x10Y\xfb\x93 \xd6mt\xa6S\xd5|\x89\xa1G\xdfb\x13\xba\x07\xa7J\xab\xf0G\xe4\x1f\xa1\x9cy8'\xd2\xb1\xcfd\x13\x15\x05\x86\xbe\x0e\x86'\x92\n\xcde\x04j\xe3\xa7@\xe1|Zpfi\x17\x19a\xa0\x14\r\xe7&\x0b\x8d#hB\xa5A\x86\xd9n.\xad\xa4\x86\xd37\xb3\xdcO\xda\x07\x01Rf\xa8\x84\xdaa\x92\xbe|\xc6c\xab+\xfc\x01S7\xdb\xb8\x1e\x7f\x90H\x80MDXx\xb8\x01)NC\x80\xec\xc2\xa5\xa9\xa1\x12-\x1a\x0f!\xbe\xff>zU\xec3\\8\\\xc3\x80\xb01&\"r\xa2\xabd*7\xb5K\x97&\xbb'\xdaX\x0b.^~\xea\x03\x7f\xe0y\"\xe6K\xbe\x05\x8f\xef\xd3\xc9\x8f\xa7\xec;\xfdRqM\x1eM\xd82\x95.\"\xd2\xd5\xec\x86\xb2J\x9f\xdb\xce\x1c\xabu\xc1\x1c6\x10\x06\xeb\xca\x85\x87\xf8/\x02R\xc4\xeb\xa7U\xf8yZ\x9c\xf6\xda\xc9K@\xa6k\x98\xb5\xad\xdc\x04\xa1S\x12\xff\xfb\xcb\xd5W\xad\x13j\xfd\xd3\x95\x91\x7f\xdb\xae\x1a\x81S\x0c6\xe1CB\xb1\xc8\xe0\xcf]\xef\xd9\x86\xc4}\xc9c\x13lf\xe6~\xa8qz8\xf8\xec\xa9b\xd9\xa3\xfah9\r\xe4I\xc8%\x9e\xd4 \xfdYwM\x95\xf5\xdeS\x8d\xc6\xd0KF\x16@\xe5\x12\x07\x96\x90\xacU\xa5,6_G\xc3\x8e\xbf\x8a`\x9d\x8b\xd3\xb0\x03&m\xe2\xf3hT_-\xe7\xe7\xb9\x11uew8\xc1B\xb8\x9dzO4\xd2cD%\x9d\xcb\x1fa\xb8\x0b\xe5\xffp\x02\r\x10\xf9\xd0hzGTP\xfb\xcc1)\xa2 \xe4\x8f\xce",
    b"s\x02\xeae\x18\xd7#@p[\x1e\xc8u[n\xb1\xc4\x1e\x9f\x19\xee.\xffW(\xed\x96\xe7t\xaa\xf8\xa8&\x13\x84+-\x15\x86\xd3I\xe9\xfd\xb1'\xc7\xe5\x99\xc7\x87\xd5Hz\xff42\xd3\xac\xf5\xbf\xb2\xad_\xcd\xa8\xfc^\x9f\xf6G^;)\xd4\x9dD\xc5~\xder\x81aiW\xca\x08\x87\xed\xea\xd2m\x9e\xa5\xdf\x1b=\xc2$X\xca\xa00\xd2?F\xdb^\xf1\xc1\xb0\x87Vu\xd2\xaf\x9a\xb4\x8e\xbfx\xa4}O\x11\xf4\x92\xd1\x7f\xf7&\x8d\x8dM\x91\xde,3;\xb4\xa0+\xe2\x9a!\xca4\x8e\xff&U\xd8\xfe7\xc3\x1f\xa2\xb1\xd2\xed\xae,\xf1\xa4\xe0\xcag\x17\xe7\x98H\\\xe0{\xb1?\xa9\xa7\xdeH\xb4\xf9~\xab\xf0\xc9\xadj\xcb\xfa\xd5G\x07\xf2\xc0\x87\xcb\xed\xac\xab\x95\x06\x8f\xe7H$U\x9709\x13\xc8\xa1\xa4\xe86\x03b\xbf\xe7\xdf\xc9\xda\xb8\xb3\x12D\xef\t\xef\xfa\xcf/d\xdc\x81cv\x88\x94\xbe&\xac\x92\xd4\xbd\xf9a\x05E\x9c\x02\xb1\xe7Yp\x0bk\x94\x0f\xd1\xa1\xdb)gE_ ,\xdd\x98\xb2\xff^\xcb\x02]\x0c\xcftw\xc3q\x88\xc5\x1a6\x87\xb3\xaf\xbe\xaaL\x0cw\x96\xa0\x92\x98\x03\xcd\xed\xea\x1cM\xcfB\xda\x13\xf93\x97o\xc5\xfc\x83\xf6'3G~\x86M\xdb\xf6\xa9m< \xabnIz\xa5\x03{\xb1,\xabuSG\xe9\x0cH\xa0\xfc\x11p\xe8\xff\xfe\x8ex\xecv\xf1\xe8\xe5C\x1c\x99dt\xc4!#\xe6.\xa0!L\xdf8\x18\xb3\x8f\xb0\xaf_ez\x82?e\x95\x83\x95>3\x97Tt\xc0J\xb0\xdc\x81\xb9\xa5mG\xd5&6oE\x98\xb1\x02\xd7`\x1a2w\xac\xc0&\x8fTCD\xfa\xefy\x98S\x1d\xa7G\xf0gd\xc38/\xc8G\x9e\x12\x92\x8di4B7\xc6X\xc2\xb3\xc4\xf4\x82]I/\xbbM9\x067\xf6\xa6_Sw\x9b\x1dt\xd3\xf8\xf4|M5(\xa8(\xc9\x94i\xac\xc4j",
    b"s\x02\xeae\x18\xd7#@p[\x1f\x08\xcb\x10NS\x93\xa9tD\xaf\xb3R\xff\x97%\xa3f'\xcbr<\xb6& \xf5\x13[. \x16\xb8\xb4#\xdc\xefV\x13\x06,Y\x8b\xff\x03\x9c\x8bx[\x9e`\xf9\x08\xf8L\x8b\xba\x13\xfcg\xb4\xb9\xca\x9e\x8cC\x90Au\xa7\xd4\x81\xf2\x16r2\xec\x84\x93\x9a\xfa\x8d|>k\x1b\xf7\x08\xdc\x86'\xa6D\x07\x0e\x9bJ\t*7\x9c\xde\xf2\xbb\x08ALl\x8e\x01\x19\x18\xe9U\xe5z\x00\x89Q,s\x03\x81t|\xf2\x85\x97\x97S\xfc\xdcg\xef,\x05:\x82F7\xec]\xab\xbd\xd7\xfa\x00 \x1bd\xaa\x9b\xe5\x0c\xa68\xd6X`{\xa0\x9d\xcd\xc2\xd6\xb7&z\xddd\xd4\x1cC\x90\x9ah\xbb\x1b\x03\x9a-\xabgn\x10\xe5:\x96\x15\xe9\x04\x96Dc\xd9\xa8$\xb1\xe2Ra\xc4\xa1\xc7]bV\x9bK[<\x92\x9e\xb499\xeb\xa5\xea\x10dyV\\\x06)\x87S\xec\xc7\x05[\x1d\x19\x06\xcee\x013\x92\xa5\rQ}\xc5\xef\x8bZpi\"\r\x08QE\xbc\xae\x91tC\x1a\xd7|\xf0X\xcb\xf9Q\xfe\xdc4\x81\xa1\xf6:\r\x93G\x96-+\xcbq`u\x96\x8cy\x0f\x12\x86P\xf5Z \x9e7U\xf6\xbbT\x99G#6SD\xdf\x1b\xecA\x02\x93u\xbe\xaa\xdaKSr\x98\x85\x84\xb1\xfe\nn\xa6\xb2l\x97Q\xa7\x15Z\xb7\xdf\x00\xb0Z;\xef1h\xd8\xac\xd4\r\x88fWews:\xc1q pK|^\x01\xca\x1a\x7fH\xc07\x8fG\xd2Y\x9fU\x98\x9c\xab\xf1\xcd\xb3\x1dM\xbb;\xe2\x85\xf0I/O\xec\x8d5\x9c\x1fw?\xb4\xab\x1b \x08\x07R\x017\xf9\xc2IjmsH\xc5M\xaf\x04\x88/\xcf\x88\x96\t\xb0\xd06\"\xcb%q1w\xc9\xc8\xf3\xef\\\xa6\x1d\xb6\xee\xbfL<O\xa5\x07y\xb5\xa4\xdd\t\r\xdc\x0b\xb2~\xc7z\r\x97w\n\x80-\x7fB\x94\x85@\xd1\xe9/",
    b"s\x02\x1e\xef\x18\xd7#@p[\x1fH\x0f\xb5<2\xad\xe8f\x93.\xe2\xa4\xd6\x1c8\x1f4\x84\xa0Z\x00/%\xeeR\x9fR\xfc\x9c\xdb\xe7!\x81wg\xc8\xef\xb1\xa98H\x95\xd2&\x99\xc70j?\x92ql\x1f\xb1\x15P\xa1\xda\xff\x01p\xef\xcc\xbc\x01",
    b's\x02\xc3\xe28\xd7!\xef\x10[\x1d\x88}\xb8\xdeu\t^<;\x03J\xe9x\xfd\xda\xbe;D\x01\x94L\xaa6\xe04\x18\xa6\xac<3j\xe0\xa8\x1a\xf3\xccB\x0c\x9d\x1b\xa5/\xe4:\x90yJ\xfb\xe7\xc2\xa0"~\xf0\x8c\xa9\x90\xac\xc0\xadyr1$\xf5\xd5\x1e\x8cb\x89\xe6\xa8\xf5\x8aG\xd5\xfa\x8ax\x91\x97\xc7a\x0b\xd6\xa41\x91u\xf2\xdf\xd4\x10K\x8a\xdc\xe9X\x9e\xeea\xdd\x0c\xe4\xa56M\xa4\xe6\xf4j\x86\x1f\xe8nLbT\xde\xb5\\u@@\x8c\x96\x04u \xb2\xd4j\xf6\xce\xcd^\xfe\\\x07\'(\x08n\x83\x9d\x1bB\x9e%\xe8\xa4\x150\xee\x1aA\'\x8d\n\xae\xd1\r\xd4\xa9F\x98\xc0\x1fi\x8c$3\x1d\xe4N\x9b~1\xc3\x0c\x06\xbc=\x89\x7f\x03\xa4<\x9dr,\xea\x82\xbfN\x91qQ\xd8D6X\xac3\xb1\xb8\xe9\x8d\xaf\xe9\xce\xaeN\xab\x04\xc3e;\xf81F\x8d\xe8s\x10\xe5\xd1`\xf3\xdc\xacP\xe0\xfeK#\xa6\x1aE~\x7f>\xf2\x05\r\x82%[\xde)\x89\x80\x8f\x9e \x9c\x0e\xacm\xad\xc0\xd9\xd5]\xde&\x18."\xa2\x0b\x87\xc3\xf9\x16\x158\x16-\x824\xd0\xf48\x17\x9f\xef\xa8\x15\xa8\x1b\xf2\x8d#\xd4\x19LK\xd04G\xd6zIN\x80\x10\x9f\x1b||L\xf3s\xb0S\xb0=\xf8\xb6\xb5\x9e\xb5\xfd\x1e\x8d\x9eE/\xf4\x103\x8b\xd9\x93\xb5aX\xfc"\xf3\xaa^\xb6\x06\x99\xe4\x82\xb9(\xc3\xfd\x00&\xbeM;\xaf\x07)\xe6\x8b\xb7L\xa6!z\xa6\xb7\x0eR\xca\x8b\x91\xcca\xe9\xe3Uvu(\xca\xdc$\xd6\t\x9e\x96\x1c\x97E\xad=M',
    b"s\x02\xc6`8\xd7!\xef\x10[\x1d\xc8A\x94\xf7\xb7\xae\xdc\x7f\xa3\xeaPq\xdeB\xb1\xd3\x9dt\xf8)\xf2\xabg\xfbr6\xba\xa9V\xac7\xa5\xef\xf1\x96\xd0H\xc3\xab\xbf\xee?\x99\x14a\xc4O8\x08;\x06\xc8\xccW<\xff4K\x92)\n\x0c\xd9Kt\xd6\xe50\x0f v/\xbb\xa1\xbe810.\xdc\x97\xfa:dA\xc3\xe5\xa3E]W\xc2\xa1\xb4p\"E3c\xc7\xc0\xc5\x99r)y\x80\x07\xb6\xbc\xbd\x17fv|\xc0\x1c\x1d\x89I\xe30\x88\xfaN\xddb\xbc\xcd\xb21\xb1)\x03t\xf5\xbf\xc9\xb3]U\xb6\x9a\x1d\x06\xf4\x03\x83\x85;\xa7\x7f\xe2\x16\x10\x95\x8d\x12\x98\x8d\xe5\xc5\xbdA\xf7{\x81\x04AZD\xa2`\x81\xe4\xc8KF\xf6\xaf\xd1\xe8;\t!r\xef&p{g\xf9\xdb'\x04\x90o\x16}U,-\xa4\xc8\xb2\x18\xdf\xf6\x1cw)\x95\xc2\x8cCo\xbf6\xdc\x13y\xcb\t\x92\x87\xd8\x0f%\x19\xcb%\x00\x05\\\xb5\xe6\xb4\x04\x96\xe9\x06\x0ff)0\xdf\xbc\xc3\x80\xf5ce\x17Nr)\xc1\x1f\xf9\x9b\xb8\xe5\xe8\xcf\xe3I\xac\x18\xc1\xbe\x11\x8dG*\xa3u\xd4i\xed\xd1\xd3\x9d\xce9\xde\xb9\xd0oN\x7f`\xe4\xe2LUO\xee\x16\xa0\xd6\xf6\x9d\\\x8b\x82s\xc3C\xd9\xbe\xe5\x9f\xe5\xcc\xd3\xbfj]\x9e\xa2\xa6\xde\xc0\x1f;UB\xdd?\xe2\x90s\xa4d\xcf\xf3\xef\x1c\xa9\x15\xcfJl\x9fC\xaf,\x1fQ\xdc\x15\x8e\xfe{\xf6Q\xe9o\x11\x97\xeb\x086?-\xae\xc3l\xc8\x13\xb2\xc5\x95n\xeei~\xac\xbe'Dy\x04\x9a\xe4\xf70t\x80\xa2!!lMq\xc9\xc5>\x0b\xe9",
    b"s\x02\xbbb8\xd7!\xef\x10[\x1e\x08\xcc\xa4\xc8=\xfe\x91\x16F\xdf\xd6\x87\x07\x86\xba\xa8\xeb\xa7\xec\x8a\x93\xa3@\xab\xd1\xcdZ<\xad\xf6\xbaL\xbe\xf5\xd3\xe7\x14\x91\xa0D$Q\xebl\x1bt\xc6\xa8\x1e\xf3\xe4\x14\xc4\xd2\x90\xe4\x9d\x1e\x87\xeeW\xffL\x8d\xbd\xa1LxsK\xafE\xb8waU:\xbb1m]\xbbe:\xbf'\xef+\xaf\xe3j<\xe4(]I\x01\xe5\x13\xf8\xf9\x9dI\xf3\xbeS\n+m\xc5\xc0Y\xfe\xb3\x0f\x99(9Th\xa4\xb5\x8e\xb7t@\x91F'\xaf\x16(}\xee\xe0;\x1e\xd6\xc1\xe9\xf7V\x96i@,%\xa7\x19]Bc\xf4]z\x89\xf8\xb6\xd4\x91RH!9`\rXB\xfa\xc7\xf5Q1n\xc8\xcb\x9c\x92\x19\xcfi\x84\x91\xfc\xbb~\x94\xf5\x1561\xa53\xec=\x02/\x9c\xd56\xe3w\x00\tT\xd7=\x11\x8e\x9fpf{\xedI\xc9U\xa9MM\xd0%VbK[t\xe0iz\xefd\x07\xac*\xae\x0f\x9e\xe6\xcb\xd8\xcdl@\xca\x0c\xd4\xd3\xa6\xbb\x8c\xc0\nM~\xc6>\xbd\x1a$B\x07\xea\x8f\x17b\xfe\xbbC\xc5J\xd5E9\x8e \xcfy\xe2\xb9\xf7\x08m:\x83F2\xac\xe0z\xd1\x7f%\xcap\xe8\xb6TEl(P\"vY\xe4\xe3\x9dV\xa3P\x16\r\x92\xcf\x11\xc3;\x83\xbb\x82 G}\x11[\x86p6\xabi\x03t\xddY\x8efV\xc2\xb0W2%\xb8+\xb4\xc2?@\xde\xff\x87c\xec\xb8.\xe0\xcdW\x1a\xa8\x103D\xff\xa6@O\xbamJ\x04\xf2\xf0\x8a\xc9\x86\xe3",
    b"s\x02\xd2\xe38\xd7!\xef\x10[\x1eH\xd1\xe1\xcd\xb4\xae\xbf\x97\xe7\xb5\xbb A\xec\x9e_8*\x0c\xbfcOq^\xc7h\x8cb\xd7[\xec\xd1\xee\x92\xc4\x16\xeb\xef\x02\x1b9w\xd3\x0eIa\xb6T\xa0\x9c\xd2\"\xc1\x04M[\x0b\x12\xac%\x0c%y\x81\xdf\x0eL\xe6v\xe1\xd1\rf\x8ci\xe1c\xedj\x7f\xdf\xad\xee|G\x9bl\xcab\x11\x93\x9a\x02Oa=p{\xb4<\xd1\xd9\xef\xd0\x1a\xbc\xaa\xc4\xa7\xa2\t\xf58\xcd\xb3w>\xcbIC\x9d+\xf03q\xa3\xf7\xc7K\xed/\xdeD#[\x00)\x07Aiy\rt\x97\xa0\xa0\xb8\xbe\x81L\xca'\x12-K\xbe\xfc/]\xb4\x92\x12\x9d\x8d;\x9dK\xbe\xc7\x18\xad7\x82\xb7\xae\xfa\x8c`m\xd5M-M\xd6D)\xfb\x129H3\xd5U\xefA\x006\n\x1c\xb0\xa1\x05\xc0\xdeZ\xd5\xd43\xdexT\xfe*\x01\xf6\xaa+\x06I\xdc\xc3\xf6H\xad\xb4\xe5\xdd\xf0\xe7\x0fY\xc8r\x0e\xa8\xd4nY\x0b\xae0\xb3 D\x04\xb9b$\xc7\xacV\xf5\xba\xf77\xeb\xb0\x90\x07\x07\x18z\xc3\x92W\x12\x19\x8d\xf3W:\xa88#nQ\xb0\xc8\x14\x03\xf2\xed\xcb7\x0e\x05\xf9\xa3\x17}\xa1\xba\xd8\xd3\\/M9\r\x8bX(\xfaU\xce\xa1\n\x07\xe3D~\xa7\x9b\x1b\x01\x08\xa1Dl9\x01\xe2\x99*9\xd4\xb1\xdc\xc3\xaa\xd0\xb4S\xb7\xb8\xa20\x80\xadv=\xc5\xc1)\x06jd\xea%\xaa]\xb1\xb7\xb9\xa0\xe8e\x88M\xe9;AY\xed\xb8\x1f\xdb\nTx\xc6\x99\x96Z\xae\x87\x90U\xbe\xeb\x08\xcd\xec\xab\xf2:\xdc\xf6\xfb\xa1zP>qtw\xeb\x19\xcap \xf8r0\x19\xf2\xb1\x8cu\xbf\xa65\xefT9\xa7+\xcf8\xb1\xba\x06(\xc2\xab",
]
msgb = b""
for msg in SPARTNMESSAGES:
    msgb += msg
SPARTNBYTES = msgb


def progbar(i: int, lim: int, inc: int = 20):
    """
    Display progress bar on console.

    :param int i: iteration
    :param int lim: max iterations
    :param int inc: bar increments (20)
    """

    i = min(i, lim)
    pct = int(i * inc / lim)
    if not i % int(lim / inc):
        print(
            f"{int(pct*100/inc):02}% " + "\u2593" * pct + "\u2591" * (inc - pct),
            end="\r",
        )


def benchmark(**kwargs) -> float:
    """
    pyspartn Performance benchmark test.

    :param int cycles: (kwarg) number of test cycles (10,000)
    :returns: benchmark as transactions/second
    :rtype: float
    :raises: SPARTNStreamError
    """

    cyc = int(kwargs.get("cycles", 5000))
    txnc = len(SPARTNMESSAGES)
    txnt = txnc * cyc

    print(
        f"\nOperating system: {osver()}",
        f"\nPython version: {python_version()}",
        f"\npyspartn version: {spartnver}",
        f"\nTest cycles: {cyc:,}",
        f"\nTxn per cycle: {txnc:,}",
    )

    for decryptflag in (False, True):
        start = process_time_ns()
        print(f"\nBenchmark (decrypt={decryptflag}) test started at {start}")
        for i in range(cyc):
            progbar(i, cyc)
            stream = BytesIO(SPARTNBYTES)
            spr = SPARTNReader(stream, decode=decryptflag, key=KEY, basedate=BASEDATE)
            for _, _ in spr:
                pass
        end = process_time_ns()
        print(f"Benchmark test ended at {end}.")
        duration = end - start
        msglen = len(SPARTNBYTES) * cyc
        txs = round(txnt * 1e9 / duration, 2)
        kbs = round(msglen * 1e9 / duration / 2**10, 2)

        print(
            f"\n{txnt:,} messages processed in {duration/1e9:,.3f} seconds = {txs:,.2f} txns/second, {kbs:,.2f} kB/second.\n"
        )

    return txs, kbs


def main():
    """
    CLI Entry point.

    args as benchmark() method
    """

    benchmark(**dict(arg.split("=") for arg in argv[1:]))


if __name__ == "__main__":
    main()
