/*
Copyright (C) 1996-2001 Id Software, Inc.
Copyright (C) 2002-2009 John Fitzgibbons and others
Copyright (C) 2010-2014 QuakeSpasm developers

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#ifndef _GL_TEXMAN_H
#define _GL_TEXMAN_H

//gl_texmgr.h -- fitzquake's texture manager. manages opengl texture images

typedef enum
{
	TEXPREF_NONE			= 0x0000,
	TEXPREF_MIPMAP			= 0x0001,	// generate mipmaps
	// TEXPREF_NEAREST and TEXPREF_LINEAR aren't supposed to be ORed with TEX_MIPMAP
	TEXPREF_LINEAR			= 0x0002,	// force linear
	TEXPREF_NEAREST			= 0x0004,	// force nearest
	TEXPREF_ALPHA			= 0x0008,	// allow alpha
	TEXPREF_PAD				= 0x0010,	// allow padding
	TEXPREF_PERSIST			= 0x0020,	// never free
	TEXPREF_OVERWRITE		= 0x0040,	// overwrite existing same-name texture
	TEXPREF_NOPICMIP		= 0x0080,	// always load full-sized
	TEXPREF_FULLBRIGHT		= 0x0100,	// use fullbright mask palette
	TEXPREF_NOBRIGHT		= 0x0200,	// use nobright mask palette
	TEXPREF_CONCHARS		= 0x0400,	// use conchars palette
	TEXPREF_ARRAY			= 0x0800,	// array texture
	TEXPREF_CUBEMAP			= 0x1000,	// cubemap texture
	TEXPREF_BINDLESS		= 0x2000,	// enable bindless usage
	TEXPREF_ALPHABRIGHT		= 0x4000,	// use palette with lighting mask in alpha channel (0=fullbright, 1=lit)
	TEXPREF_CLAMP			= 0x8000,	// clamp UVs

	TEXPREF_HASALPHA		= (TEXPREF_ALPHA|TEXPREF_ALPHABRIGHT), // texture has alpha channel
} textureflags_t;

enum srcformat {SRC_INDEXED, SRC_LIGHTMAP, SRC_RGBA};

typedef uintptr_t src_offset_t;

typedef struct gltexture_s {
//managed by texture manager
	GLenum			target;
	GLuint			texnum;
	GLuint64		bindless_handle;
	struct gltexture_s	*next;
	qmodel_t		*owner;
//managed by image loading
	char			name[64];
	unsigned short		width; //size of image as it exists in opengl
	unsigned short		height; //size of image as it exists in opengl
	unsigned short		depth; //size of image as it exists in opengl
	unsigned char		compression; //compression rate (1 = uncompressed)
	textureflags_t		flags;
	char			source_file[MAX_QPATH]; //relative filepath to data source, or "" if source is in memory
	src_offset_t		source_offset; //byte offset into file, or memory address
	enum srcformat		source_format; //format of pixel data (indexed, lightmap, or rgba)
	unsigned int		source_width; //size of image in source data
	unsigned int		source_height; //size of image in source data
	unsigned short		source_crc; //generated by source data before modifications
	signed char			shirt; //0-13 shirt color, or -1 if never colormapped
	signed char			pants; //0-13 pants color, or -1 if never colormapped
//used for rendering
	int			visframe; //matches r_framecount if texture was bound this frame
} gltexture_t;

extern gltexture_t *notexture;
extern gltexture_t *nulltexture;
extern gltexture_t *whitetexture;
extern gltexture_t *greytexture;
extern gltexture_t *blacktexture;

extern unsigned int d_8to24table[256];
extern unsigned int d_8to24table_fbright[256];
extern unsigned int d_8to24table_nobright[256];
extern unsigned int d_8to24table_conchars[256];

extern GLint gl_max_texture_size;

typedef enum
{
	SOFTEMU_OFF,
	SOFTEMU_FINE,		// screen-space dither
	SOFTEMU_COARSE,		// world-space dither nearby, screen-space dither in the distance
	SOFTEMU_BANDED,		// no dithering

	SOFTEMU_NUMMODES,
} softemu_t;
extern softemu_t softemu;

typedef enum
{
	SOFTEMU_METRIC_NAIVE,
	SOFTEMU_METRIC_RIEMERSMA,
	SOFTEMU_METRIC_OKLAB,

	SOFTEMU_METRIC_COUNT,
	SOFTEMU_METRIC_INVALID = SOFTEMU_METRIC_COUNT,
} softemu_metric_t;

// TEXTURE FILTERING

typedef struct
{
	int	magfilter;
	int	minfilter;
	const char  *name;
	const char  *uiname;
} glmode_t;
#define NUM_GLMODES 6
extern const glmode_t glmodes[NUM_GLMODES];

typedef struct
{
	int		mode;
	float	anisotropy;
	float	lodbias;
} texfilter_t;
extern texfilter_t gl_texfilter;

// TEXTURE MANAGER

float TexMgr_FrameUsage (void);
gltexture_t *TexMgr_FindTexture (qmodel_t *owner, const char *name);
gltexture_t *TexMgr_NewTexture (void);
void TexMgr_FreeTexture (gltexture_t *kill);
void TexMgr_FreeTextures (unsigned int flags, unsigned int mask);
void TexMgr_FreeTexturesForOwner (qmodel_t *owner);
void TexMgr_NewGame (void);
void TexMgr_Init (void);
void TexMgr_DeleteTextureObjects (void);
void TexMgr_ApplySettings (void);
qboolean TexMgr_UsesFilterOverride (void);

// IMAGE LOADING
gltexture_t *TexMgr_LoadImage (qmodel_t *owner, const char *name, int width, int height, enum srcformat format,
			       byte *data, const char *source_file, src_offset_t source_offset, unsigned flags);
gltexture_t *TexMgr_LoadImageEx (qmodel_t *owner, const char *name, int width, int height, int depth, enum srcformat format,
			       byte *data, const char *source_file, src_offset_t source_offset, unsigned flags);
void TexMgr_ReloadImage (gltexture_t *glt, int shirt, int pants);
void TexMgr_ReloadImages (void);
void TexMgr_ReloadNobrightImages (void);

int TexMgr_Pad(int s);
int TexMgr_SafeTextureSize (int s);
int TexMgr_PadConditional (int s);

// TEXTURE BINDING & TEXTURE UNIT SWITCHING

qboolean GL_Bind (GLenum texunit, gltexture_t *texture);
qboolean GL_BindNative (GLenum texunit, GLenum type, GLuint handle);
void GL_BindTextures (GLuint first, GLsizei count, gltexture_t **textures);
void GL_DeleteNativeTexture (GLuint texnum);
void GL_ClearBindings (void);

#endif	/* _GL_TEXMAN_H */

