package io

import (
	"errors"
	"io"
	"strings"
	"unicode/utf8"

	"git.sr.ht/~charles/rq/util"
)

func init() {
	registerOutputHandler("md-table", func() OutputHandler { return &MarkdownTableOutputHandler{} })
}

// Declare conformance with OutputHandler interface.
var _ OutputHandler = &MarkdownTableOutputHandler{}

// MarkdownTableOutputHandler handles serializing JSON data.
type MarkdownTableOutputHandler struct {
	initialized bool
	pretty      bool
}

func (t *MarkdownTableOutputHandler) init() {
	if t.initialized {
		return
	}

	t.initialized = true
	t.pretty = true
}

// Name implements OutputHandler.Name().
func (t *MarkdownTableOutputHandler) Name() string {
	return "md-table"
}

// SetOption implements OutputHandler.SetOption().
func (t *MarkdownTableOutputHandler) SetOption(name string, value string) error {
	t.init()

	switch name {
	case "output.pretty":
		t.pretty = util.StringToValue(value).(bool)
	}

	return nil
}

// Format implements OutputHandler.Format()
func (t *MarkdownTableOutputHandler) Format(writer io.Writer, data interface{}) error {
	t.init()

	table, err := util.Tabularize(data, true)
	if err != nil {
		return err
	}

	if len(table) < 1 {
		return errors.New("failed to generate markdown table: no rows in output")
	}

	cols := 0
	for _, row := range table {
		if len(row) > cols {
			cols = len(row)
		}
	}

	headerrow := table[0]
	widths := make([]int, cols)
	for i, h := range headerrow {
		widths[i] = utf8.RuneCountInString(h)
	}

	// if pretty output is requested, scan the entire table for widths
	if t.pretty {
		for _, row := range table {
			for i, c := range row {
				l := utf8.RuneCountInString(c)
				if i >= len(widths) {
					// this can happen if the header
					// row is shorter than the rest of
					// the rows
					widths[i] = 8
				} else if l > widths[i] {
					widths[i] = l
				}
			}
		}
	}

	headerseps := make([]string, len(headerrow))
	for i, h := range headerrow {
		l := utf8.RuneCountInString(h)
		headerrow[i] = h + strings.Repeat(" ", widths[i]-l)
		headerseps[i] = strings.Repeat("-", widths[i]+2)
	}

	if _, err := writer.Write([]byte("| " + strings.Join(headerrow, " | ") + " |\n")); err != nil {
		return err
	}
	if _, err := writer.Write([]byte("|" + strings.Join(headerseps, "|") + "|\n")); err != nil {
		return err
	}

	for i, row := range table {
		if i == 0 {
			// skip headers
			continue
		}

		for i, c := range row {
			n := widths[i] - utf8.RuneCountInString(c)
			if n < 0 {
				n = 0
			}
			row[i] = c + strings.Repeat(" ", n)
		}
		if _, err := writer.Write([]byte("| " + strings.Join(row, " | ") + " |\n")); err != nil {
			return nil
		}
	}

	return nil
}
