package io

import (
	"encoding/base64"
	"encoding/json"
	"io"
)

func init() {
	registerInputHandler("base64", func() InputHandler { return &Base64InputHandler{} })
}

// Declare conformance with the InputHandler interface.
var _ InputHandler = &Base64InputHandler{}

// Base64InputHandler reads the "file data" from the 'base64.data'
// option, entirely ignoring the FilePath. This can be useful for
// testing.
//
// Note that because base64 data can contain '=' symbols for zero
// padding, it is recommended to use the JSON syntax for specifying the
// DataSpec.
//
// The base64 handler supports these input options:
//
//   - base64.data: the data to be returned when Parse() is called, in JSON
//     format.
type Base64InputHandler struct {
	data interface{}
}

// Name implements InputHandler.Name().
func (b *Base64InputHandler) Name() string {
	return "base64"
}

// Parse implements InputHandler.Parse().
//
// Unlike other input handlers, the base64 handler does not read from the given
// reader, and instead parses the data contained in the base64.data option as
// JSON and returns that.
func (b *Base64InputHandler) Parse(reader io.Reader) (interface{}, error) {
	return b.data, nil
}

func (b *Base64InputHandler) parse(data string) error {
	dataBytes, err := base64.StdEncoding.DecodeString(data)
	if err != nil {
		return err
	}

	err = json.Unmarshal(dataBytes, &(b.data))
	return err
}

// SetOption implements InputHandler.SetOption()
func (b *Base64InputHandler) SetOption(name string, value string) error {

	var err error = nil

	switch name {
	case "base64.data":
		err = b.parse(value)
	}

	return err

}
