package io

import (
	"encoding/json"
	"fmt"
	"path/filepath"
	"strings"
)

var fileExtensionAliases = map[string]string{
	"yml":   "yaml",
	"jsonl": "ndjson",
	"tf":    "hcl",
}

// DataSpec describes a single DataSpec after it has been parsed.
type DataSpec struct {
	Options  map[string]string
	Format   string
	FilePath string
}

type dataSpecJSON struct {
	Options  map[string]interface{} `json:"options"`
	Format   string                 `json:"format"`
	FilePath string                 `json:"file_path"`
}

// UnmarshalJSON implements json.Unmarshaler
//
// This is needed because we want to map bool/float values back to strings.
func (d *DataSpec) UnmarshalJSON(j []byte) error {
	dsj := &dataSpecJSON{}
	err := json.Unmarshal(j, dsj)
	if err != nil {
		panic("json.Unmarshaler promised the data would be valid JSON")
	}

	d.Format = dsj.Format
	d.FilePath = dsj.FilePath

	d.Options = make(map[string]string)
	for k, vi := range dsj.Options {
		if v, ok := vi.(string); ok {
			d.Options[k] = v
		} else {
			d.Options[k] = fmt.Sprintf("%v", vi)
		}
	}

	return nil
}

// ParseDataSpec parses a data specification string into a DataSpec object.
//
// The DataSpec format is used by rq for --data arguments, as well as for
// certain builtins that allow loading data. It intends to solve the following
// problems:
//
// - Allow specifying what format some data should loaded or stored in(e.g.
// CSV, JSON, YAML, etc.)
//
// - Allow specifying options pertinent to that format, such as the delimiter
// character for CSV.
//
// - Allow specifying where data should be loaded into Rego.
//
// - Be concise for users to specify when writing one-liners.
//
// The FilePath portion of the spec may or may not be an actual local file
// path, depending on the format used, or may be omitted entirely (e.g. when
// using a DataSpec to specify options for how something should be parsed). The
// RegoPath also may not be meaningful in all contexts, for example when
// specifying where data should be written to
//
// The pseudo-BNF for the DataSpec format follows:
//
//	DATASPEC -> FILEPATH |
//	            FORMAT ":" FILEPATH |
//	            OPTIONS ":" FILEPATH |
//	            FORMAT ":" OPTIONS ":" FILEPATH
//
//	OPTIONS ->  KEY "=" VALUE |
//	            KEY "=" VALUE ";" OPTIONS
//
//	KEY -> SEGMENT
//
//	VALUE -> SEGMENT
//
//	FORMAT -> SEGMENT
//
//	FILEPATH -> SEGMENT
//
//	SEGMENT -> /[^;:=]+/
//
// Note that this BNF does not ascribe any special value to any of the
// terminals like FORMAT or REGOPATH. A valid DataSpec may end up invalid when
// the data is actually used - validation is not the job of this parser.
//
// Leading and trailing whitespace on terminals is not considered significant
// and will be trimmed.
//
// Examples of valid DataSpecs:
//
// - /foo/bar
// - json:/foo/bar
// - csv:csv.headers=true;csv.separator=,;rego.path=foo:/foo/bar
// - csv.headers=true;csv.separator=,;rego.path=foo:/foo/bar
// - {"format": "csv", "file_path": "/foo/bar", "options": {"bool": true, "int": 7}}
//
// NOTE: if the first character of the DataPath is '{', and the last character
// of the DataPath is '}' (ignoring whitespace), then this function instead
// attempts to parse the DataPath as JSON and Unmarshal it into the DataSpec
// struct. This allows for arbitrary path and option strings, for situations
// where disallowed characters, leading or trailing whitespace, etc. are
// needed.
func ParseDataSpec(spec string) (*DataSpec, error) {
	spec = strings.TrimSpace(spec)
	s := &DataSpec{}

	// Case where the DataSpec is JSON.
	if (len(spec) > 1) && (spec[0] == '{') && (spec[len(spec)-1] == '}') {
		err := json.Unmarshal([]byte(spec), s)
		return s, err
	}

	split := strings.Split(spec, ":")
	if split == nil {
		return nil, fmt.Errorf("result of splitting DataSpec '%s' was nil", spec)
	}

	nSep := len(split) - 1

	// This is not a valid DataSpec.
	if nSep > 2 {
		return nil, fmt.Errorf("DataSpec '%s' is invalid because it contains more than 2 ':'", spec)
	}

	// Case where FORMAT is provided.
	if (nSep > 0) && (strings.Count(split[0], "=") == 0) {
		s.Format = split[0]

		// The split is now the same as if FORAMT was not provided,
		// that is either OPTIONS:FILEPATH, or just FILEPATH.
		split = split[1:]
		nSep--
	}

	// Case where OPTIONS are provided.
	options := ""
	if nSep == 1 {
		options = strings.TrimSpace(split[0])

		// The split is now the same as if we had just FILEPATH.
		split = split[1:]
	}

	// Whatever remains in split is necessarily the FILEPATH.
	s.FilePath = split[0]

	// If there are no options for us to parse, we simply return
	// immediately.
	s.Options = make(map[string]string)
	if len(options) == 0 {
		return s, nil
	}

	// Parse each individual option into s.Options
	optionsSplit := strings.Split(options, ";")
	for _, option := range optionsSplit {
		optionSplit := strings.Split(option, "=")

		if len(optionSplit) != 2 {
			return nil, fmt.Errorf("DataSpec '%s' is invalid because option '%s' does not have exactly one '='", spec, option)
		}

		key := strings.TrimSpace(optionSplit[0])
		value := strings.TrimSpace(optionSplit[1])
		s.Options[key] = value
	}

	return s, nil
}

// ResolveDefaults is used to apply default behaviors in a sensible and
// consistent way.
//
// If the FilePath is omitted, then the FilePath from the default is used
// instead.
//
// If the Format is omitted (empty string) from the DataSpec, then the
// default's Format will be used. If it is also the empty string, then the
// FilePath's extension will be used. If the FilePath is empty or does not
// contain a '.', then the Format will be set to "json".
//
// Any options which are specified in the default but not in the DataSpec will
// be applied as well.
func (d *DataSpec) ResolveDefaults(defaults *DataSpec) {
	if d.FilePath == "" {
		d.FilePath = defaults.FilePath
	}

	if d.Format == "" {
		d.Format = defaults.Format
	}

	if d.Format == "" {
		if strings.Contains(d.FilePath, ".") {
			e := strings.ToLower(filepath.Ext(d.FilePath)[1:])
			if a, ok := fileExtensionAliases[e]; ok {
				e = a
			}
			d.Format = e
		}
	}

	// For convenience of initializing DataSpecs from struct literals,
	// silently handle unallocated Option maps.
	if d.Options == nil {
		d.Options = make(map[string]string)
	}
	if defaults.Options == nil {
		return
	}

	for opt, val := range defaults.Options {
		if _, ok := d.Options[opt]; !ok {
			d.Options[opt] = val
		}
	}
}
