/* ============================================================
 *
 * This file is a part of digiKam project
 * https://www.digikam.org
 *
 * Date        : 2025-04-21
 * Description : digiKam generic GmicQt plugin supporting layers mode.
 *
 * SPDX-FileCopyrightText: 2025 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * ============================================================ */

#include "gmicqtsettingspage.h"

// Qt includes

#include <QIcon>
#include <QLabel>
#include <QUrl>
#include <QWidget>
#include <QApplication>
#include <QStyle>
#include <QComboBox>
#include <QGridLayout>
#include <QLineEdit>

// Digikam includes

#include "digikam_debug.h"
#include "dimg.h"

// Local includes

#include "gmicqtwizard.h"
#include "gmicqtcommon.h"

namespace DigikamGenericGmicQtPlugin
{

class Q_DECL_HIDDEN GmicQtSettingsPage::Private
{
public:

    explicit Private(QWizard* const dialog)
    {
        wizard = dynamic_cast<GmicQtWizard*>(dialog);

        if (wizard)
        {
            settings = wizard->settings();
            iface    = wizard->iface();
        }
    }

public:

    QWidget*        uploadWidget         = nullptr;
    QLineEdit*      fileTemplateLineEdit = nullptr;
    QComboBox*      imageFormat          = nullptr;

    GmicQtWizard*   wizard               = nullptr;
    DInfoInterface* iface                = nullptr;
    GmicQtSettings* settings             = nullptr;
};

GmicQtSettingsPage::GmicQtSettingsPage(QWizard* const dialog, const QString& title)
    : DWizardPage(dialog, title),
      d          (new Private(dialog))
{
    QWidget* const outPage         = new QWidget(this);
    const int spacing              = layoutSpacing();

    // --------------------

    QLabel* const outHeader        = new QLabel(outPage);
    outHeader->setWordWrap(true);
    outHeader->setText(tr("Configure the G'MIC filter settings below to store the output images.\n"
                          "Press the \"Next\" button to show the G'MIC-Qt tool and select the filters to apply on stack."));

    d->uploadWidget                = d->iface->uploadWidget(outPage);
    d->uploadWidget->setToolTip(tr("Select here the album where will be saved the output images generated by the G'MIC filter."));

    QLabel* const templateLbl      = new QLabel(tr("File Name Template:"), outPage);
    templateLbl->setWordWrap(false);

    d->fileTemplateLineEdit        = new QLineEdit(QLatin1String("gmic_output"), outPage);
    templateLbl->setBuddy(d->fileTemplateLineEdit);
    d->fileTemplateLineEdit->setToolTip(tr("Name of the G'MIC filter output files (without its extension)."));
    d->fileTemplateLineEdit->setWhatsThis(tr("\"File Name Template\": Set here the base name of the files that "
                                             "will be saved by the G'MIC filter. For example, if your template "
                                             "is \"gmic_output\" and if you chose a JPEG format, then the generated "
                                             "files will be saved with the name \"gmic_output_01.jpg\", "
                                             "\"gmic_output_02.jpg\", etc."));

    QLabel* const labelImageFormat = new QLabel(outPage);
    labelImageFormat->setWordWrap(false);
    labelImageFormat->setText(tr("Image Format:"));

    d->imageFormat                 = new QComboBox(outPage);
    d->imageFormat->setEditable(false);
    d->imageFormat->setToolTip(tr("Select your preferred G'MIC filter output image format."));
    d->imageFormat->addItem(QLatin1String("JPEG"), DImg::JPEG);
    d->imageFormat->addItem(QLatin1String("PNG"),  DImg::PNG);
    d->imageFormat->addItem(QLatin1String("TIFF"), DImg::TIFF);
    labelImageFormat->setBuddy(d->imageFormat);

    QGridLayout* const grid2 = new QGridLayout(outPage);
    grid2->addWidget(outHeader,               0, 0, 1, 3);
    grid2->addWidget(d->uploadWidget,         1, 0, 1, 3);
    grid2->addWidget(templateLbl,             2, 0, 1, 1);
    grid2->addWidget(d->fileTemplateLineEdit, 2, 1, 1, 2);
    grid2->addWidget(labelImageFormat,        3, 0, 1, 1);
    grid2->addWidget(d->imageFormat,          3, 2, 1, 1);
    grid2->setColumnStretch(0, 10);
    grid2->setContentsMargins(QMargins(spacing, spacing, spacing, spacing));
    grid2->setSpacing(spacing);

    if (QApplication::applicationName() == QLatin1String("digikam"))
    {
        grid2->setRowStretch(1, 10);
    }
    else
    {
        grid2->setRowStretch(4, 10);
    }

    setPageWidget(outPage);
    setLeftBottomPix(QIcon::fromTheme(QLatin1String("settings-configure")));

    connect(d->iface, SIGNAL(signalUploadUrlChanged()),
            this, SIGNAL(completeChanged()));

    connect(d->fileTemplateLineEdit, SIGNAL(textChanged(QString)),
            this, SIGNAL(completeChanged()));
}

GmicQtSettingsPage::~GmicQtSettingsPage()
{
    delete d;
}

void GmicQtSettingsPage::initializePage()
{
    d->fileTemplateLineEdit->setText(d->settings->templateFName);
    d->imageFormat->setCurrentIndex(d->imageFormat->findData(d->settings->format));
}

bool GmicQtSettingsPage::validatePage()
{
    d->settings->targetUrl     = d->iface->uploadUrl();
    d->settings->templateFName = d->fileTemplateLineEdit->text();
    d->settings->format        = d->imageFormat->currentData().toInt();

    return true;
}

bool GmicQtSettingsPage::isComplete() const
{
    bool b = (
                !d->iface->uploadUrl().toLocalFile().isEmpty() &&
                !d->fileTemplateLineEdit->text().isEmpty()
             );

    return b;
}

} // namespace DigikamGenericGmicQtPlugin

#include "moc_gmicqtsettingspage.cpp"
