use crate::render::json::JsonToStdout;
use crate::render::option::option_display;
use crate::render::spinner::spin_until_ready;
use crate::types::context::BergContext;
use crate::types::git::OwnerRepo;
use crate::{actions::GeneralArgs, types::output::OutputMode};
use clap::Parser;
use forgejo_api::structs::{IssueGetMilestonesListQuery, Milestone};

/// List all milestones in the current repository
#[derive(Parser, Debug)]
pub struct ListMilestonesArgs {
    /// Number of milestones to be displayed
    #[arg(short, long, value_name = "N", default_value_t = 5)]
    pub count: usize,
}

impl ListMilestonesArgs {
    pub async fn run(self, general_args: GeneralArgs) -> anyhow::Result<()> {
        let ctx = BergContext::new(self, general_args).await?;

        let OwnerRepo { repo, owner } = ctx.owner_repo()?;
        let (_, milestones_list) = spin_until_ready(ctx.client.issue_get_milestones_list(
            owner.as_str(),
            repo.as_str(),
            IssueGetMilestonesListQuery::default(),
        ))
        .await?;

        match general_args.output_mode {
            OutputMode::Pretty => {
                present_milestones_list(&ctx, milestones_list);
            }
            OutputMode::Json => milestones_list.print_json()?,
        }

        Ok(())
    }
}

fn present_milestones_list(ctx: &BergContext<ListMilestonesArgs>, milestones: Vec<Milestone>) {
    let milestones_empty = milestones.is_empty();

    let mut table = ctx.make_table();

    table
        .set_header(vec![format!(
            "Milestones{}",
            if milestones_empty {
                " (empty)"
            } else {
                Default::default()
            }
        )])
        .add_rows(
            milestones
                .into_iter()
                .map(|milestone| vec![option_display(&milestone.title)]),
        );

    println!("{table}", table = table.show());
}
