use anyhow::Context;
use url::Url;

use crate::actions::repo::parse_owner_and_repo;
use crate::actions::GeneralArgs;
use crate::render::spinner::spin_until_ready;
use crate::types::context::BergContext;

use clap::Parser;

/// Clone a repository
#[derive(Parser, Debug)]
pub struct RepoCloneArgs {
    /// Repository to be cloned
    #[arg(value_name = "OWNER/REPO")]
    pub owner_and_repo: String,
    /// Whether or not to clone via ssh
    #[arg(long, default_value_t = false)]
    pub use_ssh: bool,
    /// Where to clone into
    #[arg(value_name = "DESTINATION")]
    pub destination: Option<String>,
}

impl RepoCloneArgs {
    pub async fn run(self, general_args: GeneralArgs) -> anyhow::Result<()> {
        let _ = general_args;
        let ctx = BergContext::new(self, general_args).await?;

        let (owner, repo) = parse_owner_and_repo(ctx.args.owner_and_repo.as_str())?;
        let url = if ctx.args.use_ssh {
            spin_until_ready(get_ssh_url(&ctx, owner.as_str(), repo.as_str())).await?
        } else {
            ctx.config
                .url()?
                .join(format!("/{owner}/{repo}.git").as_str())?
        };

        // TODO: Add back ***opt in*** ask confirm to clone

        start_clone_repo(url, ctx.args.destination.unwrap_or(repo))?;
        Ok(())
    }
}

async fn get_ssh_url(
    ctx: &BergContext<RepoCloneArgs>,
    ownername: &str,
    reponame: &str,
) -> anyhow::Result<Url> {
    ctx.client
        .repo_get(ownername, reponame)
        .await
        .map_err(anyhow::Error::from)
        .inspect(|x| tracing::debug!("{x:#?}"))
        .and_then(|repo| repo.ssh_url.context("No SSH url on repo"))
        .and_then(|url| {
            Url::parse(url.as_str())
                .with_context(|| format!("Url wasn't a valid SSH Url: {url}", url = url.as_str()))
        })
        .context("User doesn't own the repo that was specified or at least doesn't have access'.")
}

fn start_clone_repo(ssh_url: Url, destination: String) -> anyhow::Result<()> {
    let mut cmd = std::process::Command::new("git");
    cmd.arg("clone")
        .arg(ssh_url.as_str())
        .arg(destination.as_str());
    tracing::debug!("cmd: {cmd:?}");
    let mut child = cmd.stdout(std::process::Stdio::inherit()).spawn()?;
    child.wait()?;
    Ok(())
}
