use forgejo_api::structs::{IssueListLabelsQuery, Label};

use crate::actions::GeneralArgs;
use crate::render::option::option_display;
use crate::render::spinner::spin_until_ready;

use crate::types::context::BergContext;
use crate::types::git::OwnerRepo;

use clap::Parser;

/// List all labels in the current repository
#[derive(Parser, Debug)]
pub struct ListLabelsArgs {
    /// Number of labels to be displayed
    #[arg(short, long, value_name = "N", default_value_t = 5)]
    pub count: usize,
}

impl ListLabelsArgs {
    pub async fn run(self, general_args: GeneralArgs) -> anyhow::Result<()> {
        let _ = general_args;
        let ctx = BergContext::new(self, general_args).await?;

        let OwnerRepo { repo, owner } = ctx.owner_repo()?;
        let labels_list = spin_until_ready(ctx.client.issue_list_labels(
            owner.as_str(),
            repo.as_str(),
            IssueListLabelsQuery::default(),
        ))
        .await?;

        present_labels_list(&ctx, labels_list);

        Ok(())
    }
}

fn present_labels_list(ctx: &BergContext<ListLabelsArgs>, labels: Vec<Label>) {
    let labels_empty = labels.is_empty();

    let mut table = ctx.make_table();

    table
        .set_header(vec![format!(
            "Labels{}",
            labels_empty.then_some(" (empty)").unwrap_or_default()
        )])
        .add_rows(
            labels
                .into_iter()
                .map(|label| vec![option_display(&label.name)]),
        );

    println!("{table}", table = table.show());
}
