/////////////////////////////////////////////////////////////////////////////
//  P C 6 0 0 1 V
//  Copyright 1999 Yumitaro
/////////////////////////////////////////////////////////////////////////////
#include <fstream>

#include "pc6001v.h"

#include "error.h"
#include "memblk.h"
#include "osd.h"




// ダミーメモリセル
MemCell EmptyCell( 0xff, true );


//--------------------------------------------------------------
// メモリセルクラス(最小単位)
//--------------------------------------------------------------

/////////////////////////////////////////////////////////////////////////////
// Constructor
/////////////////////////////////////////////////////////////////////////////
MemCell::MemCell( BYTE idata, bool wp ) : WPt( wp )
{
	Data.assign( PAGESIZE, idata );
}


/////////////////////////////////////////////////////////////////////////////
// Destructor
/////////////////////////////////////////////////////////////////////////////
MemCell::~MemCell( void )
{
}


/////////////////////////////////////////////////////////////////////////////
// リサイズ
/////////////////////////////////////////////////////////////////////////////
void MemCell::Resize( size_t size, BYTE idata )
{
	Data.assign( size, idata );
}


/////////////////////////////////////////////////////////////////////////////
// ROMデータをファイルから読込み
//
// 引数:	fs		ファイルストリームへの参照
// 返値:	なし
/////////////////////////////////////////////////////////////////////////////
void MemCell::SetData( std::fstream& fs )
{
	size_t sz = fs.read( (char*)Data.data(), PAGESIZE ).gcount();
	if( sz < PAGESIZE ){	// 端数だったらセルサイズ縮小
		Data.resize( sz );
	}
	WPt = true;	// データ読込んだらROM扱い
}


/////////////////////////////////////////////////////////////////////////////
// サイズ取得
//
// 引数:	なし
// 返値:	size_t	データサイズ(bytes)
/////////////////////////////////////////////////////////////////////////////
size_t MemCell::Size( void ) const
{
	return Data.size();
}


/////////////////////////////////////////////////////////////////////////////
// メモリリード
//
// 引数:	addr	アドレス
// 返値:	BYTE	データ
/////////////////////////////////////////////////////////////////////////////
BYTE MemCell::Read( WORD addr ) const
{
	try{
		return Data.at( addr & PAGEMASK );
	}
	catch( std::out_of_range& ){}
	
	return 0xff;
}


/////////////////////////////////////////////////////////////////////////////
// メモリライト
//
// 引数:	addr	アドレス
//			data	データ
// 返値:	なし
/////////////////////////////////////////////////////////////////////////////
void MemCell::Write( WORD addr, BYTE data )
{
	if( WPt ){
		return;
	}
	
	try{
		Data.at( addr & PAGEMASK ) = data;
	}
	catch( std::out_of_range& ){}
}




//--------------------------------------------------------------
// メモリセル集合体クラス(ROM/RAMチップ相当)
//--------------------------------------------------------------

/////////////////////////////////////////////////////////////////////////////
// Constructor
/////////////////////////////////////////////////////////////////////////////
MemCells::MemCells(  size_t cnum, BYTE idata, bool wp  )
{
	Cells.assign( cnum, MemCell( idata, wp ) );
}


/////////////////////////////////////////////////////////////////////////////
// Destructor
/////////////////////////////////////////////////////////////////////////////
MemCells::~MemCells( void )
{
}


/////////////////////////////////////////////////////////////////////////////
// リサイズ
//
// 引数:	size	データサイズ
//			idata	初期値
// 返値:	なし
/////////////////////////////////////////////////////////////////////////////
void MemCells::Resize( size_t size, BYTE idata )
{
	// RAMとして再確保する
	Cells.assign( (size + MemCell::PAGEBITS) >> MemCell::PAGEBITS, MemCell( idata ) );
	// 端数がある場合は末尾のメモリセルをリサイズ
	size_t sz = size & MemCell::PAGEBITS;
	if( sz ){
		Cells.rbegin()->Resize( sz );
	}
}


/////////////////////////////////////////////////////////////////////////////
// ROMデータをファイルから読込み
//
// 引数:	filepath	ファイルパス名への参照
// 返値:	bool		true:成功 false:失敗
/////////////////////////////////////////////////////////////////////////////
bool MemCells::SetData( const P6VPATH& filepath )
{
	try{
		// ファイルサイズに合わせてメモリセル再設定(端数考慮&2のべき乗)
		size_t szc = ((OSD_GetFileSize( filepath ) + MemCell::PAGEMASK) >> MemCell::PAGEBITS);
		if( szc == 0 ){
			throw Error::NoRom;
		}
		
		size_t szn = 1;
		while( szn < szc ){ szn <<= 1; }
		Cells.resize( szn );
		
		std::fstream fs;
		if( !OSD_FSopen( fs, filepath, std::ios_base::in | std::ios_base::binary ) ){
			throw Error::NoRom;
		}
		
		for( auto &mc : Cells ){
			mc.SetData( fs );
		}
		fs.close();
	}
	// 例外発生
	catch( Error::Errno i ){
		Error::SetError( i );
		return false;
	}
	
	return true;
}


/////////////////////////////////////////////////////////////////////////////
// サイズ(メモリセル数)取得
//
// 引数:	なし
// 返値:	size_t	メモリセル数
/////////////////////////////////////////////////////////////////////////////
size_t MemCells::Size( void ) const
{
	return Cells.size();
}


/////////////////////////////////////////////////////////////////////////////
// メモリリード
//
// 引数:	addr	アドレス
// 返値:	BYTE	データ
/////////////////////////////////////////////////////////////////////////////
BYTE MemCells::Read( DWORD addr ) const
{
	try{
		return Cells.at( (addr >> MemCell::PAGEBITS) & (Cells.size() - 1) ).Read( addr & 0xffff );
	}
	catch( std::out_of_range& ){}
	
	return 0xff;
}


/////////////////////////////////////////////////////////////////////////////
// メモリライト
//
// 引数:	addr	アドレス
//			data	データ
// 返値:	なし
/////////////////////////////////////////////////////////////////////////////
void MemCells::Write( DWORD addr, BYTE data )
{
	try{
		Cells.at( (addr >> MemCell::PAGEBITS) & (Cells.size() - 1) ).Write( addr & 0xffff, data );
	}
	catch( std::out_of_range& ){}
}


/////////////////////////////////////////////////////////////////////////////
// メモリセル取得 - operator ()
//
// 引数:	num			インデックス(0-)
// 返値:	MemCell&	メモリセルへの参照
/////////////////////////////////////////////////////////////////////////////
MemCell& MemCells::operator()( const int num )
{
	try{
		return Cells.at( num );
	}
	catch( std::out_of_range& ){}
	
	return EmptyCell;
}




//--------------------------------------------------------------
// メモリブロッククラス
//--------------------------------------------------------------

/////////////////////////////////////////////////////////////////////////////
// Constructor
/////////////////////////////////////////////////////////////////////////////
MemBlock::MemBlock( void ) : Name( "" ), PMem( &EmptyCell ), FName( nullptr ), FRead( nullptr ), FWrite( nullptr ), Wait( 0 )
{
}


/////////////////////////////////////////////////////////////////////////////
// Destructor
/////////////////////////////////////////////////////////////////////////////
MemBlock::~MemBlock( void )
{
}


/////////////////////////////////////////////////////////////////////////////
// メモリ割当て
//
// 引数:	name	メモリブロック名への参照
//			data	メモリセルへの参照
//			wait	アクセスウェイト(-1:変更しない)
// 返値:	なし
/////////////////////////////////////////////////////////////////////////////
void MemBlock::SetMemory( const std::string& name, MemCell& data, int wait )
{
	Name   = name;
	PMem   = &data;
	FName  = nullptr;
	FRead  = nullptr;
	FWrite = nullptr;
	Wait   = wait < 0 ? Wait : wait;
}


/////////////////////////////////////////////////////////////////////////////
// 関数割当て
//
// 引数:	name	メモリブロック名への参照
//			rd		読込み関数ポインタ(bind)
//			wr		書込み関数ポインタ(bind)
//			wait	アクセスウェイト(-1:変更しない)
// 返値:	なし
/////////////////////////////////////////////////////////////////////////////
void MemBlock::SetFunc( const std::string& name, RFunc rd, WFunc wr, int wait )
{
	Name   = name;
	PMem   = &EmptyCell;
	FName  = nullptr;
	FRead  = rd;
	FWrite = wr;
	Wait   = wait < 0 ? Wait : wait;
}

// 引数:	name	メモリブロック名取得関数ポインタ(bind)
//			rd		読込み関数ポインタ(bind)
//			wr		書込み関数ポインタ(bind)
//			wait	アクセスウェイト(-1:変更しない)
// 返値:	なし
void MemBlock::SetFunc( NFunc nm, RFunc rd, WFunc wr, int wait )
{
	Name   = "";
	PMem   = &EmptyCell;
	FName  = nm;
	FRead  = rd;
	FWrite = wr;
	Wait   = wait < 0 ? Wait : wait;
}


/////////////////////////////////////////////////////////////////////////////
// メモリブロック名取得
//
// 引数:	addr			割当先のアドレス(0000-FFFF)
//			type			true:Read false:Write
// 返値:	std::string& 	メモリブロック名への参照
/////////////////////////////////////////////////////////////////////////////
const std::string& MemBlock::GetName( WORD addr, bool type ) const
{
	return FName ? FName( addr, type ) : Name;
}


/////////////////////////////////////////////////////////////////////////////
// アクセスウェイト設定
//
// 引数:	wait	アクセスウェイト
// 返値:	なし
/////////////////////////////////////////////////////////////////////////////
void MemBlock::SetWait( int wait )
{
	Wait = wait & 0xff;
}


/////////////////////////////////////////////////////////////////////////////
// アクセスウェイト取得
//
// 引数:	なし
// 返値:	int		ウェイト
/////////////////////////////////////////////////////////////////////////////
int MemBlock::GetWait( void ) const
{
	return Wait;
}


/////////////////////////////////////////////////////////////////////////////
// メモリリード
//
// 引数:	addr	アドレス
//			wcnt	ウェイトカウンタへのポインタ
// 返値:	BYTE	データ
/////////////////////////////////////////////////////////////////////////////
BYTE MemBlock::Read( WORD addr, int* wcnt ) const
{
	if( FRead ){
		return FRead( PMem, addr, wcnt );
	}else if( PMem->Size() ){
		if( wcnt ){ *wcnt += Wait; }
		return PMem->Read( addr );
	}
	
	return 0xff;
}


/////////////////////////////////////////////////////////////////////////////
// メモリライト
//
// 引数:	addr	アドレス
//			data	データ
//			wcnt	ウェイトカウンタへのポインタ
// 返値:	なし
/////////////////////////////////////////////////////////////////////////////
void MemBlock::Write( WORD addr, BYTE data, int* wcnt ) const
{
	if( FWrite ){
		FWrite( PMem, addr, data, wcnt );
	}else if( PMem->Size() ){
		if( wcnt ){	*wcnt += Wait; }
		PMem->Write( addr, data );
	}
}

