/**********************"*****************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtestmacro.h"
#include <qsqldatabase.h>
#include <qthread.h>
#include <quuid.h>
#include <unistd.h>

void dumpConnections()
{
    const auto conNames = QSqlDatabase::connectionNames();
    for (const auto& conName : conNames) {
        SKGTRACE << "### " << conName << " ###" << Qt::endl;
        auto con = QSqlDatabase::database(conName, false);
        SKGTRACE << "    connectionName=" << con.connectionName() << Qt::endl;
        SKGTRACE << "    connectOptions=" << con.connectOptions() << Qt::endl;
        SKGTRACE << "    isOpen=" << (con.isOpen() ? "Y" : "N") << Qt::endl;
    }
}

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc)
    Q_UNUSED(argv)

    // Init test
    SKGINITTEST(true)

    // test class SKGDocument / PARAMETERS
    SKGDocument document1;
    SKGTESTERROR(QLatin1String("PARAM:initialize"), document1.initialize(), true)
    SKGTESTERROR(QLatin1String("PARAM:close"), document1.close(), true)
    SKGTESTERROR(QLatin1String("PARAM:initialize"), document1.initialize(), true)
    SKGTESTERROR(QLatin1String("PARAM:beginTransaction"), document1.beginTransaction(QLatin1String("t1")), true)
    SKGTESTERROR(QLatin1String("PARAM:setParameter"), document1.setParameter(QLatin1String("ATT1"), QLatin1String("VAL1")), true)
    SKGTESTERROR(QLatin1String("PARAM:endTransaction"), document1.endTransaction(true), true)
    SKGTEST(QLatin1String("PARAM:getCachedValue"), document1.getCachedValue(QLatin1String("NOTFOUND")), QLatin1String(""))

    SKGDocument document2;
    SKGTESTERROR(QLatin1String("PARAM:initialize"), document2.initialize(), true)
    SKGTESTERROR(QLatin1String("PARAM:beginTransaction"), document2.beginTransaction(QLatin1String("t2")), true)
    SKGTESTERROR(QLatin1String("PARAM:setParameter"), document2.setParameter(QLatin1String("ATT2"), QLatin1String("VAL2")), true)
    SKGTESTERROR(QLatin1String("PARAM:setParameter"), document2.setParameter(QLatin1String("ATT3"), QLatin1String("dates.txt"), SKGTest::getTestPath(QLatin1String("IN")) % "/dates.txt"), true)
    SKGTESTERROR(QLatin1String("PARAM:endTransaction"), document2.endTransaction(true), true)


    SKGTEST(QLatin1String("PARAM:getParameter"), document1.getParameter(QLatin1String("ATT1")), QLatin1String("VAL1"))
    SKGTEST(QLatin1String("PARAM:getParameter"), document2.getParameter(QLatin1String("ATT2")), QLatin1String("VAL2"))

    SKGTEST(QLatin1String("PARAM:getFileExtension"), document2.getFileExtension(), QLatin1String("skgc"))

    document1.formatPercentage(1.1, true);
    document1.formatPercentage(1.1, false);
    SKGTEST(QLatin1String("PARAM:getRealAttribute"), document2.getRealAttribute(QLatin1String("t_ATT")), QLatin1String(""))
    SKGTEST(QLatin1String("PARAM:getRealAttribute"), document2.getRealAttribute(QLatin1String("t_att")), QLatin1String("t_att"))

    document1.getDatabaseIdentifier();
    document1.getParameters(QLatin1String("document"), QLatin1String("t_name like 'ATT%'"));

    // Special SQL command
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSqliteOrder(QLatin1String("SELECT * FROM (SELECT CAPITALIZE(LOWER(UPPER(WORD('Abc Def', 2)))) AS V) WHERE REGEXP('D.*', V) AND WILDCARD('D*', V)")), true)
    QString result;
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT TOWEEKYEAR('2019-12-31')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-TOWEEKYEAR"), result, QLatin1String("2020-W01"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT TODATE('07162013', 'MMDDYYYY')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-TODATE"), result, QLatin1String("2013-07-16"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT TOFORMATTEDDATE('2013-07-16', 'dd-MM-yyyy')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-TODATE"), result, QLatin1String("16-07-2013"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT TOFORMATTEDDATE('2013-07-16', 'd M yy')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-TODATE"), result, QLatin1String("16 7 13"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT TODATE('ABCDEFGHIJ', 'MMDDYYYY')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-TODATE"), result, SKGServices::dateToSqlString(QDate::currentDate()))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT WORD('Abc Def Ghi', 0)"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-WORD"), result, QLatin1String("Abc"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT WORD('Abc Def Ghi', 1)"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-WORD"), result, QLatin1String("Abc"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT WORD('Abc Def Ghi', 2)"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-WORD"), result, QLatin1String("Def"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT WORD('Abc Def Ghi', 3)"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-WORD"), result, QLatin1String("Ghi"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT WORD('Abc Def Ghi', 99)"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-WORD"), result, QLatin1String("Ghi"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT WORD('Abc Def Ghi', -99)"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-WORD"), result, QLatin1String("Abc"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT WORD('Abc Def Ghi', -3)"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-WORD"), result, QLatin1String("Abc"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT WORD('Abc Def Ghi', -2)"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-WORD"), result, QLatin1String("Def"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT WORD('Abc Def Ghi', -1)"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-WORD"), result, QLatin1String("Ghi"))

    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT WORD('   Abc    Def   Ghi ', 1)"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-WORD"), result, QLatin1String("Abc"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT WORD('   Abc    Def   Ghi ', 2)"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-WORD"), result, QLatin1String("Def"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT WORD('   Abc    Def   Ghi ', 3)"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-WORD"), result, QLatin1String("Ghi"))

    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT WORD('N:1234', 1)"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-WORD"), result, QLatin1String("N"))

    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT WORD('N:1234', 2)"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-WORD"), result, QLatin1String("1234"))

    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT REGEXPCAPTURE('(.*) CARTE 1234.*', 'MyShopName CARTE 12345678 PAIEME', 0)"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-WORD"), result, QLatin1String("MyShopName CARTE 12345678 PAIEME"))

    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT REGEXPCAPTURE('(.*) CARTE 1234.*', 'MyShopName CARTE 12345678 PAIEME', 1)"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-WORD"), result, QLatin1String("MyShopName"))

    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT REGEXPCAPTURE('(.*) CARTE 1234.*', 'MyShopName CARTE 12345678 PAIEME', 12)"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-WORD"), result, QLatin1String(""))

    QMap<QString, QVariant> map;
    map[QLatin1String(":2")] = "20";
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSqliteOrder(QLatin1String("SELECT WORD('Abc Def', :2)"), map, nullptr), true)

    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT TOCURRENCY(1234, 'F')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-1234 F"), result.remove(' '), QLatin1String("1234.00F"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT TOCURRENCY(-1234, (SELECT 'F'))"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-1234 F"), result.remove(' '), QLatin1String("-1234.00F"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT PERIOD((SELECT '2013-03-05'), (SELECT 'D'))"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-PERIOD D"), result, QLatin1String("2013-03-05"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT PERIOD('2013-03-05', 'W')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-PERIOD W"), result, QLatin1String("2013-W10"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT PERIOD('2013-03-05', 'M')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-PERIOD M"), result, QLatin1String("2013-03"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT PERIOD('2013-03-05', 'Q')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-PERIOD Q"), result, QLatin1String("2013-Q1"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT PERIOD('2013-03-05', 'S')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-PERIOD S"), result, QLatin1String("2013-S1"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT PERIOD('2013-03-05', 'Y')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-PERIOD Y"), result, QLatin1String("2013"))

    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT PERIOD('2014-07-16', 'D')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-PERIOD D"), result, QLatin1String("2014-07-16"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT PERIOD('2014-07-16', 'W')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-PERIOD W"), result, QLatin1String("2014-W29"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT PERIOD('2014-07-16', 'M')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-PERIOD M"), result, QLatin1String("2014-07"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT PERIOD('2014-07-16', 'Q')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-PERIOD Q"), result, QLatin1String("2014-Q3"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT PERIOD('2014-07-16', 'S')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-PERIOD S"), result, QLatin1String("2014-S2"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT PERIOD('2014-07-16', 'Y')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-PERIOD Y"), result, QLatin1String("2014"))

    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT NEXT('12345')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-NEXT"), result, QLatin1String("12346"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT NEXT('9')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-NEXT"), result, QLatin1String("10"))
    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT NEXT('ABC')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-NEXT"), result, QLatin1String(""))

    SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT XOR('HELLO WORLD!', 'KEY')"), result), true)
    SKGTEST(QLatin1String("PARAM:executeSqliteOrder-XOR"), result, QLatin1String("# 030015070a791c0a0b070178"))
    for (int i = 1; i < 100; ++i) {
        auto string = QUuid::createUuid().toString();
        SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT XOR(XOR('") + string + "', 'KEY'), 'KEY')", result), true)
        SKGTEST(QLatin1String("PARAM:executeSqliteOrder-XOR"), result, string)
    }
    for (double i = -1200.53; i < 5023.25; i = i + 125.54) {
        auto string = SKGServices::doubleToString(i);
        SKGTESTERROR(QLatin1String("PARAM:executeSqliteOrder"), document1.executeSingleSelectSqliteOrder(QLatin1String("SELECT XORD(XORD(") + string + ", 'KEY'), 'KEY')", result), true)
        SKGTEST(QLatin1String("PARAM:executeSqliteOrder-XORD"), result, string)
    }

    SKGTRACE << "####### Before concurrent calls" << Qt::endl;
    dumpConnections();
    int nb = 5;
    {
        SKGTRACE << ">> executeSelectSqliteOrder same order" << Qt::endl;
        QString output;
        double elapse = SKGServices::getMicroTime();
        for (int i = 0; i < nb; ++i) {
            SKGStringListList oResult;
            IFOK(document1.executeSelectSqliteOrder(QLatin1String("SELECT SLEEP(1)"), oResult)) {
                output = output + SKGServices::intToString(oResult.count());
            }
        }
        double time = SKGServices::getMicroTime() - elapse;
        SKGTRACE << nb << " x executeSelectSqliteOrder:" << output << "     " << time << " ms" << Qt::endl;
        SKGTEST(QLatin1String("PARAM:executeSelectSqliteOrder-PERFO >=1000"), static_cast<unsigned int>(time >= 1000), static_cast<unsigned int>(true))
        SKGTEST(QLatin1String("PARAM:executeSelectSqliteOrder-PERFO <2000"), static_cast<unsigned int>(time < 2000), static_cast<unsigned int>(true))
        SKGTEST(QLatin1String("PARAM:executeSelectSqliteOrder"), output, QLatin1String("22222"))
    }

    {
        SKGTRACE << ">> concurrentExecuteSelectSqliteOrder same order" << Qt::endl;
        QString output;
        double elapse = SKGServices::getMicroTime();
        for (int i = 0; i < nb; ++i) {
            document1.concurrentExecuteSelectSqliteOrder(QLatin1String("SELECT SLEEP(1), 2"),
            [ &output ](const SKGStringListList & iResult) {
                output = output + SKGServices::intToString(iResult.count());
            });
        }
        double time = SKGServices::getMicroTime() - elapse;
        SKGTRACE << nb << " x concurrentExecuteSelectSqliteOrder:" << output << "     " << (SKGServices::getMicroTime() - elapse) << " ms" << Qt::endl;
        qApp->processEvents(QEventLoop::AllEvents, 500);
        for (int i = 1; i < 100; ++i) {
            QThread::msleep(100);
            qApp->processEvents(QEventLoop::AllEvents, 500);
            time = SKGServices::getMicroTime() - elapse;
            if (output == QLatin1String("22222")) {
                SKGTEST(QLatin1String("PARAM:executeSelectSqliteOrder-PERFO >=1000"), static_cast<unsigned int>(time >= 1000), static_cast<unsigned int>(true))
                SKGTEST(QLatin1String("PARAM:executeSelectSqliteOrder-PERFO <3500"), static_cast<unsigned int>(time < 3500), static_cast<unsigned int>(true))
                break;
            }
        }

        SKGTRACE << nb << " x concurrentExecuteSelectSqliteOrder:" << output << "     " << time << " ms" << Qt::endl;
        SKGTEST(QLatin1String("PARAM:concurrentExecuteSelectSqliteOrder"), output, QLatin1String("22222"))
    }

    {
        SKGTRACE << ">> executeSelectSqliteOrder different orders" << Qt::endl;
        QString output;
        double elapse = SKGServices::getMicroTime();
        for (int i = 0; i < nb; ++i) {
            SKGStringListList oResult;
            IFOK(document1.executeSelectSqliteOrder(QLatin1String("SELECT SLEEP(1), ") + SKGServices::intToString(1000 + i), oResult)) {
                output = output + SKGServices::intToString(oResult.count());
            }
        }
        double time = SKGServices::getMicroTime() - elapse;
        SKGTRACE << nb << " x executeSelectSqliteOrder:" << output << "     " << time << " ms" << Qt::endl;
        SKGTEST(QLatin1String("PARAM:executeSelectSqliteOrder-PERFO >5000"), static_cast<unsigned int>(time > 5000), static_cast<unsigned int>(true))
        SKGTEST(QLatin1String("PARAM:executeSelectSqliteOrder-PERFO <6000"), static_cast<unsigned int>(time < 6000), static_cast<unsigned int>(true))
        SKGTEST(QLatin1String("PARAM:executeSelectSqliteOrder"), output, QLatin1String("22222"))
    }

    {
        SKGTRACE << ">> concurrentExecuteSelectSqliteOrder different orders" << Qt::endl;
        QString output;
        double elapse = SKGServices::getMicroTime();
        for (int i = 0; i < nb; ++i) {
            document1.concurrentExecuteSelectSqliteOrder(QLatin1String("SELECT SLEEP(1), ") + SKGServices::intToString(2000 + i),
            [ &output ](const SKGStringListList & iResult) {
                output = output + SKGServices::intToString(iResult.count());
            });
        }
        double time = SKGServices::getMicroTime() - elapse;
        SKGTRACE << nb << " x concurrentExecuteSelectSqliteOrder:" << output << "     " << (SKGServices::getMicroTime() - elapse) << " ms" << Qt::endl;
        qApp->processEvents(QEventLoop::AllEvents, 500);
        for (int i = 1; i < 100; ++i) {
            QThread::msleep(100);
            qApp->processEvents(QEventLoop::AllEvents, 500);
            time = SKGServices::getMicroTime() - elapse;
            if (output == QLatin1String("22222")) {
                SKGTEST(QLatin1String("PARAM:executeSelectSqliteOrder-PERFO >=1000"), static_cast<unsigned int>(time >= 1000), static_cast<unsigned int>(true))
                SKGTEST(QLatin1String("PARAM:executeSelectSqliteOrder-PERFO <3500"), static_cast<unsigned int>(time < 3500), static_cast<unsigned int>(true))
                break;
            }
        }

        SKGTRACE << nb << " x concurrentExecuteSelectSqliteOrder:" << output << "     " << time << " ms" << Qt::endl;
        SKGTEST(QLatin1String("PARAM:concurrentExecuteSelectSqliteOrder"), output, QLatin1String("22222"))
    }
    SKGTRACE << "####### Before close" << Qt::endl;
    dumpConnections();
    SKGTESTERROR(QLatin1String("PARAM:close"), document1.close(), true)
    SKGTRACE << "####### Before initialize" << Qt::endl;
    dumpConnections();
    SKGTESTERROR(QLatin1String("PARAM:initialize"), document1.initialize(), true)
    SKGTRACE << "####### Before concurrent calls" << Qt::endl;
    dumpConnections();

    {
        SKGTRACE << ">> concurrentExecuteSelectSqliteOrder different orders (in other thread)" << Qt::endl;
        QString output;
        double elapse = SKGServices::getMicroTime();
        for (int i = 0; i < nb; ++i) {
            document1.concurrentExecuteSelectSqliteOrder(QLatin1String("SELECT SLEEP(1), ") + SKGServices::intToString(3000 + i),
            [ &output ](const SKGStringListList & iResult) {
                QMutex mutex;
                mutex.lock();
                output = output + SKGServices::intToString(iResult.count());
                mutex.unlock();
            }, false);
        }
        double time = SKGServices::getMicroTime() - elapse;
        SKGTRACE << nb << " x concurrentExecuteSelectSqliteOrder:" << output << "     " << (SKGServices::getMicroTime() - elapse) << " ms" << Qt::endl;
        for (int i = 1; i < 100; ++i) {
            QThread::msleep(100);
            time = SKGServices::getMicroTime() - elapse;
            if (output == QLatin1String("22222")) {
                SKGTEST(QLatin1String("PARAM:executeSelectSqliteOrder-PERFO >=1000"), static_cast<unsigned int>(time >= 1000), static_cast<unsigned int>(true))
                SKGTEST(QLatin1String("PARAM:executeSelectSqliteOrder-PERFO <3500"), static_cast<unsigned int>(time < 3500), static_cast<unsigned int>(true))
                break;
            }
        }

        SKGTRACE << nb << " x concurrentExecuteSelectSqliteOrder:" << output << "     " << time << " ms" << Qt::endl;
        SKGTEST(QLatin1String("PARAM:concurrentExecuteSelectSqliteOrder"), output, QLatin1String("22222"))
    }
    SKGTRACE << "####### End" << Qt::endl;
    dumpConnections();


    // End test
    SKGENDTEST()
}
