/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for KMY import / export.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgimportpluginkmy.h"

#include <kcompressiondevice.h>
#include <kpluginfactory.h>

#include <qdom.h>
#include <qmath.h>

#include "skgbankincludes.h"
#include "skgimportexportmanager.h"
#include "skgobjectbase.h"
#include "skgpayeeobject.h"
#include "skgservices.h"
#include "skgtraces.h"
#include <utility>

/**
 * Transactions treated.
 */
QSet<QString> SKGImportPluginKmy::m_opTreated;

/**
 * Map id / unit.
 */
QMap<QString, SKGUnitObject> SKGImportPluginKmy::m_mapIdUnit;

/**
 * Map id / account.
 */
QMap<QString, SKGAccountObject> SKGImportPluginKmy::m_mapIdAccount;

/**
 * Map id / category.
 */
QMap<QString, SKGCategoryObject> SKGImportPluginKmy::m_mapIdCategory;

/**
 * Map id / payee.
 */
QMap<QString, SKGPayeeObject> SKGImportPluginKmy::m_mapIdPayee;

/**
 * This plugin factory.
 */
K_PLUGIN_CLASS_WITH_JSON(SKGImportPluginKmy, "metadata.json")

SKGImportPluginKmy::SKGImportPluginKmy(QObject* iImporter, const QVariantList& iArg)
    : SKGImportPlugin(iImporter)
{
    SKGTRACEINFUNC(10)
    Q_UNUSED(iArg)
}

SKGImportPluginKmy::~SKGImportPluginKmy()
    = default;

bool SKGImportPluginKmy::isImportPossible()
{
    SKGTRACEINFUNC(10)
    return isExportPossible();
}

SKGError SKGImportPluginKmy::importSecurities(QDomElement& docElem)
{
    SKGError err;
    QDomElement securities = docElem.firstChildElement(QLatin1String("SECURITIES"));
    if (!err && !securities.isNull()) {
        SKGTRACEINRC(10, "SKGImportPluginKmy::importFile-SECURITIES", err)
        QDomNodeList securityList = securities.elementsByTagName(QLatin1String("SECURITY"));
        int nb = securityList.count();
        err = m_importer->getDocument()->beginTransaction("#INTERNAL#" % i18nc("Import step", "Import units"), nb);
        for (int i = 0; !err && i < nb; ++i) {
            QDomElement security = securityList.at(i).toElement();
            QString unitName = security.attribute(QLatin1String("name"));

            // We try a creation
            SKGUnitObject unitObj(m_importer->getDocument());
            SKGUnitObject::createCurrencyUnit(m_importer->getDocument(), unitName, unitObj);

            if (!err && (unitObj.getID() == 0)) {
                // Creation of unit
                err = unitObj.setName(unitName);
                QString symbol = security.attribute(QLatin1String("symbol"));
                if (symbol.isEmpty()) {
                    symbol = std::move(unitName);
                }
                IFOKDO(err, unitObj.setSymbol(symbol))
                IFOKDO(err, unitObj.setCountry(security.attribute(QLatin1String("trading-market"))))
                IFOKDO(err, unitObj.setType(SKGUnitObject::SHARE))
                IFOK(err) {
                    // Set pairs
                    QDomNodeList pairList = security.elementsByTagName(QLatin1String("PAIR"));
                    int nb2 = pairList.count();
                    for (int j = 0; !err && j < nb2; ++j) {
                        QDomElement pair = pairList.at(j).toElement();
                        if (pair.attribute(QLatin1String("key")).toLower() == QLatin1String("kmm-security-id")) {
                            err = unitObj.setInternetCode(pair.attribute(QLatin1String("value")));
                        }
                    }
                }
                IFOKDO(err, unitObj.save())
            }

            m_mapIdUnit[security.attribute(QLatin1String("id"))] = unitObj;

            IFOKDO(err, m_importer->getDocument()->stepForward(i + 1))
        }

        SKGENDTRANSACTION(m_importer->getDocument(),  err)
    }
    return err;
}

SKGError SKGImportPluginKmy::importPrices(QDomElement& docElem)
{
    SKGError err;
    QDomElement prices = docElem.firstChildElement(QLatin1String("PRICES"));
    if (!err && !prices.isNull()) {
        SKGTRACEINRC(10, "SKGImportPluginKmy::importFile-PRICES", err)
        QDomNodeList pricepairList = prices.elementsByTagName(QLatin1String("PRICEPAIR"));
        int nb = pricepairList.count();
        err = m_importer->getDocument()->beginTransaction("#INTERNAL#" % i18nc("Import step", "Import units"), nb);
        for (int i = 0; !err && i < nb; ++i) {
            QDomElement pricepair = pricepairList.at(i).toElement();

            SKGUnitObject unitObj = m_mapIdUnit.value(pricepair.attribute(QLatin1String("from")));
            if (unitObj.getID() != 0) {
                // Unit is existing
                QDomNodeList pricesList = pricepair.elementsByTagName(QLatin1String("PRICE"));
                int nb2 = pricesList.count();
                for (int j = 0; !err && j < nb2; ++j) {
                    QDomElement price = pricesList.at(j).toElement();

                    SKGUnitValueObject unitValObj;
                    err = unitObj.addUnitValue(unitValObj);
                    IFOKDO(err, unitValObj.setDate(QDate::fromString(price.attribute(QLatin1String("date")), QLatin1String("yyyy-MM-dd"))))
                    IFOKDO(err, unitValObj.setQuantity(toKmyValue(price.attribute(QLatin1String("price")))))
                    IFOKDO(err, unitValObj.save(true, false))
                }
            }

            IFOKDO(err, m_importer->getDocument()->stepForward(i + 1))
        }

        SKGENDTRANSACTION(m_importer->getDocument(),  err)
    }
    return err;
}

SKGError SKGImportPluginKmy::importInstitutions(QMap<QString, SKGBankObject>& mapIdBank, QDomElement& docElem)
{
    SKGError err;
    QDomElement institutions = docElem.firstChildElement(QLatin1String("INSTITUTIONS"));
    if (!err && !institutions.isNull()) {
        SKGTRACEINRC(10, "SKGImportPluginKmy::importFile-INSTITUTIONS", err)
        QDomNodeList institutionList = institutions.elementsByTagName(QLatin1String("INSTITUTION"));
        int nb = institutionList.count();
        for (int i = 0; !err && i < nb; ++i) {
            // Get bank object
            QDomElement bank = institutionList.at(i).toElement();
            SKGBankObject bankObject(m_importer->getDocument());
            err = bankObject.setName(bank.attribute(QLatin1String("name")));
            IFOKDO(err, bankObject.setNumber(bank.attribute(QLatin1String("sortcode"))))
            IFOKDO(err, bankObject.save())
            mapIdBank[bank.attribute(QLatin1String("id"))] = bankObject;
        }
    }
    return err;
}

SKGError SKGImportPluginKmy::importPayees(QMap<QString, SKGPayeeObject>& mapIdPayee, QDomElement& docElem)
{
    SKGError err;
    QDomElement payees = docElem.firstChildElement(QLatin1String("PAYEES"));
    if (!err && !payees.isNull()) {
        SKGTRACEINRC(10, "SKGImportPluginKmy::importFile-PAYEES", err)
        QDomNodeList payeeList = payees.elementsByTagName(QLatin1String("PAYEE"));
        int nb = payeeList.count();
        for (int i = 0; !err && i < nb; ++i) {
            // Get account object
            QDomElement payee = payeeList.at(i).toElement();
            QDomElement address = payee.firstChildElement(QLatin1String("ADDRESS"));
            SKGPayeeObject payeeObject;
            err = SKGPayeeObject::createPayee(m_importer->getDocument(), payee.attribute(QLatin1String("name")), payeeObject);
            IFOK(err) {
                QString add = address.attribute(QLatin1String("street")) % QLatin1Char(' ') % address.attribute(QLatin1String("postcode")) % QLatin1Char(' ') % address.attribute(QLatin1String("city")) % QLatin1Char(' ') % address.attribute(QLatin1String("state")) % QLatin1Char(' ') % address.attribute(QLatin1String("telephone"));
                add.replace(QLatin1String("  "), QLatin1String(" "));
                err = payeeObject.setAddress(add.trimmed());
                IFOKDO(err, payeeObject.save())
            }
            IFOK(err) {
                mapIdPayee[payee.attribute(QLatin1String("id"))] = payeeObject;
            }
        }
    }
    return err;
}

SKGError SKGImportPluginKmy::importTransactions(QDomElement& docElem, SKGAccountObject& kmymoneyTemporaryAccount, QMap<QString, SKGPayeeObject>& mapIdPayee)
{
    SKGError err;
    SKGTRACEINRC(10, "SKGImportPluginKmy::importFile-TRANSACTION", err)
    QDomNodeList transactionList = docElem.elementsByTagName(QLatin1String("TRANSACTION"));
    int nb = transactionList.count();
    err = m_importer->getDocument()->beginTransaction("#INTERNAL#" % i18nc("Import step", "Import transactions"), nb);
    QVector<QDomNode> suboperationsList;
    for (int i = 0; !err && i < nb; ++i) {
        // Get transaction object
        QDomElement operation = transactionList.at(i).toElement();

        SKGOperationObject operationObj;
        err = kmymoneyTemporaryAccount.addOperation(operationObj, true);
        IFOKDO(err, operationObj.setDate(QDate::fromString(operation.attribute(QLatin1String("postdate")), QLatin1String("yyyy-MM-dd"))))
        IFOKDO(err, operationObj.setComment(operation.attribute(QLatin1String("memo"))))
        IFOKDO(err, operationObj.setImported(true))
        IFOKDO(err, operationObj.setImportID("KMY-" % operation.attribute(QLatin1String("id"))))
        IFOK(err) {
            QString unitName = operation.attribute(QLatin1String("commodity"));
            if (unitName.isEmpty()) {
                unitName = QLatin1String("??");
            }
            SKGUnitObject unit(m_importer->getDocument());
            err = unit.setName(unitName);
            IFOKDO(err, unit.setSymbol(unitName))
            IFOK(err) {
                if (unit.exist()) {
                    err = unit.load();
                } else {
                    err = unit.save();
                }

                IFOKDO(err, operationObj.setUnit(unit))
            }
        }
        IFOKDO(err, operationObj.save(false, false))

        // Is it a schedule ?
        IFOK(err) {
            QDomElement recu = operation.parentNode().toElement();
            if (recu.tagName() == QLatin1String("SCHEDULED_TX")) {
                // Yes ==> creation of the schedule
                IFOKDO(err, operationObj.load())
                IFOKDO(err, operationObj.setTemplate(true))
                IFOKDO(err, operationObj.save(true, false))

                // Yes ==> creation of the schedule
                SKGRecurrentOperationObject recuObj;
                IFOKDO(err, operationObj.addRecurrentOperation(recuObj))
                IFOKDO(err, recuObj.setDate(operationObj.getDate()))
                IFOKDO(err, recuObj.autoWriteEnabled(recu.attribute(QLatin1String("autoEnter")) == QLatin1String("1")))
                IFOKDO(err, recuObj.setAutoWriteDays(0))

                int occu = SKGServices::stringToInt(recu.attribute(QLatin1String("occurenceMultiplier")));
                QString occuType = recu.attribute(QLatin1String("occurence"));  // krazy:exclude=spelling
                SKGRecurrentOperationObject::PeriodUnit period;
                if (occuType == QLatin1String("32") ||
                    occuType == QLatin1String("1024") ||
                    occuType == QLatin1String("256") ||
                    occuType == QLatin1String("8192")) {
                    period = SKGRecurrentOperationObject::MONTH;
                } else if (occuType == QLatin1String("1")) {
                    period = SKGRecurrentOperationObject::DAY;
                    IFOKDO(err, recuObj.timeLimit(true))
                    IFOKDO(err, recuObj.setTimeLimit(1))
                } else if (occuType == QLatin1String("2")) {
                    period = SKGRecurrentOperationObject::DAY;
                } else if (occuType == QLatin1String("4")) {
                    period = SKGRecurrentOperationObject::WEEK;
                } else if (occuType == QLatin1String("18")) {
                    period = SKGRecurrentOperationObject::WEEK;
                    occu *= 2;
                } else {
                    period = SKGRecurrentOperationObject::YEAR;
                }

                IFOKDO(err, recuObj.setPeriodIncrement(occu))
                IFOKDO(err, recuObj.setPeriodUnit(period))

                IFOK(err) {
                    QString endDate = recu.attribute(QLatin1String("endDate"));
                    if (!endDate.isEmpty()) {
                        IFOKDO(err, recuObj.timeLimit(true))
                        IFOKDO(err, recuObj.setTimeLimit(QDate::fromString(recu.attribute(QLatin1String("endDate")), QLatin1String("yyyy-MM-dd"))))
                    }
                }

                if (occuType == QLatin1String("1") && !recu.attribute(QLatin1String("lastPayment")).isEmpty()) {
                    // Single schedule already done
                    IFOKDO(err, recuObj.timeLimit(true))
                    IFOKDO(err, recuObj.setTimeLimit(0))
                }
                IFOKDO(err, recuObj.save(true, false))
            }
        }

        // Get splits
        bool parentSet = false;
        double quantity = 0;
        QDomElement splits = operation.firstChildElement(QLatin1String("SPLITS"));

        int nbSuboperations = 0;
        suboperationsList.resize(0);
        {
            QDomNodeList suboperationsListTmp = splits.elementsByTagName(QLatin1String("SPLIT"));
            nbSuboperations = suboperationsListTmp.count();

            for (int j = 0; !err && j < nbSuboperations; ++j) {
                QDomElement suboperation = suboperationsListTmp.at(j).toElement();
                if (m_mapIdCategory.contains(suboperation.attribute(QLatin1String("account")))) {
                    suboperationsList.push_front(suboperation);
                } else {
                    suboperationsList.push_back(suboperation);
                }
            }
        }

        SKGSubOperationObject suboperationObj;
        for (int j = 0; !err && j < nbSuboperations; ++j) {
            QDomElement suboperation = suboperationsList.at(j).toElement();

            // Set transaction attributes
            IFOKDO(err, operationObj.setPayee(mapIdPayee[suboperation.attribute(QLatin1String("payee"))]))
            if (!err && operationObj.getComment().isEmpty()) {
                err = operationObj.setComment(suboperation.attribute(QLatin1String("memo")));
            }
            IFOKDO(err, operationObj.setNumber(suboperation.attribute(QLatin1String("number"))))

            // Set state
            if (operationObj.getStatus() == SKGOperationObject::NONE) {
                QString val = suboperation.attribute(QLatin1String("reconcileflag"));
                IFOKDO(err, operationObj.setStatus(val == QLatin1String("1") ? SKGOperationObject::MARKED : val == QLatin1String("2") ? SKGOperationObject::CHECKED : SKGOperationObject::NONE))
            }
            IFOKDO(err, operationObj.save())

            if (m_mapIdCategory.contains(suboperation.attribute(QLatin1String("account")))) {
                // It is a split on category
                SKGCategoryObject cat = m_mapIdCategory.value(suboperation.attribute(QLatin1String("account")));

                double q = -toKmyValue(suboperation.attribute(QLatin1String("shares")));
                if (!err && ((suboperationObj.getID() == 0) || suboperationObj.getQuantity() != q)) {
                    err = operationObj.addSubOperation(suboperationObj);
                }
                IFOKDO(err, suboperationObj.setQuantity(q))
                IFOKDO(err, suboperationObj.setCategory(cat))
                IFOKDO(err, suboperationObj.setComment(suboperation.attribute(QLatin1String("memo"))))
                IFOKDO(err, suboperationObj.save(true, false))
            } else {
                QString accountSubOp = suboperation.attribute(QLatin1String("account"));
                if (!accountSubOp.isEmpty()) {
                    if (nbSuboperations == 1) {
                        SKGUnitObject unit = m_mapIdUnit.value(accountSubOp);
                        if (unit.getID() != 0) {
                            IFOKDO(err, operationObj.setUnit(unit))
                        }
                    }

                    if (!m_mapIdAccount.contains(accountSubOp) || (nbSuboperations == 2 &&
                            m_mapIdAccount.value(accountSubOp) == kmymoneyTemporaryAccount &&
                            suboperationsList.at(0).toElement().attribute(QLatin1String("action")).isEmpty() &&
                            suboperationsList.at(1).toElement().attribute(QLatin1String("action")).isEmpty())) {
                        // Set as initial balance
                        IFOKDO(err, operationObj.setAttribute(QLatin1String("d_date"), QLatin1String("0000-00-00")))
                        IFOKDO(err, operationObj.setStatus(SKGOperationObject::CHECKED))
                        IFOKDO(err, operationObj.save(true, false))

                        if (!err && (suboperationObj.getID() == 0)) {
                            err = operationObj.addSubOperation(suboperationObj);
                        }

                        IFOKDO(err, suboperationObj.setAttribute(QLatin1String("d_date"), QLatin1String("0000-00-00")))
                        IFOKDO(err, suboperationObj.save(true, false))
                    } else {
                        // It is a transfer of account
                        SKGAccountObject act = m_mapIdAccount.value(accountSubOp);

                        if (parentSet) {
                            // If the parent is already set, it means that is a transfer
                            SKGOperationObject operationObj2;
                            IFOKDO(err, act.addOperation(operationObj2, true))
                            IFOKDO(err, operationObj2.setTemplate(operationObj.isTemplate()))
                            IFOKDO(err, operationObj2.setDate(operationObj.getDate()))
                            IFOKDO(err, operationObj2.setNumber(operationObj.getNumber()))
                            IFOKDO(err, operationObj2.setComment(operationObj.getComment()))
                            SKGPayeeObject payeeObject;
                            operationObj.getPayee(payeeObject);
                            IFOKDO(err, operationObj2.setPayee(payeeObject))
                            IFOK(err) {
                                QString val = suboperation.attribute(QLatin1String("reconcileflag"));
                                err = operationObj2.setStatus(val == QLatin1String("1") ? SKGOperationObject::MARKED : val == QLatin1String("2") ? SKGOperationObject::CHECKED : SKGOperationObject::NONE);
                            }
                            IFOKDO(err, operationObj2.setImported(true))
                            IFOKDO(err, operationObj2.setImportID(operationObj.getImportID()))

                            SKGUnitObject unit = m_mapIdUnit.value(accountSubOp);
                            if (unit.getID() != 0) {
                                IFOKDO(err, operationObj2.setUnit(unit))
                            } else {
                                IFOKDO(err, operationObj.getUnit(unit))
                                IFOKDO(err, operationObj2.setUnit(unit))
                            }
                            IFOKDO(err, operationObj2.save())
                            IFOKDO(err, operationObj2.setGroupOperation(operationObj))
                            IFOKDO(err, operationObj2.save())

                            // Create sub transaction on operationObj2
                            SKGSubOperationObject suboperationObj2;
                            IFOKDO(err, operationObj2.addSubOperation(suboperationObj2))
                            IFOK(err) {
                                // We must take the quality of the split having an action
                                quantity = toKmyValue(suboperation.attribute(QLatin1String("shares")));
                                err = suboperationObj2.setQuantity(quantity);
                            }
                            IFOKDO(err, suboperationObj2.setComment(suboperation.attribute(QLatin1String("memo"))))
                            IFOKDO(err, suboperationObj2.save(true, false))
                        } else {
                            // We set the parent
                            if (Q_LIKELY(!err) && (act.getID() == 0)) {
                                err = SKGError(ERR_FAIL, i18nc("Error message", "Account with identifier %1 not found", suboperation.attribute(QLatin1String("account"))));
                            }
                            IFOKDO(err, operationObj.setParentAccount(act, true))
                            IFOKDO(err, operationObj.save())

                            // Compute quantity
                            quantity = toKmyValue(suboperation.attribute(QLatin1String("shares")));

                            // Create sub transaction on operationObj
                            quantity -= SKGServices::stringToDouble(operationObj.getAttribute(QLatin1String("f_QUANTITY")));
                            if (quantity != 0 || nbSuboperations == 1) {
                                IFOKDO(err, operationObj.addSubOperation(suboperationObj))
                                IFOKDO(err, suboperationObj.setQuantity(quantity))
                                IFOKDO(err, suboperationObj.setComment(suboperation.attribute(QLatin1String("memo"))))
                                IFOKDO(err, suboperationObj.save(true, false))
                            }

                            parentSet = true;
                        }
                    }
                }
            }
        }

        if (!err && i % 500 == 0) {
            err = m_importer->getDocument()->executeSqliteOrder(QLatin1String("ANALYZE"));
        }

        IFOKDO(err, m_importer->getDocument()->stepForward(i + 1))
    }

    SKGENDTRANSACTION(m_importer->getDocument(),  err)

    return err;
}

SKGError SKGImportPluginKmy::importBudget(QDomElement& docElem)
{
    SKGError err;
    QDomElement budgets = docElem.firstChildElement(QLatin1String("BUDGETS"));
    if (!err && !budgets.isNull()) {
        SKGTRACEINRC(10, "SKGImportPluginKmy::importFile-BUDGETS", err)
        // Build cache of categories
        QMap<int, bool> catExpense;
        {
            SKGStringListList list;
            err = m_importer->getDocument()->executeSelectSqliteOrder(QLatin1String("SELECT id, t_TYPEEXPENSE FROM v_category_display"), list);
            int nb = list.count();
            for (int i = 1; i < nb; ++i) {
                catExpense[SKGServices::stringToInt(list.at(i).at(0))] = (list.at(i).at(0) == QLatin1String("-"));
            }
        }

        QDomNodeList budgetList = budgets.elementsByTagName(QLatin1String("BUDGET"));
        int nb = budgetList.count();
        IFOKDO(err, m_importer->getDocument()->beginTransaction("#INTERNAL#" % i18nc("Import step", "Import budgets"), nb))
        for (int i = 0; !err && i < nb; ++i) {
            QDomElement budget = budgetList.at(i).toElement();
            QDomNodeList accountList = budget.elementsByTagName(QLatin1String("ACCOUNT"));
            int nb2 = accountList.count();
            for (int j = 0; !err && j < nb2; ++j) {
                QDomElement account = accountList.at(j).toElement();
                SKGCategoryObject cat = m_mapIdCategory.value(account.attribute(QLatin1String("id")));
                QString budgetlevel = account.attribute(QLatin1String("budgetlevel"));

                QDomNodeList periodList = account.elementsByTagName(QLatin1String("PERIOD"));
                int nb3 = periodList.count();
                for (int k = 0; !err && k < nb3; ++k) {
                    QDomElement period = periodList.at(k).toElement();

                    double q = toKmyValue(period.attribute(QLatin1String("amount")));

                    // Are we able to find to sign with the category ?
                    if (catExpense[cat.getID()]) {
                        q = -q;
                    }

                    QStringList dates = SKGServices::splitCSVLine(period.attribute(QLatin1String("start")), '-');
                    if (dates.count() == 3) {
                        // We try a creation
                        for (int m = 1; !err && m <= (budgetlevel == QLatin1String("monthly") ? 12 : 1); ++m) {
                            SKGBudgetObject budget2(m_importer->getDocument());
                            err = budget2.setCategory(cat);
                            IFOKDO(err, budget2.setBudgetedAmount(q))
                            IFOKDO(err, budget2.setYear(SKGServices::stringToDouble(dates.at(0))))
                            IFOKDO(err, budget2.setMonth(budgetlevel == QLatin1String("monthbymonth") ? SKGServices::stringToDouble(dates.at(1)) :
                                                         budgetlevel == QLatin1String("yearly") ? 0 : m));
                            IFOKDO(err, budget2.save(true, false))
                        }
                    }
                }
            }

            IFOKDO(err, m_importer->getDocument()->stepForward(i + 1))
        }

        SKGENDTRANSACTION(m_importer->getDocument(),  err)
    }
    return err;
}

SKGError SKGImportPluginKmy::importAccounts(SKGBankObject& bank, SKGAccountObject& kmymoneyTemporaryAccount, QMap<QString, SKGBankObject>& mapIdBank, QDomElement& docElem)
{
    SKGError err;
    IFOKDO(err, m_importer->getDocument()->addOrModifyAccount(QLatin1String("KMYMONEY-TEMPORARY-ACCOUNT"), QString(), QLatin1String("KMYMONEY")))
    IFOKDO(err, bank.setName(QLatin1String("KMYMONEY")))
    IFOKDO(err, bank.load())
    IFOKDO(err, kmymoneyTemporaryAccount.setName(QLatin1String("KMYMONEY-TEMPORARY-ACCOUNT")))
    IFOKDO(err, kmymoneyTemporaryAccount.load())

    QDomElement accounts = docElem.firstChildElement(QLatin1String("ACCOUNTS"));
    if (!err && !accounts.isNull()) {
        SKGTRACEINRC(10, "SKGImportPluginKmy::importFile-ACCOUNTS", err)
        QDomNodeList accountList = accounts.elementsByTagName(QLatin1String("ACCOUNT"));
        int nb = accountList.count();
        err = m_importer->getDocument()->beginTransaction("#INTERNAL#" % i18nc("Import step", "Import accounts"), nb);
        QMap<QString, QString> type15Account;
        QMap<QString, QString> type15Unit;
        for (int i = 0; !err && i < nb; ++i) {
            // Get account object
            QDomElement account = accountList.at(i).toElement();
            QString type = account.attribute(QLatin1String("type"));
            if (type == QLatin1String("15")) {
                // Actions: the real account is the parent
                type15Account[account.attribute(QLatin1String("id"))] = account.attribute(QLatin1String("parentaccount"));
                type15Unit[account.attribute(QLatin1String("id"))] = account.attribute(QLatin1String("currency"));
            } else if (type != QLatin1String("12") && type != QLatin1String("13") && type != QLatin1String("16")) {
                // Get the bank
                QString bankId = account.attribute(QLatin1String("institution"));
                SKGBankObject bankObj = (bankId.isEmpty() ? bank : mapIdBank[account.attribute(QLatin1String("institution"))]);

                // Creation of the account
                SKGAccountObject accountObj;
                IFOKDO(err, bankObj.addAccount(accountObj))
                IFOKDO(err, accountObj.setName(account.attribute(QLatin1String("name"))))
                IFOKDO(err, accountObj.setNumber(account.attribute(QLatin1String("number"))))
                IFOKDO(err, accountObj.setComment(account.attribute(QLatin1String("description"))))
                IFOK(err) {
                    SKGAccountObject::AccountType typeA = (type == QLatin1String("4") ? SKGAccountObject::CREDITCARD : (type == QLatin1String("7") ? SKGAccountObject::INVESTMENT : (type == QLatin1String("9") ? SKGAccountObject::ASSETS : (type == QLatin1String("3") ? SKGAccountObject::WALLET : (type == QLatin1String("10") ? SKGAccountObject::LOAN : SKGAccountObject::CURRENT)))));
                    err = accountObj.setType(typeA);
                }
                IFOK(err) {
                    // Set pairs
                    QDomNodeList pairList = account.elementsByTagName(QLatin1String("PAIR"));
                    int nb2 = pairList.count();
                    for (int j = 0; !err && j < nb2; ++j) {
                        QDomElement pair = pairList.at(j).toElement();
                        if (pair.attribute(QLatin1String("key")).toLower() == QLatin1String("mm-closed") && pair.attribute(QLatin1String("value")).toLower() == QLatin1String("yes")) {
                            err = accountObj.setClosed(true);
                        } else if (pair.attribute(QLatin1String("key")).toLower() == QLatin1String("preferredaccount") && pair.attribute(QLatin1String("value")).toLower() == QLatin1String("yes")) {
                            err = accountObj.bookmark(true);
                        } else if (pair.attribute(QLatin1String("key")).toLower() == QLatin1String("minbalanceabsolute")) {
                            err = accountObj.setMinLimitAmount(toKmyValue(pair.attribute(QLatin1String("value"))));
                            IFOKDO(err, accountObj.minLimitAmountEnabled(true))
                        } else if (pair.attribute(QLatin1String("key")).toLower() == QLatin1String("maxcreditabsolute")) {
                            err = accountObj.setMaxLimitAmount(-toKmyValue(pair.attribute(QLatin1String("value"))));
                            IFOKDO(err, accountObj.maxLimitAmountEnabled(true))
                        }
                    }
                }
                IFOKDO(err, accountObj.save())

                // Change parent bank in case of ASSETS
                if (accountObj.getType() == SKGAccountObject::WALLET) {
                    // Get blank bank
                    SKGBankObject blankBank(m_importer->getDocument());
                    IFOKDO(err, blankBank.setName(QString()))
                    if (blankBank.exist()) {
                        err = blankBank.load();
                    } else {
                        err = blankBank.save();
                    }
                    IFOKDO(err, accountObj.setBank(blankBank))
                    IFOKDO(err, accountObj.save())
                }

                m_mapIdAccount[account.attribute(QLatin1String("id"))] = accountObj;
            } else if (type == QLatin1String("16")) {
                m_mapIdAccount[account.attribute(QLatin1String("id"))] = kmymoneyTemporaryAccount;
            } else {
                // Create category
                SKGCategoryObject cat = m_mapIdCategory.value(account.attribute(QLatin1String("id")));
                if (cat.getID() != 0) {
                    // Already existing ==> we must set the right name
                    err = cat.setName(account.attribute(QLatin1String("name")));
                    IFOKDO(err, cat.save())
                } else {
                    // We must create it
                    cat = SKGCategoryObject(m_importer->getDocument());
                    err = cat.setName(account.attribute(QLatin1String("name")));
                    IFOKDO(err, cat.save())
                }
                if (err) {
                    // The category already exists
                    SKGCategoryObject catp;
                    err = cat.getParentCategory(catp);
                    QString fullName = catp.getFullName() % OBJECTSEPARATOR % cat.getName();
                    IFOKDO(err, cat.remove(false))
                    IFOKDO(err, SKGCategoryObject::createPathCategory(m_importer->getDocument(), fullName, cat))
                }
                m_mapIdCategory[account.attribute(QLatin1String("id"))] = cat;

                // Create sub categories
                QDomNodeList subaccountList = account.elementsByTagName(QLatin1String("SUBACCOUNT"));
                int nb2 = subaccountList.count();
                for (int j = 0; !err && j < nb2; ++j) {
                    QDomElement subaccount = subaccountList.at(j).toElement();

                    // Is child already existing ?
                    SKGCategoryObject cat2 = m_mapIdCategory.value(subaccount.attribute(QLatin1String("id")));
                    if (cat2.getID() != 0) {
                        // Yes ==> reparent
                        err = cat2.setParentCategory(cat);
                        IFOKDO(err, cat2.save(true, false))
                    } else {
                        // No ==> create
                        IFOKDO(err, cat.addCategory(cat2))
                        IFOKDO(err, cat2.setName(subaccount.attribute(QLatin1String("id"))))
                        IFOKDO(err, cat2.save())
                        m_mapIdCategory[subaccount.attribute(QLatin1String("id"))] = cat2;
                    }
                }
            }

            QStringList list = type15Account.keys();
            for (const auto& k : std::as_const(list)) {
                m_mapIdAccount[k] = m_mapIdAccount.value(type15Account[k]);
                m_mapIdUnit[account.attribute(QLatin1String("id"))] = m_mapIdUnit.value(account.attribute(QLatin1String("currency")));
            }

            list = type15Unit.keys();
            for (const auto& k : std::as_const(list)) {
                m_mapIdUnit[k] = m_mapIdUnit[type15Unit[k]];
            }

            IFOKDO(err, m_importer->getDocument()->stepForward(i + 1))
        }

        SKGENDTRANSACTION(m_importer->getDocument(),  err)
    }
    return err;
}

SKGError SKGImportPluginKmy::importFile()
{
    if (m_importer->getDocument() == nullptr) {
        return SKGError(ERR_ABORT, i18nc("Error message", "Invalid parameters"));
    }

    SKGError err;
    SKGTRACEINFUNCRC(2, err)

    // Initialisation
    m_mapIdUnit.clear();
    m_mapIdAccount.clear();
    m_mapIdCategory.clear();
    m_mapIdPayee.clear();

    // Open file
    KCompressionDevice file(m_importer->getLocalFileName(), KCompressionDevice::GZip);
    if (!file.open(QIODevice::ReadOnly)) {
        err.setReturnCode(ERR_INVALIDARG).setMessage(i18nc("Error message",  "Open file '%1' failed", m_importer->getFileName().toDisplayString()));
    } else {
        QDomDocument doc;

        // Set the file without uncompression
        QString errorMsg;
        int errorLine = 0;
        int errorCol = 0;
        bool contentOK = false;        
#ifdef SKG_QT6
        const auto& content = QString::fromUtf8(file.readAll());
        const auto& result= doc.setContent(content);

        contentOK = !!result;
        errorLine =result.errorLine;
        errorCol =result.errorColumn;
        errorMsg =result.errorMessage;
#else        
        contentOK = doc.setContent(file.readAll(), &errorMsg, &errorLine, &errorCol);
#endif
        file.close();

        if (!contentOK) {
            err.setReturnCode(ERR_ABORT).setMessage(i18nc("Error message",  "%1-%2: '%3'", errorLine, errorCol, errorMsg)).addError(ERR_INVALIDARG, i18nc("Error message",  "Invalid XML content in file '%1'", m_importer->getFileName().toDisplayString()));
        
        } else {
            // Get root
            QDomElement docElem = doc.documentElement();
            err = m_importer->getDocument()->beginTransaction("#INTERNAL#" % i18nc("Import step", "Import %1 file", "KMY"), 8);

            // Step 1-Get units
            IFOKDO(err, importSecurities(docElem))
            IFOKDO(err, m_importer->getDocument()->stepForward(1))

            // Step 2-Get units prices
            IFOKDO(err, importPrices(docElem))
            IFOKDO(err, m_importer->getDocument()->stepForward(2))

            // Step 3-Create banks
            QMap<QString, SKGBankObject> mapIdBank;
            IFOKDO(err, importInstitutions(mapIdBank, docElem))
            IFOKDO(err, m_importer->getDocument()->stepForward(3))

            // Step 4-Create account and categories
            // Create bank and temporary account for kmymoney import
            SKGAccountObject kmymoneyTemporaryAccount(m_importer->getDocument());
            SKGBankObject bank(m_importer->getDocument());
            IFOKDO(err, importAccounts(bank, kmymoneyTemporaryAccount, mapIdBank, docElem))
            IFOKDO(err, m_importer->getDocument()->stepForward(4))

            // Step 5-Read payees
            QMap<QString, SKGPayeeObject> mapIdPayee;
            IFOKDO(err, importPayees(mapIdPayee, docElem))
            IFOKDO(err, m_importer->getDocument()->stepForward(5))

            // Step 6-Create transactions
            IFOKDO(err, importTransactions(docElem, kmymoneyTemporaryAccount, mapIdPayee))
            IFOKDO(err, m_importer->getDocument()->stepForward(6))

            // Step 7-Get budgets
            IFOKDO(err, importBudget(docElem))
            IFOKDO(err, m_importer->getDocument()->stepForward(7))

            // Step 8-Remove useless account and temporary account
            {
                IFOKDO(err, kmymoneyTemporaryAccount.remove(false, true))
                IFOKDO(err, m_importer->getDocument()->executeSqliteOrder("DELETE FROM account WHERE rd_bank_id=" % SKGServices::intToString(bank.getID()) % " AND (SELECT COUNT(1) FROM operation WHERE operation.rd_account_id=account.id)=0"))
            }
            IFOKDO(err, m_importer->getDocument()->stepForward(8))

            SKGENDTRANSACTION(m_importer->getDocument(),  err)

            IFOKDO(err, m_importer->getDocument()->executeSqliteOrder(QLatin1String("ANALYZE")))
        }
    }

    // Clean
    m_mapIdUnit.clear();
    m_mapIdAccount.clear();
    m_mapIdCategory.clear();
    m_mapIdPayee.clear();

    return err;
}

bool SKGImportPluginKmy::isExportPossible()
{
    SKGTRACEINFUNC(10)
    return (m_importer->getDocument() == nullptr ? true : m_importer->getFileNameExtension() == QLatin1String("KMY"));
}

SKGError SKGImportPluginKmy::exportHeader(QDomDocument& doc, QDomElement& root)
{
    SKGError err;
    QDomElement fileindo = doc.createElement(QLatin1String("FILEINFO"));
    root.appendChild(fileindo);

    {
        // <CREATION_DATE>
        QDomElement creation_date = doc.createElement(QLatin1String("CREATION_DATE"));
        fileindo.appendChild(creation_date);
        creation_date.setAttribute(QLatin1String("date"), SKGServices::dateToSqlString(QDateTime::currentDateTime()));

        // <LAST_MODIFIED_DATE>
        QDomElement last_modified_date = doc.createElement(QLatin1String("LAST_MODIFIED_DATE"));
        fileindo.appendChild(last_modified_date);
        last_modified_date.setAttribute(QLatin1String("date"), SKGServices::dateToSqlString(QDateTime::currentDateTime()));

        // <VERSION>
        QDomElement version = doc.createElement(QLatin1String("VERSION"));
        fileindo.appendChild(version);
        version.setAttribute(QLatin1String("id"), QLatin1String("1"));

        // <FIXVERSION>
        QDomElement fixversion = doc.createElement(QLatin1String("FIXVERSION"));
        fileindo.appendChild(fixversion);
        fixversion.setAttribute(QLatin1String("id"), QLatin1String("2"));
    }

    // <USER>
    QDomElement user = doc.createElement(QLatin1String("USER"));
    root.appendChild(user);
    user.setAttribute(QLatin1String("email"), QString());
    user.setAttribute(QLatin1String("name"), QString());
    {
        // ADDRESS
        QDomElement address = doc.createElement(QLatin1String("ADDRESS"));
        user.appendChild(address);
        address.setAttribute(QLatin1String("street"), QString());
        address.setAttribute(QLatin1String("zipcode"), QString());
        address.setAttribute(QLatin1String("county"), QString());
        address.setAttribute(QLatin1String("city"), QString());
        address.setAttribute(QLatin1String("telephone"), QString());
    }
    return err;
}

SKGError SKGImportPluginKmy::exportSecurities(QDomDocument& doc, QDomElement& root, const QString& stdUnit)
{
    SKGError err;
    QDomElement securities = doc.createElement(QLatin1String("SECURITIES"));
    root.appendChild(securities);

    QDomElement currencies = doc.createElement(QLatin1String("CURRENCIES"));
    root.appendChild(currencies);

    SKGObjectBase::SKGListSKGObjectBase objects;
    IFOKDO(err, m_importer->getDocument()->getObjects(QLatin1String("v_unit"), QLatin1String("t_type!='I'"), objects))
    int nb = objects.count();
    securities.setAttribute(QLatin1String("count"), SKGServices::intToString(nb));

    QStringList importedCurrency;
    IFOK(err) {
        err = m_importer->getDocument()->beginTransaction("#INTERNAL#" % i18nc("Export step", "Export units"), nb);
        for (int i = 0; !err && i < nb; ++i) {
            SKGUnitObject obj(objects.at(i));
            if (obj.getType() == SKGUnitObject::SHARE || obj.getType() == SKGUnitObject::OBJECT) {
                QDomElement security = doc.createElement(QLatin1String("SECURITY"));
                securities.appendChild(security);

                SKGUnitObject parentUnit;
                obj.getUnit(parentUnit);
                QString unitP = SKGUnitObject::getInternationalCode(parentUnit.getName());
                if (unitP.isEmpty()) {
                    unitP = stdUnit;
                }

                security.setAttribute(QLatin1String("id"), obj.getName());
                security.setAttribute(QLatin1String("trading-currency"), unitP);
                security.setAttribute(QLatin1String("saf"), QLatin1String("100000"));
                security.setAttribute(QLatin1String("symbol"), obj.getSymbol());
                security.setAttribute(QLatin1String("trading-market"), obj.getCountry());
                security.setAttribute(QLatin1String("type"), QLatin1String("4"));
                security.setAttribute(QLatin1String("name"), obj.getName());

                QString internetCode = obj.getInternetCode();
                if (!internetCode.isEmpty()) {
                    QDomElement keyvaluepairs2 = doc.createElement(QLatin1String("KEYVALUEPAIRS"));
                    security.appendChild(keyvaluepairs2);

                    QDomElement pair1 = doc.createElement(QLatin1String("PAIR"));
                    keyvaluepairs2.appendChild(pair1);
                    pair1.setAttribute(QLatin1String("key"), QLatin1String("kmm-online-source"));
                    pair1.setAttribute(QLatin1String("value"), QLatin1String("Yahoo"));

                    QDomElement pair2 = doc.createElement(QLatin1String("PAIR"));
                    keyvaluepairs2.appendChild(pair2);
                    pair2.setAttribute(QLatin1String("key"), QLatin1String("kmm-security-id"));
                    pair2.setAttribute(QLatin1String("value"), internetCode);
                }
            } else {
                QDomElement currency = doc.createElement(QLatin1String("CURRENCY"));
                currencies.appendChild(currency);

                QString unit = SKGUnitObject::getInternationalCode(obj.getName());
                if (unit.isEmpty()) {
                    unit = obj.getName();
                }

                currency.setAttribute(QLatin1String("saf"), QLatin1String("100"));
                currency.setAttribute(QLatin1String("symbol"), obj.getSymbol());
                currency.setAttribute(QLatin1String("type"), QLatin1String("3"));
                currency.setAttribute(QLatin1String("id"), unit);
                currency.setAttribute(QLatin1String("name"), obj.getName());
                currency.setAttribute(QLatin1String("ppu"), QLatin1String("100"));
                currency.setAttribute(QLatin1String("scf"), QLatin1String("100"));

                importedCurrency.push_back(obj.getSymbol());
            }
            IFOKDO(err, m_importer->getDocument()->stepForward(i + 1))
        }

        SKGENDTRANSACTION(m_importer->getDocument(),  err)
    }

    // <CURRENCIES>
    QStringList units = SKGUnitObject::getListofKnownCurrencies(false);
    nb = units.count();
    int nbreal = 0;
    for (int i = 0; i < nb; ++i) {
        SKGServices::SKGUnitInfo info = SKGUnitObject::getUnitInfo(units.at(i));
        if (info.Name != info.Symbol && !importedCurrency.contains(info.Symbol)) {
            QDomElement currency = doc.createElement(QLatin1String("CURRENCY"));
            currencies.appendChild(currency);
            currency.setAttribute(QLatin1String("saf"), QLatin1String("100"));
            currency.setAttribute(QLatin1String("symbol"), info.Symbol);
            currency.setAttribute(QLatin1String("type"), QLatin1String("3"));
            currency.setAttribute(QLatin1String("id"), SKGUnitObject::getInternationalCode(info.Name));
            currency.setAttribute(QLatin1String("name"), info.Name);
            currency.setAttribute(QLatin1String("ppu"), QLatin1String("100"));
            currency.setAttribute(QLatin1String("scf"), QLatin1String("100"));

            ++nbreal;
        }
    }
    currencies.setAttribute(QLatin1String("count"), SKGServices::intToString(nbreal));

    // <PRICES>
    QDomElement prices = doc.createElement(QLatin1String("PRICES"));
    root.appendChild(prices);
    IFOKDO(err, m_importer->getDocument()->getObjects(QLatin1String("v_unit"), QLatin1String("t_type='S'"), objects))
    nb = objects.count();
    prices.setAttribute(QLatin1String("count"), SKGServices::intToString(nb));
    IFOK(err) {
        err = m_importer->getDocument()->beginTransaction("#INTERNAL#" % i18nc("Export step", "Export units"), nb);
        for (int i = 0; !err && i < nb; ++i) {
            SKGUnitObject obj(objects.at(i));
            QDomElement pricepair = doc.createElement(QLatin1String("PRICEPAIR"));
            prices.appendChild(pricepair);

            QString unitP = SKGUnitObject::getInternationalCode(obj.getName());
            if (unitP.isEmpty()) {
                unitP = stdUnit;
            }

            pricepair.setAttribute(QLatin1String("from"), obj.getName());
            pricepair.setAttribute(QLatin1String("to"), unitP);

            SKGObjectBase::SKGListSKGObjectBase unitValues;
            err = obj.getUnitValues(unitValues);
            int nb2 = unitValues.count();
            for (int j = 0; !err && j < nb2; ++j) {
                QDomElement price = doc.createElement(QLatin1String("PRICE"));
                pricepair.appendChild(price);

                SKGUnitValueObject unitval(unitValues.at(j));
                price.setAttribute(QLatin1String("price"), SKGImportPluginKmy::kmyValue(unitval.getQuantity()));
                price.setAttribute(QLatin1String("source"), QLatin1String("Utilisateur"));
                price.setAttribute(QLatin1String("date"), SKGServices::dateToSqlString(unitval.getDate()));
            }

            IFOKDO(err, m_importer->getDocument()->stepForward(i + 1))
        }

        SKGENDTRANSACTION(m_importer->getDocument(),  err)
    }

    // <REPORTS>
    QDomElement reports = doc.createElement(QLatin1String("REPORTS"));
    root.appendChild(reports);
    reports.setAttribute(QLatin1String("count"), QLatin1String("0"));

    return err;
}

SKGError SKGImportPluginKmy::exportBudgets(QDomDocument& doc, QDomElement& root)
{
    SKGError err;
    QDomElement budgets = doc.createElement(QLatin1String("BUDGETS"));
    root.appendChild(budgets);

    SKGObjectBase::SKGListSKGObjectBase objects;
    IFOKDO(err, m_importer->getDocument()->getObjects(QLatin1String("v_budget"), QLatin1String("1=1 ORDER BY i_year, i_month"), objects))
    int nb = objects.count();
    int nbBudgets = 0;
    int currentYear = 0;
    QDomElement budget;

    QMap<QString, QDomElement> mapCatAccount;
    IFOK(err) {
        err = m_importer->getDocument()->beginTransaction("#INTERNAL#" % i18nc("Export step", "Export budgets"), nb);
        for (int i = 0; !err && i < nb; ++i) {
            SKGBudgetObject obj(objects.at(i));
            SKGCategoryObject cat;
            obj.getCategory(cat);
            QString catId = getKmyUniqueIdentifier(cat);
            int year = obj.getYear();
            QString yearString = SKGServices::intToString(year);
            int month = obj.getMonth();
            QString monthString = SKGServices::intToString(month);
            if (monthString.isEmpty()) {
                monthString = QLatin1Char('0') % monthString;
            }
            if (currentYear != year) {
                budget = doc.createElement(QLatin1String("BUDGET"));
                budgets.appendChild(budget);
                budget.setAttribute(QLatin1String("version"), QLatin1String("2"));
                budget.setAttribute(QLatin1String("id"), yearString);
                budget.setAttribute(QLatin1String("start"), yearString % "-01-01");
                budget.setAttribute(QLatin1String("name"), yearString);

                currentYear = year;
                mapCatAccount.clear();
                ++nbBudgets;
            }

            QDomElement account = mapCatAccount[catId];
            if (account.isNull() && !catId.isEmpty()) {
                account = doc.createElement(QLatin1String("ACCOUNT"));
                budget.appendChild(account);
                account.setAttribute(QLatin1String("budgetsubaccounts"), QLatin1String("0"));
                account.setAttribute(QLatin1String("id"), catId);
                mapCatAccount[catId] = account;
            }
            if (!account.isNull()) {
                account.setAttribute(QLatin1String("budgetlevel"), obj.getMonth() == 0 ? QLatin1String("yearly") : QLatin1String("monthbymonth"));

                QDomElement period = doc.createElement(QLatin1String("PERIOD"));
                account.appendChild(period);
                period.setAttribute(QLatin1String("amount"), SKGImportPluginKmy::kmyValue(qAbs(obj.getBudgetedAmount())));
                period.setAttribute(QLatin1String("start"), yearString % QLatin1Char('-') % (obj.getMonth() == 0 ? QLatin1String("01") : monthString) % "-01");
            }

            IFOKDO(err, m_importer->getDocument()->stepForward(i + 1))
        }

        SKGENDTRANSACTION(m_importer->getDocument(),  err)
    }
    budgets.setAttribute(QLatin1String("count"), SKGServices::intToString(nbBudgets));
    return err;
}

SKGError SKGImportPluginKmy::exportSchedules(QDomDocument& doc, QDomElement& root)
{
    SKGError err;
    QDomElement schedules = doc.createElement(QLatin1String("SCHEDULES"));
    root.appendChild(schedules);

    SKGObjectBase::SKGListSKGObjectBase objects;
    IFOKDO(err, m_importer->getDocument()->getObjects(QLatin1String("v_recurrentoperation"), QString(), objects))
    int nb = objects.count();
    schedules.setAttribute(QLatin1String("count"), SKGServices::intToString(nb));
    IFOK(err) {
        err = m_importer->getDocument()->beginTransaction("#INTERNAL#" % i18nc("Export step", "Export scheduled transactions"), nb);
        for (int i = 0; !err && i < nb; ++i) {
            SKGRecurrentOperationObject obj(objects.at(i));
            SKGOperationObject op;
            err = obj.getParentOperation(op);
            IFOK(err) {
                QDomElement scheduled_tx = doc.createElement(QLatin1String("SCHEDULED_TX"));
                schedules.appendChild(scheduled_tx);

                scheduled_tx.setAttribute(QLatin1String("id"), getKmyUniqueIdentifier(obj));
                scheduled_tx.setAttribute(QLatin1String("name"), getKmyUniqueIdentifier(obj));
                scheduled_tx.setAttribute(QLatin1String("startDate"), obj.getAttribute(QLatin1String("d_date")));
                scheduled_tx.setAttribute(QLatin1String("lastPayment"), obj.getAttribute(QLatin1String("d_date")));
                bool autoEnter = obj.isAutoWriteEnabled();
                scheduled_tx.setAttribute(QLatin1String("autoEnter"), autoEnter  ? QLatin1String("1") : QLatin1String("0"));

                QString occuType;
                int occu = obj.getPeriodIncrement();
                SKGRecurrentOperationObject::PeriodUnit punit = obj.getPeriodUnit();
                if (punit == SKGRecurrentOperationObject::MONTH) {
                    occuType = QLatin1String("32");
                } else if (punit == SKGRecurrentOperationObject::WEEK) {
                    occuType = '4';
                } else if (punit == SKGRecurrentOperationObject::DAY) {
                    occuType = '2';
                } else {
                    occuType = QLatin1String("16384");
                }

                scheduled_tx.setAttribute(QLatin1String("occurenceMultiplier"), SKGServices::intToString(occu));
                scheduled_tx.setAttribute(QLatin1String("occurence"), occuType);    // krazy:exclude=spelling
                scheduled_tx.setAttribute(QLatin1String("weekendOption"), QLatin1String("0"));
                scheduled_tx.setAttribute(QLatin1String("paymentType"), QLatin1String("1"));
                QChar type = '1';
                SKGOperationObject op2;
                if (op.isTransfer(op2)) {
                    type = '4';
                } else if (op.getCurrentAmount() > 0) {
                    type = '2';
                }
                scheduled_tx.setAttribute(QLatin1String("type"), type);
                scheduled_tx.setAttribute(QLatin1String("fixed"), QLatin1String("1"));

                QString endDate;
                if (obj.hasTimeLimit()) {
                    QDate firstDate = obj.getDate();

                    // We must compute the date
                    int p = occu * (obj.getTimeLimit() - 1);
                    if (punit == SKGRecurrentOperationObject::DAY) {
                        firstDate = firstDate.addDays(p);
                    } else if (punit == SKGRecurrentOperationObject::MONTH) {
                        firstDate = firstDate.addMonths(p);
                    } else if (punit == SKGRecurrentOperationObject::YEAR) {
                        firstDate = firstDate.addYears(p);
                    }

                    endDate = firstDate.toString(QLatin1String("yyyy-MM-dd"));
                }
                scheduled_tx.setAttribute(QLatin1String("endDate"), endDate);

                err = exportOperation(op, doc, scheduled_tx);
            }
            IFOKDO(err, m_importer->getDocument()->stepForward(i + 1))
        }

        SKGENDTRANSACTION(m_importer->getDocument(),  err)
    }
    return err;
}

SKGError SKGImportPluginKmy::exportTransactions(QDomDocument& doc, QDomElement& root, const QString& stdUnit)
{
    SKGError err;
    QDomElement transactions = doc.createElement(QLatin1String("TRANSACTIONS"));
    root.appendChild(transactions);

    SKGObjectBase::SKGListSKGObjectBase objects;
    IFOKDO(err, m_importer->getDocument()->getObjects(QLatin1String("v_operation"), QLatin1String("t_template='N' ORDER BY d_date DESC"), objects))
    int nb = objects.count();
    transactions.setAttribute(QLatin1String("count"), SKGServices::intToString(nb));
    IFOK(err) {
        err = m_importer->getDocument()->beginTransaction("#INTERNAL#" % i18nc("Export step", "Export transactions"), nb);
        for (int i = 0; !err && i < nb; ++i) {
            SKGOperationObject op(objects.at(i));
            err = exportOperation(op, doc, transactions);
            IFOKDO(err, m_importer->getDocument()->stepForward(i + 1))
        }

        SKGENDTRANSACTION(m_importer->getDocument(),  err)
    }

    // <KEYVALUEPAIRS>
    QDomElement keyvaluepairs = doc.createElement(QLatin1String("KEYVALUEPAIRS"));
    root.appendChild(keyvaluepairs);
    {
        QDomElement pair = doc.createElement(QLatin1String("PAIR"));
        keyvaluepairs.appendChild(pair);
        pair.setAttribute(QLatin1String("key"), QLatin1String("kmm-baseCurrency"));
        pair.setAttribute(QLatin1String("value"), stdUnit);
    }
    return err;
}

SKGError SKGImportPluginKmy::exportPayees(QDomDocument& doc, QDomElement& root)
{
    SKGError err;
    QDomElement payees = doc.createElement(QLatin1String("PAYEES"));
    root.appendChild(payees);

    SKGObjectBase::SKGListSKGObjectBase listPayees;
    IFOKDO(err, m_importer->getDocument()->getObjects(QLatin1String("v_payee"), QString(), listPayees))
    int nb = listPayees.count();
    payees.setAttribute(QLatin1String("count"), SKGServices::intToString(nb));
    IFOK(err) {
        err = m_importer->getDocument()->beginTransaction("#INTERNAL#" % i18nc("Export step", "Export payees"), nb);
        for (int i = 0; !err && i < nb; ++i) {
            SKGPayeeObject payeeObject(listPayees.at(i));
            QDomElement payee = doc.createElement(QLatin1String("PAYEE"));
            payees.appendChild(payee);

            payee.setAttribute(QLatin1String("matchingenabled"), QLatin1String("0"));
            payee.setAttribute(QLatin1String("id"), getKmyUniqueIdentifier(payeeObject));
            payee.setAttribute(QLatin1String("name"), payeeObject.getName());
            payee.setAttribute(QLatin1String("email"), QString());
            payee.setAttribute(QLatin1String("reference"), QString());

            QDomElement address = doc.createElement(QLatin1String("ADDRESS"));
            payee.appendChild(address);

            address.setAttribute(QLatin1String("street"), payeeObject.getAddress());
            address.setAttribute(QLatin1String("postcode"), QString());
            address.setAttribute(QLatin1String("zip"), QString());
            address.setAttribute(QLatin1String("city"), QString());
            address.setAttribute(QLatin1String("telephone"), QString());
            address.setAttribute(QLatin1String("state"), QString());

            m_mapIdPayee[SKGServices::intToString(i + 1)] = payeeObject;

            IFOKDO(err, m_importer->getDocument()->stepForward(i + 1))
        }

        SKGENDTRANSACTION(m_importer->getDocument(),  err)
    }
    return err;
}

SKGError SKGImportPluginKmy::exportInstitutions(QDomDocument& doc, QDomElement& root)
{
    SKGError err;
    QDomElement institutions = doc.createElement(QLatin1String("INSTITUTIONS"));
    root.appendChild(institutions);
    SKGObjectBase::SKGListSKGObjectBase objects;
    IFOKDO(err, m_importer->getDocument()->getObjects(QLatin1String("v_bank"), QLatin1String("EXISTS(SELECT 1 FROM account WHERE account.rd_bank_id=v_bank.id)"), objects))
    int nb = objects.count();
    institutions.setAttribute(QLatin1String("count"), SKGServices::intToString(nb));
    IFOK(err) {
        err = m_importer->getDocument()->beginTransaction("#INTERNAL#" % i18nc("Export step", "Export banks"), nb);
        for (int i = 0; !err && i < nb; ++i) {
            SKGBankObject obj(objects.at(i));
            QDomElement institution = doc.createElement(QLatin1String("INSTITUTION"));
            institutions.appendChild(institution);

            institution.setAttribute(QLatin1String("id"), getKmyUniqueIdentifier(obj));
            institution.setAttribute(QLatin1String("name"), obj.getName());
            institution.setAttribute(QLatin1String("sortcode"), obj.getNumber());
            institution.setAttribute(QLatin1String("manager"), QString());

            QDomElement address = doc.createElement(QLatin1String("ADDRESS"));
            institution.appendChild(address);

            address.setAttribute(QLatin1String("street"), QString());
            address.setAttribute(QLatin1String("zip"), QString());
            address.setAttribute(QLatin1String("city"), QString());
            address.setAttribute(QLatin1String("telephone"), QString());

            QDomElement accountids = doc.createElement(QLatin1String("ACCOUNTIDS"));
            institution.appendChild(accountids);

            SKGObjectBase::SKGListSKGObjectBase accounts;
            err = obj.getAccounts(accounts);
            int nb2 = accounts.count();
            for (int j = 0; !err && j < nb2; ++j) {
                QDomElement accountid = doc.createElement(QLatin1String("ACCOUNTID"));
                accountids.appendChild(accountid);

                accountid.setAttribute(QLatin1String("id"), getKmyUniqueIdentifier(accounts.at(j)));
            }
            IFOKDO(err, m_importer->getDocument()->stepForward(i + 1))
        }

        SKGENDTRANSACTION(m_importer->getDocument(),  err)
    }

    return err;
}

SKGError SKGImportPluginKmy::exportCategories(QDomDocument& doc, QDomElement& accounts, const QString& stdUnit, QDomElement& accountIncome, QDomElement& accountExpense, int nbAccount)
{
    // The v_category_display are retrieved to improve performances of getCurrentAmount
    SKGError err;
    SKGObjectBase::SKGListSKGObjectBase objects;
    IFOKDO(err, m_importer->getDocument()->getObjects(QLatin1String("v_category_display"), QString(), objects))
    accounts.setAttribute(QLatin1String("count"), SKGServices::intToString(5 + nbAccount + objects.count()));
    int nb = objects.count();
    IFOK(err) {
        err = m_importer->getDocument()->beginTransaction("#INTERNAL#" % i18nc("Export step", "Export categories"), nb);
        for (int i = 0; !err && i < nb; ++i) {
            SKGCategoryObject obj(objects.at(i));
            QDomElement account = doc.createElement(QLatin1String("ACCOUNT"));
            accounts.appendChild(account);

            account.setAttribute(QLatin1String("id"), getKmyUniqueIdentifier(obj));
            account.setAttribute(QLatin1String("name"), obj.getName());
            account.setAttribute(QLatin1String("number"), QString());
            account.setAttribute(QLatin1String("type"), obj.getCurrentAmount() < 0 ? QLatin1String("13") : QLatin1String("12"));

            account.setAttribute(QLatin1String("institution"), QString());

            SKGCategoryObject parentCat;
            obj.getParentCategory(parentCat);

            QString parentId = (parentCat.getID() != 0 ? getKmyUniqueIdentifier(parentCat) : (obj.getCurrentAmount() < 0 ? QLatin1String("AStd::Expense") : QLatin1String("AStd::Income")));
            if (parentId == QLatin1String("AStd::Expense")) {
                QDomElement subaccount = doc.createElement(QLatin1String("SUBACCOUNT"));
                accountExpense.appendChild(subaccount);
                subaccount.setAttribute(QLatin1String("id"), getKmyUniqueIdentifier(obj));
            } else if (parentId == QLatin1String("AStd::Income")) {
                QDomElement subaccount = doc.createElement(QLatin1String("SUBACCOUNT"));
                accountIncome.appendChild(subaccount);
                subaccount.setAttribute(QLatin1String("id"), getKmyUniqueIdentifier(obj));
            }

            account.setAttribute(QLatin1String("parentaccount"), parentId);
            account.setAttribute(QLatin1String("lastmodified"), QString());
            account.setAttribute(QLatin1String("lastreconciled"), QString());
            account.setAttribute(QLatin1String("opened"), QString());
            account.setAttribute(QLatin1String("currency"), stdUnit);
            account.setAttribute(QLatin1String("description"), QString());

            QDomElement subaccounts = doc.createElement(QLatin1String("SUBACCOUNTS"));
            account.appendChild(subaccounts);

            SKGObjectBase::SKGListSKGObjectBase categories;
            IFOKDO(err, obj.getCategories(categories))
            int nb2 = categories.count();
            for (int j = 0; !err && j < nb2; ++j) {
                QDomElement subaccount = doc.createElement(QLatin1String("SUBACCOUNT"));
                subaccounts.appendChild(subaccount);

                subaccount.setAttribute(QLatin1String("id"), getKmyUniqueIdentifier(categories.at(j)));
            }
            IFOKDO(err, m_importer->getDocument()->stepForward(i + 1))
        }

        SKGENDTRANSACTION(m_importer->getDocument(),  err)
    }
    return err;
}

SKGError SKGImportPluginKmy::exportAccounts(QDomDocument& doc, QDomElement& root, const QString& stdUnit, QDomElement& accounts, QDomElement& accountIncome, QDomElement& accountExpense, int& nbAccounts)
{
    SKGError err;
    accounts = doc.createElement(QLatin1String("ACCOUNTS"));
    root.appendChild(accounts);

    QDomElement accountAsset;
    {
        QDomElement account = doc.createElement(QLatin1String("ACCOUNT"));
        accounts.appendChild(account);

        account.setAttribute(QLatin1String("id"), QLatin1String("AStd::Equity"));
        account.setAttribute(QLatin1String("name"), QLatin1String("Equity"));
        account.setAttribute(QLatin1String("number"), QString());
        account.setAttribute(QLatin1String("type"), QLatin1String("16"));
        account.setAttribute(QLatin1String("institution"), QString());
        account.setAttribute(QLatin1String("parentaccount"), QString());
        account.setAttribute(QLatin1String("lastmodified"), QString());
        account.setAttribute(QLatin1String("lastreconciled"), QString());
        account.setAttribute(QLatin1String("opened"), QString());
        account.setAttribute(QLatin1String("currency"), stdUnit);
        account.setAttribute(QLatin1String("description"), QString());

        QDomElement subaccounts = doc.createElement(QLatin1String("SUBACCOUNTS"));
        account.appendChild(subaccounts);
    }

    {
        QDomElement account = doc.createElement(QLatin1String("ACCOUNT"));
        accounts.appendChild(account);

        account.setAttribute(QLatin1String("id"), QLatin1String("AStd::Asset"));
        account.setAttribute(QLatin1String("name"), QLatin1String("Asset"));
        account.setAttribute(QLatin1String("number"), QString());
        account.setAttribute(QLatin1String("type"), QLatin1String("9"));
        account.setAttribute(QLatin1String("institution"), QString());
        account.setAttribute(QLatin1String("parentaccount"), QString());
        account.setAttribute(QLatin1String("lastmodified"), QString());
        account.setAttribute(QLatin1String("lastreconciled"), QString());
        account.setAttribute(QLatin1String("opened"), QString());
        account.setAttribute(QLatin1String("currency"), stdUnit);
        account.setAttribute(QLatin1String("description"), QString());

        QDomElement subaccounts = doc.createElement(QLatin1String("SUBACCOUNTS"));
        account.appendChild(subaccounts);
        accountAsset = subaccounts;
    }

    {
        QDomElement account = doc.createElement(QLatin1String("ACCOUNT"));
        accounts.appendChild(account);

        account.setAttribute(QLatin1String("id"), QLatin1String("AStd::Liability"));
        account.setAttribute(QLatin1String("name"), QLatin1String("Liability"));
        account.setAttribute(QLatin1String("number"), QString());
        account.setAttribute(QLatin1String("type"), QLatin1String("10"));
        account.setAttribute(QLatin1String("institution"), QString());
        account.setAttribute(QLatin1String("parentaccount"), QString());
        account.setAttribute(QLatin1String("lastmodified"), QString());
        account.setAttribute(QLatin1String("lastreconciled"), QString());
        account.setAttribute(QLatin1String("opened"), QString());
        account.setAttribute(QLatin1String("currency"), stdUnit);
        account.setAttribute(QLatin1String("description"), QString());

        QDomElement subaccounts = doc.createElement(QLatin1String("SUBACCOUNTS"));
        account.appendChild(subaccounts);
    }

    {
        QDomElement account = doc.createElement(QLatin1String("ACCOUNT"));
        accounts.appendChild(account);

        account.setAttribute(QLatin1String("id"), QLatin1String("AStd::Income"));
        account.setAttribute(QLatin1String("name"), QLatin1String("Income"));
        account.setAttribute(QLatin1String("number"), QString());
        account.setAttribute(QLatin1String("type"), QLatin1String("12"));
        account.setAttribute(QLatin1String("institution"), QString());
        account.setAttribute(QLatin1String("parentaccount"), QString());
        account.setAttribute(QLatin1String("lastmodified"), QString());
        account.setAttribute(QLatin1String("lastreconciled"), QString());
        account.setAttribute(QLatin1String("opened"), QString());
        account.setAttribute(QLatin1String("currency"), stdUnit);
        account.setAttribute(QLatin1String("description"), QString());

        QDomElement subaccounts = doc.createElement(QLatin1String("SUBACCOUNTS"));
        account.appendChild(subaccounts);
        accountIncome = subaccounts;
    }

    {
        QDomElement account = doc.createElement(QLatin1String("ACCOUNT"));
        accounts.appendChild(account);

        account.setAttribute(QLatin1String("id"), QLatin1String("AStd::Expense"));
        account.setAttribute(QLatin1String("name"), QLatin1String("Expense"));
        account.setAttribute(QLatin1String("number"), QString());
        account.setAttribute(QLatin1String("type"), QLatin1String("13"));
        account.setAttribute(QLatin1String("institution"), QString());
        account.setAttribute(QLatin1String("parentaccount"), QString());
        account.setAttribute(QLatin1String("lastmodified"), QString());
        account.setAttribute(QLatin1String("lastreconciled"), QString());
        account.setAttribute(QLatin1String("opened"), QString());
        account.setAttribute(QLatin1String("currency"), stdUnit);
        account.setAttribute(QLatin1String("description"), QString());

        QDomElement subaccounts = doc.createElement(QLatin1String("SUBACCOUNTS"));
        account.appendChild(subaccounts);
        accountExpense = subaccounts;
    }

    SKGObjectBase::SKGListSKGObjectBase objects;
    IFOKDO(err, m_importer->getDocument()->getObjects(QLatin1String("v_account"), QString(), objects))
    int nb = objects.count();
    IFOK(err) {
        err = m_importer->getDocument()->beginTransaction("#INTERNAL#" % i18nc("Export step", "Export accounts"), nb);
        for (int i = 0; !err && i < nb; ++i) {
            SKGAccountObject obj(objects.at(i));
            QDomElement account = doc.createElement(QLatin1String("ACCOUNT"));
            accounts.appendChild(account);

            account.setAttribute(QLatin1String("id"), getKmyUniqueIdentifier(obj));
            account.setAttribute(QLatin1String("name"), obj.getName());
            account.setAttribute(QLatin1String("number"), obj.getNumber());
            account.setAttribute(QLatin1String("type"), obj.getType() == SKGAccountObject::CREDITCARD ? QLatin1String("4") :
                                 (obj.getType() == SKGAccountObject::INVESTMENT ? QLatin1String("7") :
                                  (obj.getType() == SKGAccountObject::ASSETS ? QLatin1String("9") :
                                   (obj.getType() == SKGAccountObject::WALLET ? QLatin1String("3") :
                                    (obj.getType() == SKGAccountObject::LOAN ? QLatin1String("10") :
                                     QLatin1String("1"))))));

            SKGBankObject bank;
            err = obj.getBank(bank);
            account.setAttribute(QLatin1String("institution"), getKmyUniqueIdentifier(bank));

            account.setAttribute(QLatin1String("parentaccount"), QLatin1String("AStd::Asset"));
            account.setAttribute(QLatin1String("lastmodified"), QString());
            account.setAttribute(QLatin1String("lastreconciled"), QString());
            account.setAttribute(QLatin1String("opened"), QString());
            SKGUnitObject unit;
            obj.getUnit(unit);
            QString unitS = SKGUnitObject::getInternationalCode(unit.getName());
            if (unitS.isEmpty()) {
                unitS = QLatin1String("EUR");
            }
            account.setAttribute(QLatin1String("currency"), unitS);
            account.setAttribute(QLatin1String("description"), QString());

            // Bookmarked account
            QDomElement keyvaluepairs = doc.createElement(QLatin1String("KEYVALUEPAIRS"));
            account.appendChild(keyvaluepairs);
            if (obj.isBookmarked()) {
                QDomElement pair = doc.createElement(QLatin1String("PAIR"));
                keyvaluepairs.appendChild(pair);
                pair.setAttribute(QLatin1String("key"), QLatin1String("PreferredAccount"));
                pair.setAttribute(QLatin1String("value"), QLatin1String("Yes"));
            }
            // Closed account
            if (obj.isClosed()) {
                QDomElement pair = doc.createElement(QLatin1String("PAIR"));
                keyvaluepairs.appendChild(pair);
                pair.setAttribute(QLatin1String("key"), QLatin1String("mm-closed"));
                pair.setAttribute(QLatin1String("value"), QLatin1String("yes"));
            }

            // Maximum and minimum limits
            if (obj.isMaxLimitAmountEnabled()) {
                QDomElement pair = doc.createElement(QLatin1String("PAIR"));
                keyvaluepairs.appendChild(pair);
                pair.setAttribute(QLatin1String("key"), QLatin1String("maxCreditAbsolute"));
                pair.setAttribute(QLatin1String("value"), kmyValue(-obj.getMaxLimitAmount()));
            }
            if (obj.isMinLimitAmountEnabled()) {
                QDomElement pair = doc.createElement(QLatin1String("PAIR"));
                keyvaluepairs.appendChild(pair);
                pair.setAttribute(QLatin1String("key"), QLatin1String("minBalanceAbsolute"));
                pair.setAttribute(QLatin1String("value"), kmyValue(obj.getMinLimitAmount()));
            }

            // Add it in asset
            QDomElement subaccount = doc.createElement(QLatin1String("SUBACCOUNT"));
            accountAsset.appendChild(subaccount);
            subaccount.setAttribute(QLatin1String("id"), getKmyUniqueIdentifier(obj));

            IFOKDO(err, m_importer->getDocument()->stepForward(i + 1))
        }

        SKGENDTRANSACTION(m_importer->getDocument(),  err)
    }

    nbAccounts = nb;
    return err;
}

SKGError SKGImportPluginKmy::exportFile()
{
    // Initialisation
    m_opTreated.clear();

    if (m_importer->getDocument() == nullptr) {
        return SKGError(ERR_ABORT, i18nc("Error message", "Invalid parameters"));
    }
    SKGError err;
    SKGTRACEINFUNCRC(2, err)

    // Open file
    KCompressionDevice file(m_importer->getLocalFileName(false), KCompressionDevice::GZip);
    if (!file.open(QIODevice::WriteOnly)) {
        err.setReturnCode(ERR_INVALIDARG).setMessage(i18nc("Error message",  "Save file '%1' failed", m_importer->getFileName().toDisplayString()));
    } else {
        QDomDocument doc(QLatin1String("KMYMONEY-FILE"));
        QDomComment comment = doc.createComment(QLatin1String("Generated by libskgbankmodeler"));
        doc.appendChild(comment);

        QDomElement root = doc.createElement(QLatin1String("KMYMONEY-FILE"));
        doc.appendChild(root);
        {
            err = m_importer->getDocument()->beginTransaction("#INTERNAL#" % i18nc("Export step", "Export %1 file", "KMY"), 8);
            IFOK(err) {
                // Step 1-<FILEINFO>
                IFOKDO(err, exportHeader(doc, root))
                IFOKDO(err, m_importer->getDocument()->stepForward(1))

                // // Step 2-<INSTITUTIONS>
                IFOKDO(err, exportInstitutions(doc, root))
                IFOKDO(err, m_importer->getDocument()->stepForward(2))

                // Step 3-<PAYEES>
                IFOKDO(err, exportPayees(doc, root))
                IFOKDO(err, m_importer->getDocument()->stepForward(3))

                // Step 4-<ACCOUNTS>
                // Std accounts
                QString stdUnit = SKGUnitObject::getInternationalCode(m_importer->getDocument()->getPrimaryUnit().Name);
                if (stdUnit.isEmpty()) {
                    stdUnit = QLatin1String("EUR");
                }

                QDomElement accountIncome;
                QDomElement accountExpense;
                QDomElement accounts;

                int nbAccounts = 0;
                IFOKDO(err, exportAccounts(doc, root, stdUnit, accounts, accountIncome, accountExpense, nbAccounts))
                IFOKDO(err, m_importer->getDocument()->stepForward(4))

                // Step 5
                IFOKDO(err, exportCategories(doc, accounts, stdUnit, accountIncome, accountExpense, nbAccounts))
                IFOKDO(err, m_importer->getDocument()->stepForward(5))

                // Step 6-<TRANSACTIONS>
                IFOKDO(err, exportTransactions(doc, root, stdUnit))

                // Step 6-<SCHEDULES>
                IFOKDO(err, exportSchedules(doc, root))
                IFOKDO(err, m_importer->getDocument()->stepForward(6))

                // Step 7-<BUDGETS>
                IFOKDO(err, exportBudgets(doc, root))
                IFOKDO(err, m_importer->getDocument()->stepForward(7))

                // Step 8-<SECURITIES> and <CURRENCIES>
                IFOKDO(err, exportSecurities(doc, root, stdUnit))

                // Save file
                IFOK(err) {
                    file.write(doc.toString().toUtf8());
                }
                IFOKDO(err, m_importer->getDocument()->stepForward(8))
            }

            SKGENDTRANSACTION(m_importer->getDocument(),  err)
        }

        file.close();
    }

    // Clean
    m_opTreated.clear();

    return err;
}

SKGError SKGImportPluginKmy::exportOperation(const SKGOperationObject& iOperation, QDomDocument& iDoc, QDomElement& iTransaction)
{
    SKGError err;
    SKGTRACEINFUNCRC(2, err)
    if (!m_opTreated.contains(getKmyUniqueIdentifier(iOperation))) {
        QDomElement transaction = iDoc.createElement(QLatin1String("TRANSACTION"));
        iTransaction.appendChild(transaction);

        SKGUnitObject unit;
        iOperation.getUnit(unit);

        QString date = iOperation.getAttribute(QLatin1String("d_date"));
        transaction.setAttribute(QLatin1String("id"), getKmyUniqueIdentifier(iOperation));
        transaction.setAttribute(QLatin1String("entrydate"), date);
        transaction.setAttribute(QLatin1String("postdate"), date);
        transaction.setAttribute(QLatin1String("memo"), iOperation.getComment());
        transaction.setAttribute(QLatin1String("commodity"), SKGUnitObject::getInternationalCode(unit.getName()));

        QString reconcileflag = (iOperation.getStatus() == SKGOperationObject::MARKED ? QLatin1String("1") : (iOperation.getStatus() == SKGOperationObject::CHECKED ? QLatin1String("2") : QLatin1String("0")));

        SKGAccountObject act;
        IFOKDO(err, iOperation.getParentAccount(act))

        QDomElement splits = iDoc.createElement(QLatin1String("SPLITS"));
        transaction.appendChild(splits);

        QDomElement split = iDoc.createElement(QLatin1String("SPLIT"));
        splits.appendChild(split);

        SKGPayeeObject payeeObject;
        iOperation.getPayee(payeeObject);
        QString payeeId = (payeeObject.getID() != 0 ? getKmyUniqueIdentifier(payeeObject) : QString());

        int indexSubOp = 1;

        // Split for account
        split.setAttribute(QLatin1String("payee"), payeeId);
        split.setAttribute(QLatin1String("reconciledate"), QString());
        split.setAttribute(QLatin1String("id"), "S" % SKGServices::intToString(indexSubOp++).rightJustified(4, '0'));
        double val2 = SKGServices::stringToDouble(iOperation.getAttribute(QLatin1String("f_QUANTITY")));
        split.setAttribute(QLatin1String("shares"), SKGImportPluginKmy::kmyValue(val2));
        split.setAttribute(QLatin1String("action"), QString());
        split.setAttribute(QLatin1String("bankid"), QString());
        split.setAttribute(QLatin1String("number"), iOperation.getNumber());
        split.setAttribute(QLatin1String("reconcileflag"), reconcileflag);
        split.setAttribute(QLatin1String("memo"), iOperation.getComment());
        QString originalAmount = iOperation.getProperty(QLatin1String("SKG_OP_ORIGINAL_AMOUNT"));
        if (!originalAmount.isEmpty()) {
            val2 = qAbs(SKGServices::stringToDouble(originalAmount)) * (val2 / qAbs(val2));
        }
        split.setAttribute(QLatin1String("value"), SKGImportPluginKmy::kmyValue(val2));
        split.setAttribute(QLatin1String("account"), getKmyUniqueIdentifier(act));

        SKGOperationObject obj2;
        if (!err && iOperation.isTransfer(obj2)) {
            // It is a Transfer
            QString reconcileflag2 = (obj2.getStatus() == SKGOperationObject::MARKED ? QLatin1String("1") : (obj2.getStatus() == SKGOperationObject::CHECKED ? QLatin1String("2") : QLatin1String("0")));

            SKGAccountObject act2;
            IFOKDO(err, obj2.getParentAccount(act2))

            QDomElement split2 = iDoc.createElement(QLatin1String("SPLIT"));
            splits.appendChild(split2);

            // Split for account
            val2 = -val2;
            split2.setAttribute(QLatin1String("payee"), payeeId);
            split2.setAttribute(QLatin1String("reconciledate"), QString());
            split2.setAttribute(QLatin1String("id"), "S" % SKGServices::intToString(indexSubOp++).rightJustified(4, '0'));
            split2.setAttribute(QLatin1String("shares"), SKGImportPluginKmy::kmyValue(SKGServices::stringToDouble(obj2.getAttribute(QLatin1String("f_QUANTITY")))));
            split2.setAttribute(QLatin1String("action"), QString());
            split2.setAttribute(QLatin1String("bankid"), QString());
            split2.setAttribute(QLatin1String("number"), obj2.getNumber());
            split2.setAttribute(QLatin1String("reconcileflag"), reconcileflag2);
            split2.setAttribute(QLatin1String("memo"), obj2.getComment());
            split2.setAttribute(QLatin1String("value"), SKGImportPluginKmy::kmyValue(val2));
            split2.setAttribute(QLatin1String("account"), getKmyUniqueIdentifier(act2));

            m_opTreated.insert(getKmyUniqueIdentifier(obj2));
        } else {
            SKGObjectBase::SKGListSKGObjectBase subops;
            IFOKDO(err, iOperation.getSubOperations(subops))
            int nb2 = subops.count();
            for (int j = 0; !err && j < nb2; ++j) {
                QDomElement split2 = iDoc.createElement(QLatin1String("SPLIT"));
                splits.appendChild(split2);

                SKGSubOperationObject subop(subops.at(j));
                SKGCategoryObject cat;
                subop.getCategory(cat);
                split2.setAttribute(QLatin1String("payee"), payeeId);
                split2.setAttribute(QLatin1String("reconciledate"), QString());
                split2.setAttribute(QLatin1String("id"), "S" % SKGServices::intToString(indexSubOp++).rightJustified(4, '0'));
                QString shape3 = SKGImportPluginKmy::kmyValue(-subop.getQuantity());
                split2.setAttribute(QLatin1String("shares"), shape3);
                split2.setAttribute(QLatin1String("action"), QString());
                split2.setAttribute(QLatin1String("bankid"), QString());
                split2.setAttribute(QLatin1String("number"), iOperation.getNumber());
                split2.setAttribute(QLatin1String("reconcileflag"), reconcileflag);
                split2.setAttribute(QLatin1String("memo"), subop.getComment());
                split2.setAttribute(QLatin1String("value"), shape3);
                split2.setAttribute(QLatin1String("account"), date == QLatin1String("0000-00-00") ? QLatin1String("AStd::Equity") : (cat.getID() != 0 ? getKmyUniqueIdentifier(cat) : QString()));
            }
        }

        m_opTreated.insert(getKmyUniqueIdentifier(iOperation));
    }
    return err;
}

QString SKGImportPluginKmy::kmyValue(double iValue)
{
    QString output;
    for (int i = 0; output.isEmpty() && i < 11; ++i) {
        QString d = SKGServices::doubleToString(pow(10, i) * iValue);
        if (d.indexOf(QLatin1Char('.')) == -1) {
            output = d % QLatin1Char('/') % SKGServices::intToString(qPow(10, i));
        }
    }
    return output;
}

double SKGImportPluginKmy::toKmyValue(const QString& iString)
{
    double output = 0;
    QStringList vals = SKGServices::splitCSVLine(iString, '/');
    if (vals.count() == 1) {
        output = SKGServices::stringToDouble(vals.at(0));
    } else if (vals.count() == 2) {
        output = SKGServices::stringToDouble(vals.at(0)) / SKGServices::stringToDouble(vals.at(1));
    }
    return output;
}

QString SKGImportPluginKmy::getKmyUniqueIdentifier(const SKGObjectBase& iObject)
{
    QString id;
    if (iObject.getID() != 0) {
        QString table = iObject.getRealTable();
        if (table == QLatin1String("operation") || table == QLatin1String("suboperation")) {
            // T000000000000003623
            id = QLatin1Char('T') % SKGServices::intToString(iObject.getID()).rightJustified(18, '0');
        } else if (table == QLatin1String("payee")) {
            // P000030
            id = QLatin1Char('P') % SKGServices::intToString(iObject.getID()).rightJustified(6, '0');
        } else {
            id = iObject.getUniqueID();
        }
    }
    return id;
}

QString SKGImportPluginKmy::getMimeTypeFilter() const
{
    return "*.kmy|" % i18nc("A file format", "KMyMoney document");
}

#include <skgimportpluginkmy.moc>
