/***************************************************************************
                               kstviewline.cpp
                             -------------------
    begin                : Jun 14, 2005
    copyright            : (C) 2005 The University of Toronto
    email                :
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/ 
   
#include "kstaccessibility.h" 
#include "kstviewline.h"

#include <klocale.h>

#include <qmetaobject.h>
#include <qpainter.h>
#include <qvariant.h>

KstViewLine::KstViewLine()
: KstViewObject("Line"), _width(1) {
  setTransparent(true);
  _capStyle = Qt::FlatCap; 
  _penStyle = Qt::SolidLine; 
  _layoutActions |= Delete | Raise | Lower | RaiseToTop | LowerToBottom | Rename | MoveTo | Copy | CopyTo;
  setMinimumSize(QSize(0,0));
}


KstViewLine::KstViewLine(const QString& type)
: KstViewObject(type), _width(1) {
  setTransparent(true);
  _capStyle = Qt::FlatCap; 
  _penStyle = Qt::SolidLine; 
  _layoutActions |= Delete | Raise | Lower | RaiseToTop | LowerToBottom | Rename | MoveTo | Copy | CopyTo;
  setMinimumSize(QSize(0,0));
}


KstViewLine::KstViewLine(const QDomElement& e)
: KstViewObject(e) {
 
  _width = 1;
  _capStyle = Qt::FlatCap; 
  _penStyle = Qt::SolidLine;
 
  // always these values
  setTransparent(true);
  _type = "Line";
  _layoutActions |= Delete | Raise | Lower | RaiseToTop | LowerToBottom | Rename | MoveTo | Copy | CopyTo;
  setMinimumSize(QSize(0,0));
}


KstViewLine::~KstViewLine() {
}


void KstViewLine::paint(KstPaintType type, QPainter& p, const QRegion& bounds) {
  // figure out which direction to draw the line
  QPen pen(_foregroundColor, _width);
  pen.setCapStyle(_capStyle);
  pen.setStyle(_penStyle);
  QPen oldPen = p.pen();
  p.setPen(pen);

  switch (_orientation) {
    case UpLeft:
      p.drawLine(geometry().bottomRight(), geometry().topLeft());
      break;
    case UpRight:
      p.drawLine(geometry().bottomLeft(), geometry().topRight());
      break;
    case DownLeft:
      p.drawLine(geometry().topRight(), geometry().bottomLeft());
      break;
    case DownRight:
      p.drawLine(geometry().topLeft(), geometry().bottomRight());
      break;  
  }
  // paint this last
  p.setPen(oldPen);
  KstViewObject::paint(type, p, bounds); 
}


void KstViewLine::save(QTextStream& ts, const QString& indent) {
  ts << indent << "<" << type() << ">" << endl;
  // and save the orientation
  ts << indent + "  " << "<orientation>" << _orientation << "</orientation>" << endl;
  KstViewObject::save(ts, indent + "  ");
  ts << indent << "</" << type() << ">" << endl;
}


void KstViewLine::setFrom(const QPoint& from) {
  _from = from;
}


QPoint KstViewLine::from() const {
  switch (_orientation) {
    case DownRight:
      return geometry().topLeft();
    case DownLeft:
      return geometry().topRight();
    case UpRight:
      return geometry().bottomLeft();
    case UpLeft:
      return geometry().bottomRight();
  }
  return QPoint(-1,-1);
}


void KstViewLine::setTo(const QPoint& to) {
  _to = to;
  // line drawing finished; update size, origin, and orientation
  updateOrientation();
}


QPoint KstViewLine::to() const {
  switch (_orientation) {
    case DownRight:
      return geometry().bottomRight();
    case DownLeft:
      return geometry().bottomLeft();
    case UpRight:
      return geometry().topRight();
    case UpLeft:
      return geometry().topLeft();
  }
  return QPoint(-1,-1);
}


void KstViewLine::setWidth(int width) {
  _width = width;
  setDirty();
}


int KstViewLine::width() const {
  return _width;
}


void KstViewLine::setCapStyle(Qt::PenCapStyle style) {
  if (_capStyle != style) {
    setDirty();
  }
  _capStyle = style;
}


Qt::PenCapStyle KstViewLine::capStyle() const {
  return _capStyle;
}


void KstViewLine::setPenStyle(Qt::PenStyle style) {
  if (_penStyle != style) {
    setDirty();
  }
  _penStyle = style;
}


Qt::PenStyle KstViewLine::penStyle() const {
  return _penStyle;
}


void KstViewLine::updateOrientation() {
  if (_from.x() < _to.x()) {
    if (_from.y() < _to.y()) {
      _orientation = DownRight;  
      move(_from);
      resize(QSize(_to.x() - _from.x() + 1, _to.y() - _from.y() + 1));
    } else {
      _orientation = UpRight;  
      move(QPoint(_from.x(), _to.y()));
      resize(QSize(_to.x() - _from.x() + 1, _from.y() - _to.y() + 1));
    }
  } else {
    if (_from.y() < _to.y()) {
      _orientation = DownLeft;  
      move(QPoint(_to.x(), _from.y()));
      resize(QSize(_from.x() - _to.x() + 1, _to.y() - _from.y() + 1));
    } else {
      _orientation = UpLeft;  
      move(_to);
      resize(QSize(_from.x() - _to.x() + 1, _from.y() - _to.y() + 1));
    }
  }
}


void KstViewLine::drawFocusRect(QPainter& p) {
  // draw the hotpoints
  QPoint point1, point2;
  
  int dx = KST_RESIZE_BORDER_W/2;
  
  if (_orientation == UpLeft || _orientation == DownRight) {
    point1 = QPoint(geometry().left() - dx, geometry().top() - dx);
    point2 = QPoint(geometry().right() - dx, geometry().bottom() - dx);
  } else {
    point1 = QPoint(geometry().right() - dx, geometry().top() - dx);
    point2 = QPoint(geometry().left() - dx, geometry().bottom() - dx);
  }
  QRect rect;
  rect.setSize(QSize(2*dx+1,2*dx+1));
  rect.moveTopLeft(point1);
  p.drawRect(rect);
  rect.moveTopLeft(point2);
  p.drawRect(rect);
}


void KstViewLine::drawSelectRect(QPainter& p) {
  p.setPen(QPen(Qt::black, 0));
  p.setBrush(QBrush(Qt::green, Qt::SolidPattern));
  drawFocusRect(p);
}


inline bool linePointsCloseEnough(const QPoint& point1, const QPoint& point2) {
  const int dx = KST_RESIZE_BORDER_W/2;
  return point1.x() <= point2.x() + dx &&
      point1.x() >= point2.x() - dx &&
      point1.y() <= point2.y() + dx &&
      point1.y() >= point2.y() - dx;
}


signed int KstViewLine::directionFor(const QPoint& pos) {
 
  if (!isSelected()) {
    return NONE;  
  }
  
  signed int direction = 0;
  // FIXME: use a switch to avoid too many evaluations
  if ((_orientation == UpLeft && 
       linePointsCloseEnough(pos, geometry().topLeft())) ||
      (_orientation == DownRight &&
       linePointsCloseEnough(pos, geometry().bottomRight())) ||
      (_orientation == UpRight &&
       linePointsCloseEnough(pos, geometry().topRight())) ||
      (_orientation == DownLeft &&
       linePointsCloseEnough(pos, geometry().bottomLeft()))) {
    direction = direction | ENDPOINT | DOWN; 
  } else if ((_orientation == UpLeft && 
              linePointsCloseEnough(pos, geometry().bottomRight())) ||
             (_orientation == DownRight &&
              linePointsCloseEnough(pos, geometry().topLeft())) ||
             (_orientation == UpRight &&
              linePointsCloseEnough(pos, geometry().bottomLeft())) ||
             (_orientation == DownLeft &&
              linePointsCloseEnough(pos, geometry().topRight()))) {
    direction = direction | ENDPOINT | UP;          
  }
  return direction;
}


QMap<QString, QVariant> KstViewLine::widgetHints(const QString& propertyName) const {
  QMap<QString, QVariant> map = KstViewObject::widgetHints(propertyName);
  if (!map.empty()) {
    return map;  
  }
  
  if (propertyName == "width") {
    map.insert(QString("_kst_widgetType"), QString("QSpinBox"));
    map.insert(QString("_kst_label"), i18n("Line width"));  
    map.insert(QString("minValue"), 0);
  } else if (propertyName == "lineStyle") {
    map.insert(QString("_kst_widgetType"), QString("PenStyleWidget"));
    map.insert(QString("_kst_label"), i18n("Line style"));  
  } else if (propertyName == "lineColor") {
    map.insert(QString("_kst_widgetType"), QString("KColorButton"));
    map.insert(QString("_kst_label"), i18n("Line color"));    
  }
  return map;
}


int KstViewLine::penStyleWrap() const {
  switch (penStyle()) {
    case Qt::SolidLine:
      return 0;
    case Qt::DashLine:
      return 1;
    case Qt::DotLine:
      return 2;
    case Qt::DashDotLine:
      return 3;
    case Qt::DashDotDotLine:
      return 4;
    default:
      return 0;  
  }
  return 0;
}


void KstViewLine::setPenStyleWrap(int style) {
  switch (style) {
    case 0:
      setPenStyle(Qt::SolidLine);
      break;
    case 1:
      setPenStyle(Qt::DashLine);
      break;
    case 2:
      setPenStyle(Qt::DotLine);
      break;
    case 3:
      setPenStyle(Qt::DashDotLine);
      break;
    case 4:
      setPenStyle(Qt::DashDotDotLine);
      break;  
    default:
      setPenStyle(Qt::SolidLine);
      break;
  }
}


void KstViewLine::setForegroundColor(const QColor& color) {
  KstViewObject::setForegroundColor(color);  
}


QColor KstViewLine::foregroundColor() const {
  return KstViewObject::foregroundColor();  
}


void KstViewLine::drawShadow(QPainter& p, const QPoint& pos) {
  QPoint point1, point2;
  QRect rect = geometry();
  rect.moveTopLeft(pos);
  if (_orientation == UpLeft || _orientation == DownRight) {
    point1 = pos;
    point2 = rect.bottomRight();
  } else {
    point1 = rect.topRight();
    point2 = rect.bottomLeft();
  }
  p.drawLine(point1, point2);
}


QRect KstViewLine::surroundingGeometry() const {
  if (from().x() == to().x()) {
    //vertical line
    QRect geom = geometry();
    geom.setLeft(geom.left() - width()/2 - 1);
    geom.setRight(geom.right() + width()/2 + 1);
    return geom;
  }  
  if (from().y() == to().y()) {
    //horizontal line
    QRect geom = geometry();
    geom.setTop(geom.top() - width()/2 - 1);
    geom.setBottom(geom.bottom() + width()/2 + 1);
    return geom;
  }
  return geometry();
}


#include "kstviewline.moc"
// vim: ts=2 sw=2 et
