// -*- mode: C++; c-file-style: "Stroustrup" -*-
//
// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifndef KROT_RAWDATA_H
#define KROT_RAWDATA_H


#include <kapp.h>

#include <qobject.h>
#include <qstring.h>

#include <sys/types.h>
#include <set>
#include <vector>


class Simulation;


/**
 * Structure to hold one raw data point.
 *
 * @author Jochen Kpper
 * @version 1998/12/02
 */
class RawDataPoint
{
public:
    RawDataPoint();
    explicit RawDataPoint( int val );
    double etalon,
	iodine,
	power,
	signal;
};




/**
 * @short Raw data file import.
 *
 * This class reads the data, determines the marker positions, and gives an
 * spectrum of any trace in native krot format ( start, delta, int, int, ... ).
 *
 * @author Jochen Kpper
 * @version 1998/12/02
 */
class RawData : public QObject
{
    Q_OBJECT

public:
    
    enum LinearizationModel { LINEAR, QUADRATIC, CUBICSPLINE };


public:

    /**
     * Construct an object from an data file.
     *
     * @param filename Name of file to read data from.
     */
    RawData( const QString& filename );

    /**
     * Give back reference to data point index.
     *
     * @param  index Index of data point, starting at zero.
     * @return Data point.
     */
    const RawDataPoint& operator[]( const int index );

    /**
     * @return Maxima of each trace.
     */
    const RawDataPoint& max() const;
    
    /**
     * @return Minima of each trace.
     */
    const RawDataPoint& min() const;
    
    /**
     * @return markers.
     */
    const set< int >& marker() const;

    /**
     * Give the name of the imported file.
     * Caution: This interferes with Qts QObject::name() !
     *
     * @return Filename of imported data.
     */
    const QString& name() const;
    
    /**
     * @return Number of markers.
     */
    const unsigned int numMarker() const;

    /**
     * Set avergaing interval widht
     */
    void setAveraging( unsigned int val );

    /**
     * @return Number of data points in this set.
     */
    const int size() const;

    /**
     * Linearization of spectrum.
     *
     * Linearize a spectrum and give back the data to put it into a class
     * Spectrum.
     * Most parameters are actually output-variables !
     *
     * @param delta  Frequency steps (point to point distances) to use for
     *               linearized spectrum.
     * @param start  Return start frequency of the spectrum - that is the
     *               frequency of the first data point.
     * @param size   Return the number of data points - that's the size of the
     *               intensity array as well.
     * @return       Return an array with size elements containing the
     *               intensity values of the spectrum. This is assumed to be an
     *               valid pointer on entrance and any previous content will be
     *               deleted !
     *               This array is allocated by >>new int[]<< and needs to be
     *               deleted by the caller !!!
     */ 
    double* linearize( const unsigned int delta, int64_t& start, u_int64_t& size ) const;


protected slots:

    /**
     * Add marker or remove the marker that is closest to the given position.
     *
     * @param pos Marker position.
     */
    void toggleMarker( const int pos );


protected:
    
    /**
     * Calculate exact marker positions.
     */
    void determineMarkerPositions();
    
    /**
     * Read data from file.
     *
     * Needs to be implemented for every kind of data file to read in,
     * derive a class and implement it there !
     *
     * @param file stream to read data from
     */
    virtual void read() = 0;


private:

    /**
     * Disabled standard constructor.
     */
    RawData();


protected:

    /**
     * Averaging interval width to use.
     */
    unsigned int avg;
    
    /**
     * Cutoff value of etalon trace.
     *
     * Any intensity below that value will be treated as though it is zero
     * with regards to the marker positioning.
     */
    int cutoff;

    /**
     * raw data
     */
    vector< RawDataPoint > data;

    /**
     * operator[] returns a reference to this variable if averaging is done.
     */
    RawDataPoint avgData;

    /**
     * We return a reference to this variable, if the index for [] is out of range.
     */
    const RawDataPoint dataZero;

    /**
     * Filename of the spectrum.
     */
    QString filename;

    /**
     * marker positions
     */
    set< int > mks;

    /**
     * maximum value of each trace.
     */
    RawDataPoint mx;

    /**
     * minimum value of each trace.
     */
    RawDataPoint mn;

    /**
     * Number of fitted coefficients for marker profile.
     */
    int numCoeff;
};



/**
 * Class for KHiDaq data file import.
 *
 * This class provides a function to read the data, all other work is done by
 * the superclass RawData.
 *
 * @author Jochen Kpper
 * @version 1998/12/02
 */
class KHiDaqData : virtual public RawData
{
    Q_OBJECT

public:

    /**
     * Constructor.
     *
     * Import data from a KHiDaq file.
     */    
    KHiDaqData( const QString& filename );


protected:
    
    virtual void read();
};



/**
 * Class for jba data file import.
 *
 * This class provides a function to read the data, all other work is done by
 * the superclass RawData.
 *
 * @author Jochen Kpper
 * @version 1998/12/02
 */
class JbaData : virtual public RawData
{
    Q_OBJECT

public:
    
    JbaData( const QString& file );


protected:
    
    virtual void read();
};



#include "rawData_inline.h"



#endif
