// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#include "importRawData.h"
#include "krot.h"
#include "krotData.h"
#include "rawData.h"

#include "arnirot/arnirot.h"

#include <fstream.h>
#ifdef HAVE_STDINT_H
#  define __STDC_LIMIT_MACROS
#  include <stdint.h>
#else
#  define INT64_MAX    ( 1<<62 )
#  define INT64_MIN ( -( 1<<62 ) )
#endif



void DataRange::extend( const DataRange& two )
{
    start = std::min( start, two.start );
    stop = std::max( stop, two.stop );
    min = std::min( min, two.min );
    stop = std::max( max, two.max );
}



void DataRanges::add( const DataRange& range )
{
    insert( begin(), range );	
/*
    if( range.stop < begin()->start ) {
	// insert before all others
	insert( begin(), range );
    } else if( range.start > end()->stop ) {
	// insert behind all others
	insert( end(), range );
    } else if( range.start > end()->start ) {
	// extend last set to the end
	back().extend( range );
    } else {
	iterator r = begin();
	while( range.start < r->start )
	    r++;
	if( range.start < r->stop ) {
	    // start is in an existing range *r
	    iterator r2 = r;
	    while( range.stop >= r2->start )
		r2++;
	    r2--;
	    // now r2 points to the last range to merge
	    if( r == r2 ) {
		// range doesn't overlap with anything else but *r
		r->extend( range );
	    } else {
		// range is overlaping with one or more ranges [ r, r2 ]
		r2->extend( range );
		r2->extend( *r );
		while( r != r2 ) {
		    r++;
		    r2->extend( *r );
		    erase( r );
		}
	    }
	} else {
	    // start is in a empty area behind *r
	    iterator r2 = r;
	    while( range.stop >= r2->start )
		r2++;
	    r2--;
	    // now r2 points to the last range to merge
	    if( r == r2 ) {
		// range doesn't overlap with anything else,
		    if( end() == r2 )
			    insert( r2, range );
			else
			    insert( ++r2, range );
	    } else {
		// range is overlaping with one or more ranges ] r, r2 ]
		r2->extend( range );
		while( r != r2 ) {
		    r++;
		    r2->extend( *r );
		    erase( r );
		}
	    }
	}
    }
*/
    sort( begin(), end() );
    unique( begin(), end() );    
}



void KRotData::add( const AssignedFrequency& asn )
{
    KROT_LAUNCH7( "Launching KRotData::assignment, assignment is %d %d %d <-%d %d %d  %f",
		 asn[ QNum::eJ ], asn[ QNum::eKa ], asn[ QNum::eKc ],
		 asn[ QNum::gJ ], asn[ QNum::gKa ], asn[ QNum::gKc ],
		 asn.frequency() );
    // try to insert the assignment,
    pair< mapAssignment::iterator, bool > p = ass.insert( pairAssignment( QNum( asn ), asn.frequency() ) );
    if( false == p.second ) {
	KROT_DEBUG_VERBOSE( "  calling ourself recursively." );
	// if it fails, delete the old entry and call ourself recursevly.
	ass.erase( p.first );
	add( asn );
    }
    return;
}



void KRotData::arnirot( const CalculationParameter& par )
{
    KROT_LAUNCH( "Launching KRotData::arnirot( const CalculationParameter )" );
    pars = par;
    pars.writeConfig( config );
    Arnirot *calc = new Arnirot( pars, ass );
    connect( calc, SIGNAL( done( const CalculationParameter, vector< Transition > ) ),
	     this, SLOT( arnirot( const CalculationParameter, vector< Transition > ) ) );
    calc->start();
    return;
}



void KRotData::arnirot( const CalculationParameter par, vector< Transition > lines )
{
    KROT_LAUNCH( "Launching KRotData::arnirot( const CalculationParameter, vector< Transition > )" );
    pars = par;
    // create simulation here
    KROT_DEBUG1_VERBOSE( "Number of calculated lines %d", lines.size() );
    Simulation sim( lines, this );
    add( sim );
    return;
}



void KRotData::checkLimits()
{
    KROT_LAUNCH( "Launching KRotData::checkLimits" );
    // check for the frequency and intensity ranges we do have data in
    ranges.erase( ranges.begin(), ranges.end() );
    if( empty() ) {
	ranges.insert( ranges.begin(), DataRange() );
    } else {
	if( ! sims.empty() ) {
	    mapSimulation::iterator iter = sims.begin();
	    while( sims.end() != iter ) {
		DataRange r( iter->second.start(), iter->second.stop(),
			     static_cast< int64_t >( iter->second.min() ),
			     static_cast< int64_t >( iter->second.max() ) );
		ranges.add( r );
		iter++;
	    }
	}
	// scan spectra
	if( ! spec.empty() ) {
	    mapSpectrum::iterator iter = spec.begin();
	    while( spec.end() != iter ) {
		DataRange r( iter->second.start(), iter->second.stop(),
			     static_cast< int64_t >( iter->second.min() ),
			     static_cast< int64_t >( iter->second.max() ) );
		ranges.add( r );
		iter++;
	    }
	}
    }
    // tell the outside world
    emit limits( ranges );
    return;
}



void KRotData::eraseAll()
{
    KROT_LAUNCH( "Launching KRotData::eraseAll" );
    sims.erase( sims.begin(), sims.end() );
    spec.erase( spec.begin(), spec.end() );
    checkLimits();
    emit message( i18n( "All sets deleted !" ) );
    return;
}



void KRotData::exportAssignments( const QString& filename ) const
{
    KROT_LAUNCH( "Launching KRotData::exportAssignments" );
    std::ofstream stream( filename.data(), ofstream::out | ofstream::trunc );
    stream.setf( ofstream::fixed );
    stream.precision( 1 );
    mapAssignment::const_iterator iter = ass.begin();
    while( ass.end() != iter ) {
	stream << string( iter->first ) << "  " << iter->second << endl;
	iter++;
    }
    return;
}



void KRotData::exportXY( const QString& setName, const QString& fileName )
{
    KROT_LAUNCH( "Launching KRotData::exportXY" );
    getDataSet( setName.data() )->save( fileName, DataSet::XY_DATA );
    QString msg(" ");
    msg += setName + i18n( " saved." );
    emit message( msg );
    return;
}



void KRotData::exportLines( const QString& simName, const QString& fileName )
{
    KROT_LAUNCH( "Launching KRotData::exportLines - NOT IMPLEMENTED !!!" );
    return;
}



DataSet *KRotData::getDataSet( const char *name )
{
    if( sims.end() != sims.find( name ) )
	return &( sims[ name ] );
    if( spec.end() != spec.find( name ) )
	return &( spec[ name ] );
    return 0;
}



void KRotData::importLines( const QString& filename )
{
    KDEBUG1( KDEBUG_INFO, KDEBUG_KROT_LAUNCH,
	    "Launching KRotData::importLines with filename %s", filename.data() );
    Simulation sim( this );
    sim.load( filename, DataSet::LINES_DAT_DATA );
    add( sim );
    // tell the user
    QString msg( i18n( " Simulation " ) );
    msg += sim.shortName() + i18n( " imported" );
    emit message( msg );
    return;
}



void KRotData::importJba( const QString& filename )
{
    KROT_LAUNCH( "Launching KRotData::importJba with filename " );
    // open file
    QFile file( filename );
    if(! file.open( IO_ReadOnly) ) {
	KMsgBox::message( 0, i18n( "File Error" ),
			  i18n( "Could not open file to import KHiDaq data." ),
			  KMsgBox::EXCLAMATION, i18n( "OK" ) );
	return;
    }
    file.close();
    JbaData *jba = new JbaData( filename );
    QString msg( i18n( " jba data from " ) );
    msg += file.name();
    msg += i18n( " imported" );
    emit message( msg );
    // show raw data + marker positions
#warning Needs to be substituted by jba specific dialog ! (Jochen)
    DialogImportKHiDaq *dialog = new DialogImportKHiDaq( *jba );
    dialog->show();
    connect( dialog, SIGNAL( marker( const int ) ),
	     jba,    SLOT( toggleMarker( const int ) ) );
    // create spectrum from the data

    return;
}



void KRotData::importKHiDaq( const RawData& khidaq )
{
    KDEBUG( KDEBUG_INFO, KDEBUG_KROT_LAUNCH, "Launching KRotData::importKHiDaq( RawData& )" );
    Spectrum spec( khidaq, this );
    add( spec );
    // tell the user
    QString msg( i18n( " Spectrum " ) );
    msg += spec.shortName() + i18n( " imported" );
    emit message( msg );
    emit newSet();
    return;
}



void KRotData::importXY( const QString& filename )
{
    KROT_LAUNCH1( "Launching KRotData::importXY with filename %s", filename.data() );
    Spectrum spectrum( this );
    spectrum.load( filename, DataSet::XY_DATA );
    add( spectrum );
    // tell the user
    QString msg( i18n( " Spectrum " ) );
    msg += spectrum.shortName() + i18n( " imported" );
    emit message( msg );
    emit newSet();
    return;
}



//* Local Variables:
//* mode: C++
//* c-file-style: "Stroustrup"
//* End:
