// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 , USA
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#include "editSets.h"
#include "lined.h"
#include "krotData.h"
#include "simulation.h"

#include <kapp.h>
#include <kbuttonbox.h>

#include <qbuttongroup.h>
#include <qlistbox.h>
#include <qlayout.h>
#include <qradiobutton.h>
#include <qstring.h>

#include <iostream>




DialogFoldSet::DialogFoldSet( KRotData *krotdata, QWidget* parent, const char *name )
    : QDialog( parent, name ),
      krotData( krotdata )
{
    KROT_LAUNCH( "Launching DialogFoldSet constructor." );
    setCaption( i18n( "krot [Folding]" ) );
    QGridLayout *layout = new QGridLayout( this, 3, 3, KROT_DIALOGMARGIN, 5, "DialogFoldSetLayout" );
    // list
    list = new QListBox( this, "FoldSetList" );
    list->setMultiSelection( false );
    layout->addMultiCellWidget( list, 0, 1, 0, 1 );
    connect( list, SIGNAL( highlighted( const char* ) ),
	     SLOT( newSet( const char* ) ) );
    // line shape function
    QButtonGroup *lineshapeGroup = new QButtonGroup( i18n( "lineshape function" ), this, "lineshapeGroup" );
    QGridLayout *lineshapeLayout = new QGridLayout( lineshapeGroup, 4, 1, 15, 5, "lineshapeLayout" );
    stickRB = new QRadioButton( i18n( "sticks" ), lineshapeGroup, "SticksRB" );
    lineshapeLayout->addWidget( stickRB, 0, 0 );
    gaussRB = new QRadioButton( i18n( "Gauss" ), lineshapeGroup, "GaussRB" );
    lineshapeLayout->addWidget( gaussRB, 1, 0 );
    lorentzRB = new QRadioButton( i18n( "Lorentz" ), lineshapeGroup, "LorentzRB" );
    lineshapeLayout->addWidget( lorentzRB, 2, 0 );
    voigtRB = new QRadioButton( i18n( "Voigt" ), lineshapeGroup, "VoigtRB" );
    lineshapeLayout->addWidget( voigtRB, 3, 0 );
    lineshapeLayout->activate();
    layout->addWidget( lineshapeGroup, 0, 2 );
    connect( lineshapeGroup, SIGNAL( clicked( int ) ),
	     this, SLOT( newShape( int ) ) );
    // line width
    QButtonGroup *linewidthGroup = new QButtonGroup( i18n( "line width" ), this, "linewidthGroup" );
    QGridLayout *widthLayout = new QGridLayout( linewidthGroup, 4, 1, 15, 5, "LineWidthLayout" );
    QLabel *gaussLBL = new QLabel( i18n( "Gaussian FWHM [MHz]:" ), linewidthGroup, "GaussLabel" );
    widthLayout->addWidget( gaussLBL, 0, 0 );
    gaussFWHM = new Floated( 1, 0.0, linewidthGroup, "gaussLine" );
    widthLayout->addWidget( gaussFWHM, 1, 0 );
    QLabel *lorentzLBL = new QLabel( i18n( "Lorentzian FWHM [MHz]:" ), linewidthGroup, "LorentzLabel" );
    widthLayout->addWidget( lorentzLBL, 2, 0 );
    lorentzFWHM = new Floated( 1, 0.0, linewidthGroup, "gaussLine" );
    widthLayout->addWidget( lorentzFWHM, 3, 0 );
    widthLayout->activate();
    layout->addWidget( linewidthGroup, 1, 2 );
    // buttons
    KButtonBox *bbox = new KButtonBox( this );
    bbox->addStretch( 5 );
    QPushButton *fold = bbox->addButton( i18n( "Fold" ) );
    bbox->addStretch( 10 );
    QPushButton *close = bbox->addButton( i18n( "Close" ) );
    bbox->addStretch( 10 );
    QPushButton *help   = bbox->addButton( i18n( "Help" ) );
    bbox->addStretch( 5 );
    bbox->layout();
    bbox->setFixedHeight( bbox->sizeHint().height() + 10 );
    connect( fold, SIGNAL( clicked() ),
	     this, SLOT( fold() ) );
    connect( close, SIGNAL( clicked() ),
	     this, SLOT( reject() ) );
    connect( help, SIGNAL( clicked() ),
	     this, SLOT( help() ) );
    layout->addMultiCellWidget( bbox, 2, 2, 0, 2 );
    layout->activate();

    KROT_ASSERT( 0 < list->count(), "DialogFoldSet called without any simulation" );
    list->setCurrentItem( list->topItem() );
    return;
}



void DialogFoldSet::fold()
{
    KROT_LAUNCH( "Launching DialogFoldSet::fold." );
    QString name( list->text( list->currentItem() ) );
    Simulation& sim = ( krotData->simulations() )[ name ];
    if( gaussRB->isChecked() ) {
	sim.setLineshape( Simulation::LINESHAPE_GAUSS );
	sim.setGauss( atof( gaussFWHM->text() ) );
    } else if( lorentzRB->isChecked() ) {
	sim.setLineshape( Simulation::LINESHAPE_LORENTZ );
	sim.setLorentz( atof( lorentzFWHM->text() ) );
    } else if( voigtRB->isChecked() ) {
	sim.setLineshape( Simulation::LINESHAPE_VOIGT );
	sim.setGauss( atof( gaussFWHM->text() ) );
	sim.setLorentz( atof( lorentzFWHM->text() ) );
    } else {
        sim.setLineshape( Simulation::LINESHAPE_STICK );
    }
    sim.fold();
    emit changed();
    return;
}



void DialogFoldSet::help()
{
    KROT_LAUNCH( "Launching DialogFoldSet::help." );
    kapp->invokeHTMLHelp( "krot/folding.html", "" );
    return;
}



void DialogFoldSet::newSet( const char *name )
{
    KROT_LAUNCH( "Launching DialogFoldSet::newSet." );
    const Simulation& sim = krotData->simulations().find( name )->second;
    switch( sim.lineshape() ) {
    case Simulation::LINESHAPE_STICK:
	stickRB->setChecked( true );
	gaussFWHM->setValue( 0.0 );
	gaussFWHM->setEnabled( false );
	lorentzFWHM->setValue( 0.0 );
	lorentzFWHM->setEnabled( false );
	break;
    case Simulation::LINESHAPE_GAUSS:
	gaussRB->setChecked( true );
	gaussFWHM->setEnabled( true );
	gaussFWHM->setValue( sim.gauss() );
	lorentzFWHM->setValue( 0.0 );
	lorentzFWHM->setEnabled( false );
	break;
    case Simulation::LINESHAPE_LORENTZ:
	lorentzRB->setChecked( true );
	gaussFWHM->setValue( 0.0 );
	gaussFWHM->setEnabled( false );
	lorentzFWHM->setEnabled( true );
	lorentzFWHM->setValue( sim.lorentz() );
	break;
    case Simulation::LINESHAPE_VOIGT:
	voigtRB->setChecked( true );
	gaussFWHM->setEnabled( true );
	gaussFWHM->setValue( sim.gauss() );
	lorentzFWHM->setEnabled( true );
	lorentzFWHM->setValue( sim.lorentz() );
	break;
    }
    return;
}



void DialogFoldSet::newShape( int id )
{
    KROT_LAUNCH( "Launching DialogFoldSet::newShape." );
    if( 0 < list->count() ) {
	QString name( list->text( list->currentItem() ) );
	const Simulation& sim = krotData->simulations().find( name )->second;
	switch( id ) {
	case 0:
	    gaussFWHM->setValue( 0.0 );
	    gaussFWHM->setEnabled( false );
	    lorentzFWHM->setValue( 0.0 );
	    lorentzFWHM->setEnabled( false );
	    break;
	case 1:
	    gaussFWHM->setEnabled( true );
	    gaussFWHM->setValue( sim.gauss() );
	    lorentzFWHM->setValue( 0.0 );
	    lorentzFWHM->setEnabled( false );
	    break;
	case 2:
	    gaussFWHM->setValue( 0.0 );
	    gaussFWHM->setEnabled( false );
	    lorentzFWHM->setEnabled( true );
	    lorentzFWHM->setValue( sim.lorentz() );
	    break;
	case 3:
	    gaussFWHM->setEnabled( true );
	    gaussFWHM->setValue( sim.gauss() );
	    lorentzFWHM->setEnabled( true );
	    lorentzFWHM->setValue( sim.lorentz() );
	    break;
	default:
	    KROT_DEBUG( "Invalid lineshape in DialogFoldSet::newShape !" );
	    break;
	}
    }
    return;
}



void DialogFoldSet::show()
{
    KROT_LAUNCH( "Launching DialogFoldSet::show." );
    list->setAutoUpdate( false );
    list->clear();
    mapSimulation::const_iterator sims = krotData->simulations().begin();
    while( sims != krotData->simulations().end() ) {
	list->inSort( sims->first.data() );
	sims++;
    }
    list->setCurrentItem( list->topItem() );
    list->setAutoUpdate( true );
    QDialog::show();
}



//* Local Variables:
//* c-file-style: "Stroustrup"
//* mode: C++
//* End:
