c     This file is part of krot,
c     a program for the simulation, assignment and fit of HRLIF spectra.
c
c     Copyright (C) 1994-1998 Arnim Westphal
c     Copyright (C) 1997-1999 Jochen Kpper
c
c     If you use this program for your scientific work, please cite it according to
c     the file CITATION included with this package.
c
c     krot-arnirot
c     a program to calculate rotational resolved vibrational/vibronic bands


#include "arni.h"


      subroutine rrde( J, deriv, dmeval, npar, dmham, iact, mat, vecr, veci, fcompl )
c     evaluate the rigid rotor derivatives for complex hamiltonia

      implicit none

      integer        dmham, dmeval, npar
      integer        iact, iept, is
      integer        g, J
c     flag for complex eigenvectors
      integer        fcompl
      real*8         deriv( dmeval, npar )
      real*8         mat( dmham, dmham ), vecr( dmham, dmham ), veci( dmham, dmham )

      ARNIROT_LAUNCH ( "Launching rrde." )

c     no derivatives for J=0 block
      if ( J .eq. 0 ) return
      
c     calculate offset to current J block for correct derivative loading
      iept = J*J
c     loop to count the three derivatives
      do g = 1, 3
c        zero the work space
         call smat0( mat, dmham, iact )
         if ( g .eq. 1 ) then
c           A constant derivative: <i|Jz**2|i> matrix
            call jz2( mat, dmham, J )
         else if ( g .eq. 2 ) then
c           B constant derivative: <i|Jx**2|j> matrix
            call jx2( mat, dmham, J )
         else
c           C constant derivative: <i|Jy**2|j> matrix
            call jy2( mat, dmham, J )
         end if

c        loop to count the states
         if ( fcompl .eq. 1 ) then
c           consider complex eigenvectors
            do is = 1, iact
c              zero current derivative position
               deriv( is+iept, g ) = 0.d0
c              diagonal contributions to <Jg**2> operator derivative; g = z, x, y
               call decalc( deriv( is+iept, g ), is, mat, vecr, veci, dmham, iact, 0, 0 )
c              two-off-diagonal contributions to <Jg**2> operator derivative; g = z, x, y
               call decalc( deriv( is+iept, g ), is, mat, vecr, veci, dmham, iact, 2, 0 )
            end do
         else
c           consider only real eigenvectors (standard rr, not axis switched)
            do is = 1, iact
c              zero current derivative position
               deriv( is+iept, g ) = 0.d0
c              diagonal contributions to <Jg**2> operator derivative; g = z, x, y
               call decalr( deriv( is+iept, g ), is, mat, vecr, dmham, iact, 0 )
c              two-off-diagonal contributions to <Jg**2> operator derivative; g = z, x, y
               call decalr( deriv( is+iept, g ), is, mat, vecr, dmham, iact, 2 )
            end do
         end if
      end do
      
      return
      end


c------------------------------------------------------------------------------
      subroutine jz2( mat, dmham, J )
c     evaluate <i|Jz**2|i> matrix
c     - used for calculation of rigid rotor derivatives

      implicit none

      integer        dmham, i, J, K
      real*8         mat( dmham, dmham )

      ARNIROT_LAUNCH ( "Launching jz2." )

c     only diagonal <i|Jz**2|i> contributions
      do K = -J, J
         i = K + J + 1
         mat( i, i ) = K * K
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine jy2( mat, dmham, J )
c     evaluate <i|Jy**2|j> matrix
c     - used for calculation of rigid rotor derivatives

      implicit none

      integer        dmham, i, J, K
      real*8         mat( dmham, dmham )

      ARNIROT_LAUNCH ( "Launching jy2." )

c     most diagonal <i|Jy**2|i> and all two-off-diagonal <i+-2|Jy**2|i> contributions
      do K = -J, J - 2
         i = K + J + 1
         mat( i, i )     = 0.5 * ( J * ( J + 1 ) - K * K )
         mat( i + 2, i ) = -0.25 * dsqrt( dfloat( ( J * ( J + 1 ) - K * ( K + 1 ) )
     *                                          * ( J * ( J + 1 ) - ( K + 1 ) * ( K + 2 ) ) ) )
         mat( i, i + 2 ) = mat( i + 2, i )
      end do

c     residual diagonal elements
      do K = J - 1, J
         i = K + J + 1
         mat( i, i )     = 0.5 * ( J * ( J + 1 ) - K * K )
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine jx2( mat, dmham, J )
c     evaluate <i|Jx**2|j> matrix
c     - used for calculation of rigid rotor derivatives

      implicit none

      integer        dmham, i, J, K
      real*8         mat( dmham, dmham )

      ARNIROT_LAUNCH ( "Launching jx2." )

c     most diagonal <i|Jx**2|i> and all two-off-diagonal <i+-2|Jx**2|i> contributions
      do K = -J, J - 2
         i = K + J + 1
         mat( i, i )     = 0.5 * ( J * ( J + 1 ) - K * K )
         mat( i + 2, i ) = +0.25 * dsqrt( dfloat( ( J * ( J + 1 ) - K * ( K + 1 ) )
     *                                          * ( J * ( J + 1 ) - ( K + 1 ) * ( K + 2 ) ) ) )
         mat( i, i + 2 ) = mat( i + 2, i )
      end do

c     residual diagonal elements
      do K = J - 1, J
         i = K + J + 1
         mat( i, i )     = 0.5 * ( J * ( J + 1 ) - K * K )
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine jzjxde( J, deriv, dmeval, npar, dmham, iact, mat, vecr, veci )
c     evaluate the JzJx+JxJz derivatives

      implicit none

      integer        dmham, dmeval, npar
      integer        iact, iept, is, J
      real*8         deriv( dmeval, npar )
      real*8         mat( dmham, dmham ), vecr( dmham, dmham ), veci( dmham, dmham )

      ARNIROT_LAUNCH ( "Launching jzjxde." )

c     calculate offset for correct derivative loading
      iept = J*J
c     zero the work space
      call smat0( mat, dmham, iact )
c     fill < |JzJx+JxJz| > matrix Dzx = 1
      call jzjx( mat, dmeval, J, 1.d0 )
      do is = 1, iact, 1
c        initialize derivative at zero
         deriv( is+iept, 5 ) = 0.d0
c        eigenvector sum over JzJx+JxJz operator matrix in K basis
         call decalc( deriv( is+iept, 5 ), is, mat, vecr, veci, dmham, iact, 1, 0 )
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine jzjyde( J, deriv, dmeval, npar, dmham, iact, mat, vecr, veci )
c     evaluate the JzJy+JyJz derivatives

      implicit none

      integer        dmham, dmeval, npar
      integer        iact, iept, is, J
      real*8         deriv( dmeval, npar )
      real*8         mat( dmham, dmham ), vecr( dmham, dmham ), veci( dmham, dmham )

      ARNIROT_LAUNCH ( "Launching jzjyde." )

c     calculate offset for correct derivative loading
      iept = J*J
c     zero the work space
      call smat0( mat, dmham, iact )
c     fill < |JzJy+JyJz| > matrix Dzy = 1
      call jzjy( mat, dmeval, J, 1.d0 )
      do is = 1, iact, 1
c        initialize derivative at zero
         deriv( is+iept, 6 ) = 0.d0
c        eigenvector sum over JzJy+JyJz operator matrix in K basis
         call decalc( deriv( is+iept, 6 ), is, mat, vecr, veci, dmham, iact, 1, 1 )
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine jyjxde( J, deriv, dmeval, npar, dmham, iact, mat, vecr, veci )
c     evaluate the JyJx+JxJy derivatives

      implicit none

      integer        dmham, dmeval, npar
      integer        iact, iept, is, J
      real*8         deriv( dmeval, npar )
      real*8         mat( dmham, dmham ), vecr( dmham, dmham ), veci( dmham, dmham )

      ARNIROT_LAUNCH ( "Launching jyjxde." )

c     calculate offset for correct derivative loading
      iept = J*J
c     zero the work space
      call smat0( mat, dmham, iact )
c     fill < |JyJx+JxJy| > matrix Dyx = 1
      call jyjx( mat, dmeval, J, 1.d0 )
      do is = 1, iact, 1
c        initialize derivative at zero
         deriv( is+iept, 4 ) = 0.d0
c        eigenvector sum over JyJx+JxJy operator matrix in K basis
         call decalc( deriv( is+iept, 4 ), is, mat, vecr, veci, dmham, iact, 2, 1 )
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine jxde( J, deriv, dmeval, npar, dmham, iact, mat, vecr, veci )
c     evaluate the Jx derivatives

      implicit none

      integer        dmham, dmeval, npar
      integer        iact, iept, is, J
      real*8         deriv( dmeval, npar )
      real*8         mat( dmham, dmham ), vecr( dmham, dmham ), veci( dmham, dmham )

      ARNIROT_LAUNCH ( "Launching jxde." )

c     calculate offset for correct derivative loading
      iept = J*J
c     zero the work space
      call smat0( mat, dmham, iact )
c     fill < |Jx| > matrix Dx = 1
      call jx( mat, dmham, J, 1.d0 )
      do is = 1, iact, 1
c        initialize derivative at zero
         deriv( is+iept, 4 ) = 0.d0
c        eigenvector sum over Jx operator matrix in K basis
         call decalc( deriv( is+iept, 4 ), is, mat, vecr, veci, dmham, iact, 1, 0 )
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine jyde( J, deriv, dmeval, npar, dmham, iact, mat, vecr, veci )
c     evaluate the Jy derivatives

      implicit none

      integer        dmham, dmeval, npar
      integer        iact, iept, is, J
      real*8         deriv( dmeval, npar )
      real*8         mat( dmham, dmham ), vecr( dmham, dmham ), veci( dmham, dmham )

      ARNIROT_LAUNCH ( "Launching jyde." )

c     calculate offset for correct derivative loading
      iept = J*J
c     zero the work space
      call smat0( mat, dmham, iact )
c     fill < |Jy| > matrix Dy = 1
      call jy( mat, dmham, J, 1.d0 )
      do is = 1, iact, 1
c        initialize derivative at zero
         deriv( is+iept, 5 ) = 0.d0
c        eigenvector sum over Jy operator matrix in K basis
         call decalc( deriv( is+iept, 5 ), is, mat, vecr, veci, dmham, iact, 1, 1 )
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine jzde( J, deriv, dmeval, npar, dmham, iact, mat, vecr, veci )
c     evaluate the Jz derivatives

      implicit none

      integer        dmham, dmeval, npar
      integer        iact, iept, is, J
      real*8         deriv( dmeval, npar )
      real*8         mat( dmham, dmham ), vecr( dmham, dmham ), veci( dmham, dmham )

      ARNIROT_LAUNCH ( "Launching jzde." )

c     calculate offset for correct derivative loading
      iept = J*J
c     zero the work space
      call smat0( mat, dmham, iact )
c     fill < |Jz| > matrix Dz = 1
      call jz( mat, dmham, J, 1.d0 )
      do is = 1, iact, 1
c        initialize derivative at zero
         deriv( is+iept, 6 ) = 0.d0
c        eigenvector sum over Jz operator matrix in K basis
         call decalc( deriv( is+iept, 6 ), is, mat, vecr, veci, dmham, iact, 0, 0 )
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine nrrde( J, deriv, dmeval, npar, dmham, iact, mat, vecr, veci, fcompl )
c     evaluate the non-rigid rotor derivatives (Watson hamiltonian quartic terms) for real hamiltonia

      implicit none

      integer        dmham, dmeval, npar
      integer        iact, iept, is
      integer        J
c     integer        g
c     flag for complex eigenvectors
      integer        fcompl
      real*8         deriv( dmeval, npar )
      real*8         mat( dmham, dmham ), vecr( dmham, dmham ), veci( dmham, dmham )

      ARNIROT_LAUNCH ( "Launching nrrde." )

c     no derivatives for J=0 block
      if ( J .eq. 0 ) return
      
c     calculate offset to current J block for correct derivative loading
      iept = J*J

c     DJ constant derivative
      call smat0( mat, dmham, iact )
      call j4( mat, dmham, J )
      do is = 1, iact, 1
         deriv( is+iept, 7 ) = 0.d0
         call decalr( deriv( is+iept, 7 ), is, mat, vecr, dmham, iact, 0 )
      end do
c     DJK constant derivative
      call smat0( mat, dmham, iact )
      call j2jz2( mat, dmham, J )
      do is = 1, iact, 1
         deriv( is+iept, 8 ) = 0.d0
         call decalr( deriv( is+iept, 8 ), is, mat, vecr, dmham, iact, 0 )
      end do
c     DK constant derivative
      call smat0( mat, dmham, iact )
      call jz4( mat, dmham, J )
      do is = 1, iact, 1
         deriv( is+iept, 9 ) = 0.d0
         call decalr( deriv( is+iept, 9 ), is, mat, vecr, dmham, iact, 0 )
      end do
c     dJ constant derivative
      call smat0( mat, dmham, iact )
      call sdjde( mat, dmham, J )
      do is = 1, iact, 1
         deriv( is+iept, 10 ) = 0.d0
         call decalr( deriv( is+iept, 10 ), is, mat, vecr, dmham, iact, 2 )
      end do
c     dK constant derivative
      call smat0( mat, dmham, iact )
      call sdkde( mat, dmham, J )
      do is = 1, iact, 1
         deriv( is+iept, 11 ) = 0.d0
         call decalr( deriv( is+iept, 11 ), is, mat, vecr, dmham, iact, 2 )
      end do

c     loop to count the five derivatives
c     do g = 7, 11
c        zero the work space
c        call smat0( mat, dmham, iact )
c        if ( g .eq. 7 ) then
c           DJ constant derivative
c           call j4( mat, dmham, J )
c        else if ( g .eq. 8 ) then
c           DJK constant derivative
c           call j2jz2( mat, dmham, J )
c        else if ( g .eq. 9 ) then
c           DK constant derivative
c           call jz4( mat, dmham, J )
c        else if ( g .eq. 10 ) then
c           dJ constant derivative
c           call sdjde( mat, dmham, J )
c        else
c           dK constant derivative
c           call sdkde( mat, dmham, J )
c        end if
c
c        loop to count the states
c        consider only real eigenvectors (Watson nrr, unswitched)
c        do is = 1, iact
c           zero current derivative position
c           deriv( is + iept, g ) = 0.d0
c           diagonal contributions to operator derivative
c           call decalr( deriv( is + iept, g ), is, mat, vecr, dmham, iact, 0 )
c           two-off-diagonal contributions to operator derivative
c           call decalr( deriv( is + iept, g ), is, mat, vecr, dmham, iact, 2 )
c        end do
c     end do
      
      return
      end


c------------------------------------------------------------------------------
      subroutine j4( mat, dmham, J )
c     evaluate <i|(J**2)**2|i> matrix
c     - used for calculation of centrifugal coefficients derivatives

      implicit none

      integer        dmham, i, J, K
      real*8         mat( dmham, dmham )

      ARNIROT_LAUNCH ( "Launching j4." )

c     only diagonal <i|Op|i> contributions
      do K = -J, J
         i = K + J + 1
         mat( i, i ) = -( J*J ) * ( ( J + 1 ) * ( J + 1 ) )
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine j2jz2( mat, dmham, J )
c     evaluate <i|(J**2)*(Jz**2)|i> matrix
c     - used for calculation of centrifugal coefficients derivatives

      implicit none

      integer        dmham, i, J, K
      real*8         mat( dmham, dmham )

      ARNIROT_LAUNCH ( "Launching j2jz2." )

c     only diagonal <i|Op|i> contributions
      do K = -J, J
         i = K + J + 1
         mat( i, i ) = -J * ( J + 1 ) * ( K*K )
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine jz4( mat, dmham, J )
c     evaluate <i|Jz**4|i> matrix
c     - used for calculation of centrifugal coefficients derivatives

      implicit none

      integer        dmham, i, J, K
      real*8         mat( dmham, dmham )

      ARNIROT_LAUNCH ( "Launching jz4." )

c     only diagonal <i|Op|i> contributions
      do K = -J, J
         i = K + J + 1
         mat( i, i ) = -( K*K ) * ( K*K )
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine sdjde( mat, dmham, J )
c     evaluate dJ derivative matrix
c     - used for calculation of centrifugal coefficients derivatives

      implicit none

      integer        dmham, i, J, K
      real*8         mat( dmham, dmham )

      ARNIROT_LAUNCH ( "Launching sdjde." )

c     only two-off-diagonal <i+-2|Op|i> contributions
      do K = -J, J - 2
         i = K + J + 1
         mat( i + 2, i ) = -J * ( J + 1 )
     *                    * dsqrt( dfloat( ( J * ( J + 1 ) - K * ( K + 1 ) )
     *                                   * ( J * ( J + 1 ) - ( K + 1 ) * ( K + 2 ) ) ) )
         mat( i, i + 2 ) = mat( i + 2, i )
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine sdkde( mat, dmham, J )
c     evaluate dK derivative matrix
c     - used for calculation of centrifugal coefficients derivatives

      implicit none

      integer        dmham, i, J, K
      real*8         mat( dmham, dmham )

      ARNIROT_LAUNCH ( "Launching sdkde." )

c     only two-off-diagonal <i+-2|Op|i> contributions
      do K = -J, J - 2
         i = K + J + 1
         mat( i + 2, i ) = -0.5 * ( ( K + 2 ) * ( K + 2 ) + K*K )
     *                    * dsqrt( dfloat( ( J * ( J + 1 ) - K * ( K + 1 ) )
     *                                   * ( J * ( J + 1 ) - ( K + 1 ) * ( K + 2 ) ) ) )
         mat( i, i + 2 ) = mat( i + 2, i )
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine decalc( deriv, state, mat, vecr, veci, dim, act, ord, re0im1 )
c     generic routine to calculate derivatives for real or imaginary operators

c     deriv        = derivative being worked on
c     state        = number of state the derivative corresponds to
c     mat          = work space of operator matrix elements in K basis
c     vecr, veci   = real and imaginary eigenvectors for state
c     dim          = maximum (physical) dimension of matrices and vectors
c     act          = actual dimension of matrices and vectors
c     ord          = order of derivative summation off diagonal
c     re0im1       = flag for operator type, real(0) OR imaginary(1)

      implicit none

      integer        dim
      integer        act, state
      integer        ord, re0im1
      real*8         deriv
      real*8         mat( dim, dim ), vecr( dim, dim ), veci( dim, dim )

      integer        col, row

      ARNIROT_LAUNCH ( "Launching decalc." )

c     real operators
      if ( re0im1 .eq. 0 ) then
c        off-diagonal by ord terms in summation row=col-ord
         if ( ord .ne. 0 ) then
            do row = 1, act - ord
               col = row + ord
c              <i,j> real contribution ( <i,i> if ord == 0 )
               deriv = deriv + mat( row, col ) * vecr( row, state ) * vecr( col, state )
c              <i,j> imaginary contribution
               deriv = deriv + mat( row, col ) * veci( row, state ) * veci( col, state )
c              <j,i> contributions
c              real <j,i> contributions
               deriv = deriv + mat( col, row ) * vecr( col, state ) * vecr( row, state )
c              imaginary <j,i> contributions
               deriv = deriv + mat( col, row ) * veci( col, state ) * veci( row, state )
            end do
         else
c           diagonal elements only need to be counted once, only the first and second line (v.s.)
c           remain. Since ord == 0 => row == col, we can put them into one line:
            do row = 1, act - ord
               deriv = deriv + mat( row, row ) * ( vecr( row, state ) * vecr( row, state )
     *                                             + veci( row, state ) * veci( row, state ) )
            end do
         end if

c     imaginary operators
      else
         if ( ord .ne. 0 ) then
            do row = 1, act-ord
               col = row + ord
c              <i,j> imaginary contributions
               deriv = deriv + mat( row, col ) * ( veci( row, state ) * vecr( col, state )
     *                                             - vecr( row, state ) * veci( col, state ) )
c              <j,i> imaginary contributions
               deriv = deriv + mat( col, row ) * ( veci( col, state ) * vecr( row, state )
     *                                             - vecr( col, state ) * veci( row, state ) )
            end do
c        else
c           prevent diagonal elements from being counted twice, need only <i,j> contributions
c           Here these cancel each other, we don't need them.
c           do row = 1, act - ord
c              col = row
c              deriv = deriv - mat( row, row ) * vecr( row, state ) * veci( row, state )
c              deriv = deriv + mat( row, row ) * veci( row, state ) * vecr( row, state )
c           end do
         end if
      end if
      
      return
      end


c------------------------------------------------------------------------------
      subroutine decalr( deriv, state, mat, vecr, dim, act, ord )
c     generic routine to calculate derivatives for real or imaginary operators

c     deriv        = derivative being worked on
c     state        = number of state the derivative corresponds to
c     mat          = work space of operator matrix elements in K basis
c     vecr         = real eigenvectors for state
c     dim          = maximum (physical) dimension of matrices and vectors
c     act          = actual dimension of matrices and vectors
c     ord          = order of derivative summation off diagonal

      implicit none

      integer        dim
      integer        act, ord, state
      real*8         deriv
      real*8         mat( dim, dim ), vecr( dim, dim )

      integer        col, row

      ARNIROT_LAUNCH ( "Launching decalr." )

c     reduced calculation expense for purely real operators and eigenvectors
c     off-diagonal by ord terms in summation row=col-ord
      if ( ord .ne. 0 ) then
         do row = 1, act - ord
            col = row + ord
c           <i,j> and <j,i> real contributions
            deriv = deriv + ( mat( row, col ) + mat( col, row ) )
     *                      * vecr( row, state ) * vecr( col, state )
         end do
      else
c        if ord==0 => row==col, only mat element <i,i> is considered:
         do row = 1, act
            deriv = deriv + mat( row, row ) * ( vecr( row, state ) )**2
         end do
      end if
      
      return
      end


cc Local Variables:
cc mode: FORTRAN
cc End:
