/* Emacs, this is -*- C++ -*- */

#ifndef _TABLE_H_
#define _TABLE_H_


#include "column.h"


/** Representation of a worksheets data.


	This class combines multiple columns to a table. (Therefore the class's
	name, Table.) It provides a confortable and homogenous access to all
	the table's cells.

	The interface of the Table class hides most of the Column structure
	of the internal representation of a table. However, some functions
	still require direct contect with class Column, most notably the
	insertColumn() function.

	@see Column class for a single column.
	@author Patrick Schemitz
*/

class Table
{
public:
  /// Construct an empty table.
  Table ();
  /// Destruct table.
  ~Table ();

  /// Clone the table, i.e. make a deep copy.
  Table* clone ();

  /// Get the width of the table, i.e. the number of columns.
  int width ();
  /// Get the height of the table, i.e. the length of the longest column.
  int height ();

  /// Get the content of the given cell.
  const char* cell (int col, int row);
  /** Get the content of the given cell.

	  Make sure that the buffer, value, is big enough to hold
	  the contents!!!

	  @param value into here the value of the cell is copied.
  */
  void getCell (int col, int row, char* value);
  /// Set the content of the given cell.
  void setCell (int col, int row, const char* value);

  /// Return index of the column named "title".
  int findColumn (const char* title);

  /// Return Column of the given column.
  Column column (int col);

  /// Dump the whole table on the console. For debugging.
  void dump ();

  /** Insert a newly created column at the position specified.

	  The column must be created before/while the call to insertColumn(),
	  because insertColumn() has no way to know what kind of
	  Column to create. You would, for example, create a new
	  Column(Column::columnDouble), and then insert it into the table.
	  Insertion means that the former column at the given position (and all
	  following columns) are shifted one position to the right.

	  Note that the Column will be deleted by the Table's destructor.

	  Example:

	  myTable.insertColumn(0,Column(Column::columnDouble));

	  @param at where to insert the column.
	  @param newCol the newly created column to be inserted.
  */
  void insertColumn (int at, Column newCol);
  /// Insert an empty row in all columns.
  void insertRow    (int at);
  /// Delete a column. Free the memory it uses.
  void deleteColumn (int at);
  /// Delete a row across all columns. Decrememts height!!!
  void deleteRow    (int at);

  /// Get the name of the table (its title).
  const char* name ();
  /// Set the name of the table (its title).
  void setName (const char* new_name);

  /// Get type type string of a column.
  Column::ColumnType type (int col);
  /// Get the height of a column.
  int     height (int col);
  /// Get the title of a column.
  const char* title  (int col);
  /// Get the format string of a column.
  const char* format (int col);
  /// Set the title of a column.
  void setTitle  (int col, const char* new_title);
  /// Set the format string of a column.
  void setFormat (int col, const char* new_format);

  /** Mark the table as active.
	  This is especially important for the formula interpreter,
	  since columns are looked up in the active table only.

	  @see Formula class Formula.
  */
  void markActive ();

  /// To this value the table's name is initialized.
  static const char* TableUnnamedStr;

private:
  /// Class constant: Maximum with of a table.
  const int MaxSize = 1024;

  /// Copy constructor. Locked.
  Table (const Table&);
  /// Assignment operator. Locked
  Table& operator= (const Table&);

  /// variable: actual dimension (width) of the table.
  int the_dim;
  /// variable: pointers to the columns.
  Column the_column [MaxSize];
  /// variable: the table's name.
  char* the_name;
};


#endif /* _TABLE_H_ */
