#include "histo.h"
#include "histo.moc"

#include <iostream.h>
#include <stdio.h>

#include "debug.h"
#include "page.h"
#include "worksheet.h"

#include <kiconloader.h>
#include <kapp.h>


/*
 * Registration of the plot type.
 * 
 * This macro declares a CreatorHistoPlot class
 * and registers it at the class prototype list.
 * Only a small trade-off for not having to modify
 * (at least) four files each time you write a new
 * plot type!
 *
 */

REGISTER_PLOT(HistoPlot);


/*
 *
 *
 */

HistoPlotRecord::HistoPlotRecord (int bins)
{
  bin = new int [bins];
}

HistoPlotRecord::~HistoPlotRecord ()
{
  delete [] bin;
}


/*
 * Implementation of a 1D Histogram
 *
 */

HistoPlot::HistoPlot ()
  : Plot()
{
  IFDEBUG("korigin.histo",4)
	cout << "HistoPlot::HistoPlot() constructor" << endl;
  ENDDEBUG;

  bins = 10;
  auto_x = auto_y = true;
  xmin = ymin = 0.0;
  ymax = ymax = 1.0;
  axisFont = 0;
  axisColor = 0;
  frameColor = 0;
  the_menu = 0;
  records.setAutoDelete(true);

  IFDEBUG("korigin.histo",4)
	cout << "HistoPlot::HistoPlot() constructor ends" << endl;
  ENDDEBUG;
}

void HistoPlot::instantiate (Page* p, QWidget* parent=0, const char* name=0,
							 WFlags f=0)
{
  IFDEBUG("korigin.histo",4)
	cout << "HistoPlot::instantiate() pseudo-constructor" << endl;
  ENDDEBUG;

  active_icon = new QPixmap(
	KApplication::getKApplication()->getIconLoader()->loadIcon("histo-a.xpm"));
  inactive_icon = new QPixmap(
	KApplication::getKApplication()->getIconLoader()->loadIcon("histo-i.xpm"));

  Plot::instantiate(p,parent,name,f);

  delete axisFont;
  axisFont = new QFont(the_widget->font());
  delete axisColor;
  axisColor = new QColor(black);
  delete frameColor;
  frameColor = new QColor(blue);
  delete the_menu;
  the_menu = menu(0);
  delete the_treeitem;
  the_treeitem = new TreeItem();
  the_treeitem->setPixmap(active_icon);
  the_treeitem->setText(description());
  the_treeitem->setObject(this);

  IFDEBUG("korigin.histo",4)
	cout << "HistoPlot::instantiate() pseudo-constructor ends" << endl;
  ENDDEBUG;
}


Plot* HistoPlot::clone ()
{
  return new HistoPlot();
}


HistoPlot::~HistoPlot ()
{
  delete axisFont;
  delete axisColor;
  delete frameColor;
  delete the_menu;
}


void HistoPlot::slotDumbaer ()
{
}


void HistoPlot::slotRebin ()
{
  binning();
  page()->repaint();
}


void HistoPlot::plotSelectedColumns ()
{
  char msg [1024];
  HistoPlotRecord* new_record;
  Column* data;

  if (activeWS == 0) {
	QMessageBox::critical(0,"Histogram 1D: cant plot",
						  "I cant plot on this Histogram 1D since there\n"\
						  "is no active worksheet.");
	return;
  }
  data = activeWS->getSelected();
  if (data == 0) {
	sprintf(msg,"The currently active worksheet, %s,\n"\
			"has no column marked as \"Event\".",
			activeWS->caption());
	QMessageBox::critical(0,"Histogram 1D: cant plot",msg);
	return;
  }
  if (data[0].type() != Column::columnDouble) {
	sprintf(msg,"The column marked as \"Event\", %s,\n"\
			"does not contain numerical data as required by\n"\
			"the histogram.",
			data[0].title());
	QMessageBox::critical(0,"Histogram 1D: cant plot",msg);
	return;
  }

  new_record = new HistoPlotRecord(bins);
  new_record->data = data[0];
  new_record->re_bin = true;
  records.append(new_record);
  paintEvent(0);

  treeitem()->setText(description());
  emit changedTreeItem();
}


const char* HistoPlot::name ()
{
  return "Histogram 1D";
}


const char* HistoPlot::description ()
{
  static char buffer [128];
  sprintf(buffer,"Histogram 1D, %i data sets, %i bins",records.count(),bins);
  return buffer;
}


QPopupMenu* HistoPlot::menu (QPopupMenu* predefined)
{
  QPopupMenu* menue;
  if (predefined) {
	menue = predefined;
	menue->clear();
  } else {
	menue = new QPopupMenu();
  }
  menue->insertItem("Plot selected",this,SLOT(plotSelectedColumns()));
  menue->insertSeparator();
  menue->insertItem("Options",this,SLOT(optionsDialog()));
  menue->insertItem("Datasets",this,SLOT(datasetDialog()));
  menue->insertItem("Re-bin all",this,SLOT(slotRebin()));
  return menue;
}


QPixmap* HistoPlot::icon ()
{
  if (activePlot == this) {
	return active_icon;
  } else {
	return inactive_icon;
  }
}


int HistoPlot::paramCount ()
{
  return 1;
}


const char* HistoPlot::paramName (int i)
{
  if (i == 0)
	return "Event";
  return "I dont like index checks!";
}


void HistoPlot::mousePressEvent (QMouseEvent* me)
{
  QPoint global_pos;
  global_pos = the_widget->mapToGlobal(me->pos());
  if (me->button() == RightButton)
	{
	  the_menu->popup(global_pos);
	}
}


int HistoPlot::x2pixel (double x)
{
  return int((x-xmin)/(xmax-xmin) * the_widget->width());
}

int HistoPlot::y2pixel (double y)
{
  return the_widget->height() - int((y-ymin)/(ymax-ymin) 
									* the_widget->height());
}

double HistoPlot::pixel2x (int u)
{
  return xmin + (xmax-xmin) * u / the_widget->width();
}

double HistoPlot::pixel2y (int v)
{
  return ymin + (ymax-ymin) * (the_widget->height()-v) / the_widget->height();
}


void HistoPlot::reset_xrange ()
{
  xmax = -1.E100;
  xmin = 1.E100;
}

bool HistoPlot::find_xrange ()
{
  HistoPlotRecord* iter;
  int i, n;
  double value;
  Column col;
  
  for (iter=records.first(); iter!=0; iter=records.next())
	{
	  col = iter->data;
	  if (col.type() != Column::columnDouble) {
		QMessageBox::critical(0,"Error in Histogram 1D",
							  "The column you selected as \"Event\" does\n"\
							  "not contain numerical data.");
		return false;
	  }
	  
	  n = col.dim();
	  for (i=0; i<n; i++)
		{
		  value = col[i];
		  if (value < xmin) xmin = value;
		  if (value > xmax) xmax = value;
		}
	}

  IFDEBUG("korigin.histo",3)
	cout << "HistoPlot::find_xrange() min, max = " 
		 << xmin << ", " << xmax << endl;
  ENDDEBUG;
  return true;
}

void HistoPlot::reset_yrange ()
{
  ymax = 0.0;
  ymin = 0.0;
}

void HistoPlot::find_yrange ()
{
  HistoPlotRecord* iter;
  int i;
  double value;

  for (iter=records.first(); iter!=0; iter=records.next())
	{  
	  for (i=0; i<bins; i++)
		{
		  value = iter->bin[i];
		  if (value < ymin) ymin = value;
		  if (value > ymax) ymax = value;
		}
	}

  IFDEBUG("korigin.histo",3)
	cout << "HistoPlot::find_yrange() min, max = " 
		 << ymin << ", " << ymax << endl;
  ENDDEBUG;
}

bool HistoPlot::binning ()
{
  HistoPlotRecord* iter;
  int i, n, index;
  double delta;
  Column col;

  reset_xrange();
  if (find_xrange() == false) return false;
  delta = (xmax-xmin) / bins;

  reset_yrange();
  for (iter=records.first(); iter!=0; iter=records.next())
	{
	  for (i=0; i<bins; i++) iter->bin[i] = 0;
	  col = iter->data;
	  if (col.type() != Column::columnDouble)
		return false; // already handled by find_xrange().
	  n = col.dim();
	  for (i=0; i<n; i++)
		{
		  index = int((col[i]-xmin)/delta);
		  if (index >= 0 && index < bins)
			iter->bin[index]++;
		}
	  iter->re_bin = false;
	}
  find_yrange();
  return true;
}


void HistoPlot::paint (QPainter* p)
{
  HistoPlotRecord* iter;
  int i, j;
  double delta;
  delta = (xmax-xmin) / bins;

  p->setPen(*frameColor);
  p->drawRect(0,0,the_widget->width(),the_widget->height());

  for (iter=records.first(),i=0; iter!=0; iter=records.next(),i++)
	{
	  IFDEBUG("korigin.histo",3)
		cout << "HistoPlot::paint(QPainter*) set #" << i 
			 << ", " << bins << " bins." << endl;
	  ENDDEBUG;

	  if (iter->re_bin) binning();

	  p->setPen(*axisColor);
	  p->setFont(*axisFont);
	  p->drawText(5,the_widget->height()-20,iter->data.title());

	  p->setPen(iter->color);
	  for (j=0; j<bins; j++)
		{
		  p->drawLine( x2pixel(j*delta), y2pixel(iter->bin[j]),
					   x2pixel((j+1)*delta), y2pixel(iter->bin[j]) );
		  if (j < bins-1)
			p->drawLine( x2pixel((j+1)*delta), y2pixel(iter->bin[j]),
						 x2pixel((j+1)*delta), y2pixel(iter->bin[j+1]) );
		}
	}
}


bool HistoPlot::store (QFile& file, bool with_data)
{
  file.writeBlock(start_mark(),strlen(start_mark()));
  PageObject::store(file);
  file.writeBlock(end_mark(),strlen(end_mark()));
}


bool HistoPlot::restore (QFile& file, bool with_data)
{
  char buffer [128];
  PageObject::restore(file);
  file.readLine(buffer,126);
  if (file.atEnd()) return false;
  if (strcmp(buffer,end_mark()) == 0) return true;
  return false;
}
