/*
    This file is part of ktvision, a front end to frame grabber cards 
	for the KDE-Project
    
    Copyright (C) 1998 Oliver Burtchen <oliver.burtchen@rz.hu-berlin.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
#include "tvwidget.h"


Indicator::Indicator(QWidget * parent) : QWidget(parent)
{
	QColor x( red );
	QColorGroup g( x, black, x.light(), x.dark(), x.dark(120),
			x, white);
	QPalette p( g, g, g );
	setPalette( p ); 
	
	lcd = new QLCDNumber(3,this);
	lcd->setFrameStyle(QFrame::NoFrame);
	lcd->setSegmentStyle(QLCDNumber::Flat);
	lcd->setPalette( p ); 
	label = new QLabel(this);
	label->setAlignment(AlignCenter);
	label->setPalette( p ); 
	label->show();
	bar = new QProgressBar(100, this);
	bar->setPalette( p ); 
	bar->setStyle(MotifStyle);
	actionFlag = PresetAction;
	indicatorTimer = 0;
}

void Indicator::show()
{
	if (actionFlag==PresetAction){
		resize(label->width()+lcd->width()+10, label->height()+10);
		lcd->show();
		bar->hide();}
	else {
 		resize(bar->width()+10, label->height()+bar->height()+10);
		lcd->hide();
		bar->show();}
	QWidget::show();

	if (indicatorTimer) killTimer(indicatorTimer);
	indicatorTimer = startTimer(3000);
}

void Indicator::setText(const char * text)
{
	label->setText(text);
	show();
}

void Indicator::setBar(int i)
{
	bar->reset();
	bar->setProgress(i);
	if (indicatorTimer) killTimer(indicatorTimer);
	indicatorTimer = startTimer(3000);
}

void Indicator::setLcd(int i)
{
	lcd->display(i);
}

void Indicator::specialResize( QResizeEvent *e)
{
	int width = e->size().width();
	int height = e->size().height();
	int point = height / 50 + 8;

	QFont fo = label->font();
	fo.setPointSize(point);
	label->setFont(fo);

    QFontMetrics fm = label->fontMetrics();
	lcd->setGeometry(5,5, width/8, width/13);
	label->setGeometry(5 + width/8,5, width/3, width/13);
	bar->setGeometry(5, label->height()+6, 11*width/24, width/13);
 	move(width/20, height/20);
}

void Indicator::timerEvent(QTimerEvent *e)
{
	if ( e->timerId()==indicatorTimer)
	{
		hide();
		killTimer(indicatorTimer);
		indicatorTimer = 0;
		actionFlag=PresetAction;
	}
}


TvWidget::TvWidget(KApplication * app) : 
	Qv4lWin ( (QWidget *) NULL, (const char *) NULL)
{
	setCaption( app->getCaption() );
	cfg = new KConfig(KApplication::kde_datadir()+"/ktvision/regions",
			KApplication::localconfigdir()+"/ktvisionrc");
	mouseState = 0;
	mouseTimer = 0;
	aotTimer = 0;
	
	pic = new QPixmap();
	if ( !pic->load(KApplication::kde_datadir()+"/ktvision/pic/ktvision.bmp"))
	{	
		delete pic;
		pic = 0L;
	}

	for (int n=0; n<ACTIONNUM;n++)
		val[n] = 50;
		
	/*If you change the sequence, you MUST also change it in tvwidget.h*/
	actionText = new QStrList();
	actionText->append(i18n("Preset"));
	actionText->append(i18n("Volume"));
	actionText->append(i18n("Brightness"));
	actionText->append(i18n("Contrast"));
	actionText->append(i18n("Color"));
	actionText->append(i18n("Hue"));
	
	indicator = new Indicator(this);

	chanlist = new ChanList(cfg);
	chanlist->setCaption(i18n("Presets"));
	preferences = new QTabDialog();
	settings = new Settings(preferences);
	pictureDlg = new PictureDlg(preferences, 0, actionText);
	deviceDlg = new DeviceDlg(preferences);
	optionsDlg = new OptionsDlg(preferences);
	popup = new QPopupMenu();
	poppic = new QPopupMenu();

	cfg->setGroup("General");
	v4linit = initialize(cfg->readEntry("viddev", ""));

	preferences->setCaption(caption()+QString(" - ")+i18n("Preferences"));
	preferences->setCancelButton(i18n("Cancel"));
	connect(preferences, SIGNAL(applyButtonPressed()),SLOT(writeCfg()));
	connect(preferences, SIGNAL(applyButtonPressed()),
			chanlist,SLOT(hide()));
	connect(preferences, SIGNAL(cancelButtonPressed()),SLOT(readCfg()));
	connect(preferences, SIGNAL(cancelButtonPressed()),
			chanlist, SLOT(hide()));

			
	poppic->insertItem(actionText->at(BrightAction),BrightAction);
	poppic->insertItem(actionText->at(ContrastAction),ContrastAction);
	poppic->insertItem(actionText->at(ColorAction),ColorAction);
	poppic->insertItem(actionText->at(HueAction),HueAction);
	connect(poppic, SIGNAL(activated(int)),this, SLOT(changeAction(int)));
	
	popup->insertItem(i18n("Presets..."), this, SLOT(chanlistShow()));
	popup->setItemEnabled(popup->insertItem(i18n("Volume")), false);
	popup->setItemEnabled(popup->insertItem(i18n("Mute")), false);
	popup->setItemEnabled(popup->insertItem(i18n("TV Mode")), false);
	popup->insertItem(i18n("Picture"), poppic);
	popup->insertItem(i18n("On / Off"), this, SLOT(toggleCap()));
	popup->insertItem(i18n("Preferences..."), this, SLOT(settingsShow()));
	popup->setItemEnabled(popup->insertItem(i18n("Help")), false);
	
	settings->input->clear();
	settings->input->insertStrList(inputList());
	preferences->addTab(settings, i18n("Presets"));	

	connect(pictureDlg, SIGNAL(valueChanged(int,int)),
			SLOT(setValByValue(int,int)));
	preferences->addTab(pictureDlg, i18n("Picture"));	

	preferences->addTab(deviceDlg, i18n("Device"));	
	preferences->addTab(optionsDlg, i18n("Options"));	

	connect(chanlist, SIGNAL(highlighted(const char *)),
			indicator, SLOT(setText(const char *)));
	connect(chanlist, SIGNAL(highlighted(int)),
			settings->memLcd, SLOT(display(int)));
	connect(chanlist, SIGNAL(highlighted(int)),
			indicator, SLOT(setLcd(int)));

	connect(settings->chanEdit, SIGNAL(textChanged(const char *)),
			chanlist, SLOT(setName(const char *)));
	connect(chanlist, SIGNAL(nameChanged(const char *)),
			settings->chanEdit, SLOT(setText(const char *)));
	
	settings->chanSlider->setTracking(false);
	connect(settings->chanSlider, SIGNAL(valueChanged(int)),
			chanlist, SLOT(setChan(int)));
	connect(chanlist, SIGNAL(chanChanged(int)),
			settings, SLOT(setChanSlider(int)));
	
	connect(settings->fineSlider, SIGNAL(valueChanged(int)),
			chanlist, SLOT(setFine(int)));
	connect(chanlist, SIGNAL(fineChanged(int)),
			settings, SLOT(setFineSlider(int)));

	connect(settings->input, SIGNAL(highlighted(int)),
			chanlist, SLOT(setInput(int)));
	connect(chanlist, SIGNAL(inputChanged(int)),
			settings, SLOT(setInputCombo(int)));
	connect(settings->input, SIGNAL(highlighted(int)),
			SLOT(setFormatList(int)));
	connect(chanlist, SIGNAL(inputChanged(int)),SLOT(setFormatList(int)));
	connect(chanlist, SIGNAL(inputChanged(int)),SLOT(setInput(int)));

	
	connect(settings->format, SIGNAL(highlighted(int)),
			chanlist, SLOT(setFormat(int)));
	connect(chanlist, SIGNAL(formatChanged(int)),
			settings, SLOT(setFormatCombo(int)));
	connect(chanlist, SIGNAL(formatChanged(int)),SLOT(setFormat(int)));

	connect(settings->delBut, SIGNAL(clicked()),
			chanlist, SLOT(removeCurrentItem()));
	
	connect(settings->addBut, SIGNAL(clicked()),
			chanlist, SLOT(cloneCurrentItem()));

	connect(settings->reset, SIGNAL(clicked()),
			settings, SLOT(resetFineSlider()));

	connect(chanlist, SIGNAL(freqChanged(unsigned long)),
			SLOT(setFreq(unsigned long)));

	cfg->setGroup(QString("General"));
	regions = new QStrList();
	QString str;
	for (int r=cfg->readNumEntry("regions", 0); r; r--)
	{
		cfg->setGroup(str.sprintf("Region %u",r-1));
		regions->insert(0,cfg->readEntry("name", str));
	}
	deviceDlg->regC->clear();
	deviceDlg->regC->insertStrList(regions);
	readCfg();
	cfg->setGroup(QString("General"));
	KWM::setSticky(winId(), (bool)cfg->readNumEntry("sticky", false));
	indicator->actionFlag = PresetAction;

	if(optionsDlg->restoreCB->isChecked())
	{
		chanlist->setCurrentItem(cfg->readNumEntry("lastmem",0));
		resize(cfg->readNumEntry("width",240),cfg->readNumEntry("height",180));
	}
	else
	{
		chanlist->setCurrentItem(0);
		resize( 240, 180);
	}				

	connect(this, SIGNAL(visibilityChanged()), SLOT(aot()));	
	if (optionsDlg->capCB->isChecked()) toggleCap();
}


TvWidget::~TvWidget()
{
	cfg->setGroup(QString("General"));
	cfg->writeEntry("lastmem", chanlist->currentItem());
	cfg->writeEntry("width", width());
	cfg->writeEntry("height", height());
	cfg->sync();
	if(pic)
		delete pic;
	delete preferences;
	delete actionText;
	delete chanlist;
	delete indicator;
}

void TvWidget::paintEvent ( QPaintEvent *)
{

	if ((!getCapture() || v4linit) && pic){
		QPainter paint(this);
		QPoint offset((width()-320)/2, (height()-240)/2);
		paint.drawPixmap( offset, *pic );
		paint.flush();
	}
}

void TvWidget::mouseDoubleClickEvent ( QMouseEvent * e )
{
//	if (e->button() == RightButton)
//			popUpOldShow();
//	else
		mousePressEvent(e);
	
}
		
		
void TvWidget::mousePressEvent ( QMouseEvent * e )
{
	switch (e->button()){
		case RightButton:
			popup->popup(mapToGlobal(e->pos()));
			break;
		case LeftButton:
			if (indicator->actionFlag == PresetAction)
				if (!indicator->isVisible())
					chanlist->whatHighlighted();
				else
					chanlist->previous();
			else
				if (val[indicator->actionFlag]){
					val[indicator->actionFlag]--;
					setVal();
					indicator->setBar(val[indicator->actionFlag]);
					if (!mouseTimer){
						mouseTimer = startTimer(150);
						mouseState = e->button();
					}
				}
			break;

		case MidButton:
			if (indicator->actionFlag == PresetAction)
				if (!indicator->isVisible())
					chanlist->whatHighlighted();
				else
					chanlist->next();
			else
				if (val[indicator->actionFlag]<100){
					val[indicator->actionFlag]++;
					setVal();
					indicator->setBar(val[indicator->actionFlag]);
					if (!mouseTimer){
						mouseTimer = startTimer(150);
						mouseState = e->button();
					}
				}
			break;
		}
}



void TvWidget::mouseReleaseEvent ( QMouseEvent * )
{
	if (mouseTimer){
		killTimer(mouseTimer);
		mouseTimer = 0;}
}


void TvWidget::chanlistShow()
{
	int w = 125;
	int h = 245;

// need this for upcoming feature TV Mode	
//	int x1 = x()+5;
//	int y1 = y()+5;
//	if (x1+w>qv4lApp->desktop()->width()) x1=qv4lApp->desktop()->width()-w-10;
//	if (y1+h>qv4lApp->desktop()->height()) y1=qv4lApp->desktop()->height()-h-10;
	
	chanlist->resize(w,h);
//	chanlist->recreate( 0 ,WStyle_NoBorder, QPoint(x1,y1), false);
	chanlist->setBackgroundColor(colorGroup().base());
//	chanlist->repaint();
	chanlist->show();
	
}

void TvWidget::settingsShow()
{
	chanlist->setDND(true);
	chanlistShow();
	preferences->show();
}

void TvWidget::setFormatList(int i)
{
	settings->format->clear();
	settings->format->insertStrList(formatList(i));
}


void TvWidget::readCfg()
{
	QString str;
	QStrList * spaceList;
	int oldcur = chanlist->currentItem();
	chanlist->clear();
	int count;
	
	cfg->setGroup("General");
	deviceDlg->deviceC->setText(cfg->readEntry("viddev", ""));
	deviceDlg->regC->setCurrentItem(cfg->readNumEntry("region", 0));
	str.sprintf("Region %u", deviceDlg->regC->currentItem());
	chanlist->chanToFreqClear();
	cfg->setGroup(str);
	for (int s=cfg->readNumEntry("spaces", 0); s; s--)
	{
		str.sprintf("space%u", s-1);
		spaceList = new QStrList();
		cfg->readListEntry(str, *spaceList);
		chanlist->chanToFreqAdd(spaceList);
	}
	cfg->setGroup("General");
	
	val[BrightAction] = cfg->readNumEntry("brightness", 50);
	val[HueAction] = cfg->readNumEntry("hue", 50);
	val[ContrastAction] = cfg->readNumEntry("contrast", 50);
	val[ColorAction] = cfg->readNumEntry("color", 50);
	optionsDlg->aotCB->setChecked(cfg->readNumEntry("aot", false));
	optionsDlg->capCB->setChecked(cfg->readNumEntry("capture", false));
	optionsDlg->stickyCB->setChecked(cfg->readNumEntry("sticky", false));
	optionsDlg->fixCB->setChecked(cfg->readNumEntry("fix4/3", false));
	optionsDlg->restoreCB->setChecked(cfg->readNumEntry("restore", false));

	setBright(val[BrightAction]*655);
	pictureDlg->setValue(BrightAction, val[BrightAction]);
	setHue(val[HueAction]*655);
	pictureDlg->setValue(HueAction, val[HueAction]);
	setContrast(val[ContrastAction]*655);
	pictureDlg->setValue(ContrastAction, val[ContrastAction]);
	setColor(val[ColorAction]*655);
	pictureDlg->setValue(ColorAction, val[ColorAction]);
		
	count = cfg->readNumEntry("count",0);

	chanlist->setAutoUpdate(false);
	if (count) {
		for (int i=0; i<count; i++){
			QString str;
			str.sprintf("Memory %d", i);
			cfg->setGroup(str);
	
			chanlist->insertItem(new ChanListItem(
				(const char *)cfg->readEntry("name", str),
				cfg->readNumEntry("channel",1),
				cfg->readNumEntry("fine",0),
				cfg->readNumEntry("input",0),
				cfg->readNumEntry("format",0)),-1);
		}
	}	
	chanlist->setAutoUpdate(true);
	chanlist->repaint();
	indicator->actionFlag = PresetAction;
	chanlist->setCurrentItem(oldcur);
}

void TvWidget::writeCfg()
{
	ChanListItem * tmp;
	int count = chanlist->count();
	cfg->setGroup("General");
	cfg->writeEntry("count", count);
	cfg->writeEntry("viddev", deviceDlg->deviceC->text());
	cfg->writeEntry("region", deviceDlg->regC->currentItem());
	cfg->writeEntry("brightness", val[BrightAction]);
	cfg->writeEntry("hue", val[HueAction]);
	cfg->writeEntry("contrast", val[ContrastAction]);
	cfg->writeEntry("color", val[ColorAction]);
	cfg->writeEntry("aot", optionsDlg->aotCB->isChecked());
	cfg->writeEntry("capture", optionsDlg->capCB->isChecked());
	cfg->writeEntry("sticky", optionsDlg->stickyCB->isChecked());
	cfg->writeEntry("fix4/3", optionsDlg->fixCB->isChecked());
	cfg->writeEntry("restore", optionsDlg->restoreCB->isChecked());
	
	
	if (count)
		for (int i=0; i<count; i++){
			QString str;
			str.sprintf("Memory %d", i);
			cfg->setGroup(str);
	
			tmp = chanlist->getItem(i);
			cfg->writeEntry("name", tmp->name());
			cfg->writeEntry("channel",tmp->chan());
			cfg->writeEntry("fine",tmp->fine());
			cfg->writeEntry("input",tmp->input());
			cfg->writeEntry("format",tmp->format());
			delete tmp;
		}
	cfg->sync();
	readCfg();	
}


void TvWidget::changeAction(int f)
{
	indicator->actionFlag = (Flag)f;
	indicator->setText(actionText->at(f));
	indicator->setBar(val[indicator->actionFlag]);
}	


void TvWidget::resizeEvent(QResizeEvent *e)
{
	static bool inLoopFlag=false;
	int w = e->size().width();
	int h = e->size().height();

	if (inLoopFlag)
	{
		inLoopFlag=false;
		return;
	}

	if (optionsDlg->fixCB->isChecked() && (w/4*3) != h)
	{
		if (isVisible())
			inLoopFlag = true;
		if (w >= h) 
			resize(h/3*4, h);
		else
			resize(w, w/4*3);
		
	}
	indicator->specialResize(e);
}

void TvWidget::timerEvent(QTimerEvent *e)
{
	if ( e->timerId()==mouseTimer)
	{
		mousePressEvent(& QMouseEvent(Event_MouseButtonPress,
				QPoint(0,0), mouseState,0));

	}

	if (e->timerId()==aotTimer)
	{
		killTimer(aotTimer);
		aotTimer = 0;
	}
}

void TvWidget::setValByValue(int f,int i)
{
	switch (f){
		case BrightAction:
			setBright((val[f]=i)*655);
			break;
		case HueAction:
			setHue((val[f]=i)*655);
			break;
		case ContrastAction:
			setContrast((val[f]=i)*655);
			break;
		case ColorAction:
			setColor((val[f]=i)*655);
			break;
	}
	indicator->actionFlag = (Flag)f;
	indicator->setBar(val[indicator->actionFlag]);
	indicator->setText(actionText->at(f));
}
	

void TvWidget::setVal()
{
	int f = (int)indicator->actionFlag;
	switch (f){
		case BrightAction:
			setBright(val[f]*655);
			pictureDlg->setValue(BrightAction, val[f]);
			break;
		case HueAction:
			setHue(val[f]*655);
			pictureDlg->setValue(HueAction, val[f]);
			break;
		case ContrastAction:
			setContrast(val[f]*655);
			pictureDlg->setValue(ContrastAction, val[f]);
			break;
		case ColorAction:
			setColor(val[f]*655);
			pictureDlg->setValue(ColorAction, val[f]);
			break;
	}					
}

void TvWidget::aot()
{
	if (!aotTimer)
		if (optionsDlg->aotCB->isChecked() 
			&& !popup->isVisible() && !preferences->isVisible())
		{
			KWM::raise(winId());
			aotTimer = startTimer(500);
		}
	else
	{
		killTimer(aotTimer);
		aotTimer = startTimer(500);
	}	
}

void TvWidget::toggleCap()
{
	setCapture(!getCapture());
}

#include "tvwidget.moc"
