#include <qpainter.h>
#include "table.h"


CardTable::
CardTable(int max_hands, QWidget *parent, const char *name) : 
          QWidget(parent, name) {
   QColor color("forestgreen");
   _player_hands.setAutoDelete(TRUE);
   if (color.isValid())
      setBackgroundColor(color);
   else
      setBackgroundColor(green);

   _pixmap_p = new QPixmap(10 + max_hands * 200, 180);
   _pixmap_p->fill(backgroundColor());
   _scroll_p = new QScrollBar(0, 0, 20, 200, 0, QScrollBar::Horizontal, 
                              this, "table scroll");
   connect(_scroll_p, SIGNAL(valueChanged(int)), 
           this, SLOT(displayPlayerHands(int)));

   setMinimumSize (220, 350);
   resizeEvent(NULL);
}


void CardTable::
adjustScrollBar(void) {
   int scroll_size;

   _scroll_p->setGeometry(0, height() - 20, width(), 20);
   scroll_size = (200 * numHands()) - width();
   if (scroll_size < 0)
      scroll_size = 0;
      
   _scroll_p->setRange(0, scroll_size);
}


void CardTable::
addPlayerHand(Hand *hp) { 
   _player_hands.append(hp);
   adjustScrollBar();
}


void CardTable::
addCard(Hand *hp, Card *cp) {
   hp->addCard(cp);
   drawHand(hp);
   if (hp != &_dealer_hand) {
      displayPlayerHands(_scroll_p->value());
      valueText(hp);
   }
}


Card *CardTable::
removeCard(Hand *hp) {
   Card *cp = hp->removeCard();
   drawHand(hp);
   if (hp != &_dealer_hand)
      displayPlayerHands(_scroll_p->value());
}


void CardTable::
drawHands(void) {
   drawHand(&_dealer_hand);
   drawPlayerHands();
}


void CardTable::
drawPlayerHands(void) {
   for (int i = 0; i < _player_hands.count(); i++)
      drawHand(_player_hands.at(i));  
   displayPlayerHands(_scroll_p->value());
}


void CardTable::
updatePlayerHands(void) {
   if (_pixmap_dirty)
      displayPlayerHands(_scroll_p->value());
}


void CardTable::
displayPlayerHands(int x_offset) {
   QPainter painter;
   painter.begin(this);
   painter.drawPixmap(0, 150, *_pixmap_p, x_offset, 0, width(), -1);
   painter.end();
   _pixmap_dirty = 0;
}


void CardTable::
drawHand(Hand *hp) {
   int i, x, y;
   if (hp == &_dealer_hand) {
      x = y = 10;
      hp->draw(this, x, y);
   } else {
      x = 10 + 200 * _player_hands.find(hp); 
      y = 0;
      hp->draw(_pixmap_p, x, y);
      _pixmap_dirty = 1;
   }
}


void CardTable::
clear(void) {
   _player_hands.clear();			// delete all hands
   _dealer_hand.clear();
   adjustScrollBar();
   _pixmap_p->fill(backgroundColor());		// clear the pixmap 
   _pixmap_dirty = 0;
   erase();					// clear the CardTable
}


void CardTable::
betText(Hand *hp) {
   QString str;
   QRect rect(10 + 200 * _player_hands.find(hp), BET_Y, TEXT_W, TEXT_H);
   QPainter painter;
 
   str.sprintf("$%4.2f", hp->getBet());
   painter.begin(_pixmap_p);
   painter.setBackgroundColor(backgroundColor());
   painter.eraseRect(rect);
   painter.drawText(rect, AlignHCenter, str);
   painter.end();
   _pixmap_dirty = 1;
}


void CardTable::
valueText(Hand *hp) {
   QString value;
   QRect rect(10 + 200 * _player_hands.find(hp), VALUE_Y, TEXT_W, TEXT_H);
   QPainter painter;

   if (hp->value() > 21)
      value = "Busted";
   else if (numHands() == 1 && hp->value() == 21 && hp->numCards() == 2)
      value = "Blackjack!"; 
   else
      value.sprintf("%d", hp->value());

   painter.begin(_pixmap_p);
   painter.setBackgroundColor(backgroundColor());
   painter.eraseRect(rect);
   painter.drawText(rect, AlignHCenter, value);
   painter.end();
   _pixmap_dirty = 1;
}


void CardTable::
statusText(Hand *hp, QString text) {
   QRect rect(10 + 200 * _player_hands.find(hp), STATUS_Y, TEXT_W, TEXT_H);
   QPainter painter;
 
   painter.begin(_pixmap_p);
   painter.setBackgroundColor(backgroundColor());
   painter.eraseRect(rect);
   painter.drawText(rect, AlignHCenter, text);
   painter.end();
   _pixmap_dirty = 1;
}

